#pragma once

#include <QJsonObject>

#include "Angle.h"
#include "Energy.h"
#include "Position.h"
#include "Velocity.h"

class AbstractGameObject
{
	public:
		AbstractGameObject()
			: m_id(0)
			, m_alive(true)
		{
		}

		AbstractGameObject(int id, const Angle &rotation, const Position &position)
			: m_id(id)
			, m_rotation(rotation)
			, m_position(position)
			, m_alive(true)
		{
		}

		AbstractGameObject(const QJsonObject& jsonObject)
			: m_alive(true)
		{
			m_id = jsonObject.find("id").value().toInt();
			m_energy.set(jsonObject.find("energy").value().toInt());

			m_rotation = Angle::fromDegrees(jsonObject.find("rotation").value().toInt());

			const auto& vx = jsonObject.find("velocityX").value().toDouble();
			const auto& vy = jsonObject.find("velocityY").value().toDouble();
			m_velocity = Velocity(vx, vy);

			const auto& px = jsonObject.find("x").value().toDouble();
			const auto& py = jsonObject.find("y").value().toDouble();
			m_position = Position(px, py);
		}

		virtual void step() = 0;

		int id() const
		{
			return m_id;
		}

		Energy& energy()
		{
			return m_energy;
		}

		const Energy& energy() const
		{
			return m_energy;
		}

		Angle& rotation()
		{
			return m_rotation;
		}

		const Angle& rotation() const
		{
			return m_rotation;
		}

		Velocity& velocity()
		{
			return m_velocity;
		}

		const Velocity& velocity() const
		{
			return m_velocity;
		}

		Position& position()
		{
			return m_position;
		}

		const Position& position() const
		{
			return m_position;
		}

		Angle headingTo(const Position& pos) const
		{
			// I DON'T KNOW WHAT I'M DOING, OKAY?
			auto angle = position().angleTo(pos).asDegrees() - rotation().asDegrees();
			while (angle > 180)
			{
				angle -= 360;
			}
			while (angle < -180)
			{
				angle += 360;
			}
			return Angle::fromDegrees(angle);
		}

		bool isAlive() const
		{
			return m_alive;
		}

		void setAlive(bool alive)
		{
			m_alive = alive;
		}

		bool operator==(const AbstractGameObject& other) const
		{
			// Compare stuff that depends on the rotation with fairly low
			// precision since rotation is sent as int from the server...
			return m_id == other.m_id
				&& m_energy == other.m_energy
				&& qAbs(m_rotation.asDegrees() - other.m_rotation.asDegrees()) < 1
				&& qAbs(m_velocity.x() - other.m_velocity.x()) < 0.001
				&& qAbs(m_velocity.y() - other.m_velocity.y()) < 0.001
				&& qAbs(m_position.x() - other.m_position.x()) < 0.001
				&& qAbs(m_position.y() - other.m_position.y()) < 0.001;
		}

	private:
		int m_id;
		Energy m_energy;
		Angle m_rotation;
		Velocity m_velocity;
		Position m_position;

		bool m_alive;
};
