// Define our playing state
Game.State.PlayState = {
	_hpLastRender: null,
	_legendItems: {},
    _player: null,
    _gameEnded: false,
    _subState: null,
	_totalMessages: 0,
	_minimapContext: null,
	_minimalUI: false,
	_minimapCellSize: 5,
	
	getPlayer: function() {
		return this._player;
	},
	enter: function() {
		console.log("Entered play state.");
		$('#threeDisplay').hide();
		$('#threeDisplay').fadeIn(3000);
		$('#minimap').fadeIn(3000);
        // Create a map based on our size parameters
        /*
		For caverns:
		var width = 100;
        var height = 48;
        */
		
		var width = 50;
        var height = 50;
        var depth = 6;
        // Create our map from the tiles and player
        this._player = new Game.Entity(Game.PlayerTemplate);
        var tiles = new Game.Builder(width, height, depth).getTiles();
        var map = new Game.Map.Cave(tiles, this._player);
        // Start the map's engine
        map.getEngine().start();
		
		// Set up the minimap
		this._minimapContext = document.getElementById("minimap").getContext("2d");
		this._minimapContext.canvas.width = width*this._minimapCellSize;
		this._minimapContext.canvas.height = height*this._minimapCellSize;
		this.rebuildMinimap();
    },
    exit: function() { console.log("Exited play state."); },
    render: function() {
		var player = this._player, map = player.getMap(), display=RotEngine.getDisplay(), displayFOV=RotEngine.getDisplayFOV();
		
        // Render substate if there is one
        if (this._subState) {
            this._subState.render(display);
            return;
        }

		display.clear();
        displayFOV.clear();

		// Start a damage flash if req'd
		if (this._hpLastRender!==null && this._hpLastRender > player.getHp())
			ThreeEngine.startDamageFlash();
		this._hpLastRender = player.getHp();
		
        // Render the tiles
        this.renderTiles();

        // Get the messages in the player's queue and render them
        var messages = player.getMessages(),str="";
		for (var i = 0; i < messages.length; i++) {
			str+=messages[i]+"<br/>";
			this._totalMessages++;
		}
		player.clearMessages();	// Clear the mixin message queue
		
		var messageDisplay = document.getElementById("oldMessages");
		var newMessageDisplay = document.getElementById("newMessages");
		var messagesContainer = document.getElementById("messagesContainer");
		messageDisplay.innerHTML+=newMessageDisplay.innerHTML;
		newMessageDisplay.innerHTML=str;
		messagesContainer.style.top = Math.min(18*7,this._totalMessages*18)+"px";
		
        // Render stats, messages, hunger, location description
		document.getElementById("playerHP").innerHTML = this.getHealthBarHTML(player.getHp(), player.getMaxHp(), 25);
		
		var stats = vsprintf('%s <span class="statlabel">XP</span> %d<span class="statlabel">/</span>%d <span class="statlabel">Att</span> %d <span class="statlabel">Def</span> %d <span class="statlabel">Depth</span> %d ft', [Game.RankNames[player.getLevel()-1], player.getExperience(), player.getNextLevelExperience(), player.getAttackValue(), player.getDefenseValue(), Game.levelToFeet(player.getZ())]);
        document.getElementById("playerStats").innerHTML=stats;
		if(!this._minimalUI || player.getHungerPercent()>95 || player.getHungerPercent()<25)
			document.getElementById("playerHunger").innerHTML=player.getHungerState();
		else
			document.getElementById("playerHunger").innerHTML="";
		
		document.getElementById("locationDescription").innerHTML = map.getTile(player.getX(), player.getY(), player.getZ()).getDescription();
		
		// Build legend
		var legend="";
		for(key in this._legendItems)
			legend += this._legendItems[key];
		
		// Build neighbour HP list
		var entstr="",neighbours=Game.getNeighborPositions(player.getX(),player.getY());
		for(var i=0;i<neighbours.length;i++) {
			var ent = map.getEntityAt(neighbours[i].x, neighbours[i].y, player.getZ());
			if (ent!==undefined){
				entstr += ent.getDescription() + "<br/>";
				entstr += this.getHealthBarHTML(ent.getHp(),ent.getMaxHp()) + "<br/>";
			}
		}
		
		// Output 'em
		document.getElementById("legend").innerHTML = "";
		if (entstr!="")
			document.getElementById("legend").innerHTML = entstr + "<br/>";
		document.getElementById("legend").innerHTML += legend;
		
		// Set shader effects
		ThreeEngine.setEffects(player.getHungerPercent(), player.getHp() / player.getMaxHp() * 100);
    },
	
	getHealthBarHTML: function(currentHP, maxHP, numTicks){
		if (!numTicks)
			numTicks = 10;
		var hpBar="<span class='healthOK'>",flipped=false,tick=maxHP/numTicks;
		for(var i=0;i<numTicks;i++){
			if(currentHP > i*tick)
				hpBar+="&#9829;";
			else {
				if(!flipped){
					flipped=true;
					hpBar+="</span><span class='healthBad'>"
				}
				hpBar+="&#9829;";
			}
		}
		hpBar+='</span>';
		if(!this._minimalUI)
			hpBar+=' '+Math.ceil(currentHP)+'<span class="statlabel">/</span>'+maxHP;
		return hpBar;
	},
	
    getScreenOffsets: function() {
		var topLeftX = this._player.getX() - (RotEngine.getConsoleWidth() / 2);
		var topLeftY = this._player.getY() - (RotEngine.getConsoleHeight() / 2);
		
        return {
            x: topLeftX,
            y: topLeftY
        };
    },
	
    renderTiles: function() {
		var display=RotEngine.getDisplay(), displayFOV=RotEngine.getDisplayFOV();
        var screenWidth = RotEngine.getConsoleWidth();
        var screenHeight = RotEngine.getConsoleHeight();
        var offsets = this.getScreenOffsets();
        var topLeftX = offsets.x;
        var topLeftY = offsets.y;
        
		var visibleCells = {};
        var player = this._player, map = player.getMap();
        var currentDepth = player.getZ();
        var ctx = this._minimapContext;
		
        map.getFov(currentDepth).compute(
            player.getX(), player.getY(), 
            player.getSightRadius(), 
            function(x, y, radius, visibility) {
                visibleCells[x + "," + y] = visibility;
                map.setExplored(x, y, currentDepth, true);
            });
        
		this._legendItems = {};
        for (var x = topLeftX; x < topLeftX + screenWidth; x++) {
            for (var y = topLeftY; y < topLeftY + screenHeight; y++) {
                if (map.isExplored(x, y, currentDepth)) {			// For each explored cell in current screen area
                    var glyph = map.getTile(x, y, currentDepth);
					var playerDist = Game.distanceBetween(x, y, player.getX(), player.getY());
					var background = glyph.getBackground();
					var bgVariance = glyph.getColourVariance();
					
					// Fill in the minimap
					if (ctx!==null){
						var c = ROT.Color.fromString(glyph.getMinimapCol());
						ctx.fillStyle = "rgba("+c[0]+","+c[1]+","+c[2]+",1)";
						ctx.fillRect(x*this._minimapCellSize,y*this._minimapCellSize,this._minimapCellSize,this._minimapCellSize);
					}

					// Render the cell into multimap and diffuse map 
					var light = 0,vis = 0,foreground;
					var visibility = visibleCells[x + ',' + y];
					if (visibility) {								// It's visible, so render nicely with ents + items
						vis = 255;
						var items = map.getItemsAt(x, y, currentDepth);
                        if (items)									// If we have items, we want to render the top most item
                            glyph = items[items.length - 1];
                        if (map.getEntityAt(x, y, currentDepth))	// Render entities over items
                            glyph = map.getEntityAt(x, y, currentDepth);
                        
						foreground = glyph.getForeground();			// Grab the foreground colour of either the tile or the ent/item
						var seed = ROT.RNG.getSeed();				// Tweak the colours (backing up the seed)
						ROT.RNG.setSeed(x*y);
						foreground = ROT.Color.toHex(ROT.Color.randomize(ROT.Color.fromString(foreground), glyph.getColourVariance()));	// Vary the colours
						if(glyph.getVertical()===0)					// Don't mess the chromakeying up, leave backgrounds alone if there's a vert component!
							background = ROT.Color.toHex(ROT.Color.randomize(ROT.Color.fromString(background), bgVariance));
						ROT.RNG.setSeed(seed);
						
						light = 1 - playerDist/(player.getSightRadius()+1);
						//light *= visibility;						// MARK: This needs preference testing :)
						light = Math.floor(light * 255);			// Get base light value for multimap
						light = Math.max(light,30);
                    } else {										// Not visible, apply standard fg and bg
						foreground = "#252530";
						background = "black";
					}

					// Render diffuse and multimap
					var vert = glyph.getVertical()*10;
					displayFOV.draw(x-topLeftX,y-topLeftY,' ','black',ROT.Color.toHex([light,vis,vert]));
					display.draw(x - topLeftX, y - topLeftY, glyph.getChar(), foreground, background);
					
					// Add the glyph to the legend array
					var legendGraphic="<div class='legendCell' style='background-color:"+glyph.getBackground()+";color:"+glyph.getForeground()+";'>"+glyph.getChar()+"</div><div class='legendLabel'>";
					var legendName = glyph.getDescription();		// Tiles just have description
					if (typeof glyph.describeA == 'function')		// But we'll have the a/an for entities/items
						legendName = glyph.describeA(true);
					legendGraphic += legendName+"</div><br/>";
					this._legendItems[legendName] = legendGraphic;
                }
            }
        }
		
		if (ctx!==null){			// Add player to minimap
			ctx.fillStyle = "white";
			ctx.fillRect(player.getX()*this._minimapCellSize,player.getY()*this._minimapCellSize,this._minimapCellSize,this._minimapCellSize);
		}
    },
	rebuildMinimap: function() {
		var currentDepth=this._player.getZ(), map=this._player.getMap(), ctx=this._minimapContext;
		ctx.clearRect(0,0,map.getWidth()*this._minimapCellSize,map.getHeight()*this._minimapCellSize);
		
		for (var x = 0; x < map.getWidth(); x++)
            for (var y = 0; y < map.getHeight(); y++)
				if (map.isExplored(x, y, currentDepth)) {
					var c = ROT.Color.fromString(map.getTile(x, y, currentDepth).getMinimapCol());
					ctx.fillStyle = "rgba("+c[0]+","+c[1]+","+c[2]+",1)";
					ctx.fillRect(x*this._minimapCellSize,y*this._minimapCellSize,this._minimapCellSize,this._minimapCellSize);
				}
	},
    handleInput: function(inputType, inputData) {
        // If the game is over, controls are locked out.
        if (this._gameEnded)
            return;
			
        // Handle substate input if there is one
        if (this._subState) {
            this._subState.handleInput(inputType, inputData);
            return;
        }
        if (inputType === 'keydown') {
            // Movement
            if (inputData.keyCode === ROT.VK_LEFT || inputData.keyCode === ROT.VK_NUMPAD4 || inputData.keyCode === ROT.VK_A)
                this.move(-1, 0, 0);
            else if (inputData.keyCode === ROT.VK_RIGHT || inputData.keyCode === ROT.VK_NUMPAD6 || inputData.keyCode === ROT.VK_D)
                this.move(1, 0, 0);
            else if (inputData.keyCode === ROT.VK_UP || inputData.keyCode === ROT.VK_NUMPAD8 || inputData.keyCode === ROT.VK_W)
                this.move(0, -1, 0);
            else if (inputData.keyCode === ROT.VK_DOWN || inputData.keyCode === ROT.VK_NUMPAD2 || inputData.keyCode === ROT.VK_S)
                this.move(0, 1, 0);
            else if (inputData.keyCode === ROT.VK_NUMPAD7)
                this.move(-1, -1, 0);
            else if (inputData.keyCode === ROT.VK_NUMPAD9)
                this.move(1, -1, 0);
            else if (inputData.keyCode === ROT.VK_NUMPAD1)
                this.move(-1, 1, 0);
            else if (inputData.keyCode === ROT.VK_NUMPAD3)
                this.move(1, 1, 0);
			else if (inputData.keyCode === ROT.VK_Z || inputData.keyCode === ROT.VK_NUMPAD5) {			// Rest
				// Do nothing except not return, so that the engine will be restarted
			} else if (inputData.keyCode === ROT.VK_I) {		// Show the inventory screen
                this.showItemsSubState(Game.State.inventoryState, this._player.getItems(), 'You are not carrying anything.');
                return;
            } else if (inputData.keyCode === ROT.VK_P) {		// Show the drop screen
                this.showItemsSubState(Game.State.dropState, this._player.getItems(), 'You have nothing to drop.');
                return;
            } else if (inputData.keyCode === ROT.VK_E) {		// Show the eat screen
                this.showItemsSubState(Game.State.eatState, this._player.getItems(), 'You have nothing to eat.');
                return;
            } else if (inputData.keyCode === ROT.VK_Y) {		// Show the wield screen
				this.showItemsSubState(Game.State.wieldState, this._player.getItems(), 'You have nothing to wield.');
                return;
            } else if (inputData.keyCode === ROT.VK_U) {		// Show the wear screen
                this.showItemsSubState(Game.State.wearState, this._player.getItems(), 'You have nothing to wear.');
                return;
            } else if (inputData.keyCode === ROT.VK_X) {		// Show the examine screen
                this.showItemsSubState(Game.State.examineState, this._player.getItems(), 'You are not carrying anything.');
                return;
            } else if (inputData.keyCode === ROT.VK_L) {		// Toggle stats display
                ThreeEngine.toggleStatsDisplay();
                return;
            } else if (inputData.keyCode === ROT.VK_M) {		// Toggle min UI
				this._minimalUI=!this._minimalUI;
				if(this._minimalUI) {
					$("#legend").fadeOut(500);
					$("#playerStats").fadeOut(500);
					$("#messagesContainer").fadeOut(500);
					$("#locationDescription").fadeOut(500,function(){
						$("#playerHP").css("bottom","4px");
						//$("#minimap").css("bottom","4px");
						Game.refresh();			// So tweaked UI gets drawn
					});					
				} else {
					Game.refresh();
					$("#playerHP").css("bottom","28px");
					//$("#minimap").css("bottom","28px");
					$("#legend").fadeIn(500);
					$("#playerStats").fadeIn(500);
					$("#messagesContainer").fadeIn(500);
					$("#locationDescription").fadeIn(500);
				}
                return;
            } else if (inputData.keyCode === ROT.VK_R) {		// Reset the cam
				$('#threeDisplay').fadeOut(100, function(){
					ThreeEngine.resetCamera();
					$('#threeDisplay').fadeIn(100);
                });
                return;					
            } else if (inputData.keyCode === ROT.VK_SPACE) {	// Multi function
				var items = this._player.getMap().getItemsAt(this._player.getX(), this._player.getY(), this._player.getZ());
				if (items) {									// Check for items first
					if (items.length === 1) {					// If there is only one item, pick it up
						var item = items[0];
						if (this._player.pickupItems([0]))
							Game.sendMessage(this._player, "You "+Game.randomMessage(['pick up','grab','sieze','grasp'])+" %s.", [item.describeA()]);
						else
							Game.sendMessage(this._player, "You're too encumbered to pick it up!");
					} else										// Multiple items, show selection state
						this.showItemsSubState(Game.State.pickupState, items, 'There is nothing here to pick up.');
				} else {										// No items, check for stairs!
					var tile = this._player.getMap().getTile(this._player.getX(), this._player.getY(), this._player.getZ());
					if (tile===Game.Tile.stairsUpTile) {
						this.move(0, 0, -1);
						return;
					} else if (tile===Game.Tile.stairsDownTile || tile===Game.Tile.holeToCavernTile) {
						this.move(0, 0, 1);
						return;
					} else										//  Nope, not stairs either...
						return;
				}
            } else												// Not a valid key
                return;
				
            this._player.getMap().getEngine().unlock();			// If we haven't returned, a valid turn has happened, so unlock the engine to let shit proceed!
        } else if (inputType === 'keypress') {					// To handle special keys we can only extract from char code...
            var keyChar = String.fromCharCode(inputData.charCode);
            if (keyChar === '?') {	// Setup the help state.
                this.setSubState(Game.State.HelpState);
                return;		
            } else	// Not a valid key
                return;
				
            // Unlock the engine
            this._player.getMap().getEngine().unlock();
        } 
    },
    move: function(dX, dY, dZ) {
        var newX = this._player.getX() + dX;
        var newY = this._player.getY() + dY;
        var newZ = this._player.getZ() + dZ;
        // Try to move to the new cell
        this._player.tryMove(newX, newY, newZ, this._player.getMap());
    },
    setGameEnded: function(gameEnded) {
        this._gameEnded = gameEnded;
    },
    setSubState: function(subState) {
        this._subState = subState;
        Game.refresh();		// Force a render, we changed the subscene
    },
    showItemsSubState: function(subState, items, emptyMessage) {
        if (items && subState.setup(this._player, items) > 0)
            this.setSubState(subState);
        else {
            Game.sendMessage(this._player, emptyMessage);
            Game.refresh();
        }
    }
};
