Game.State.ItemListState = function(template) {
    this._title = template['title'];
    this._caption = template['caption'];
	this._okFunction = template['ok'];
    
    this._isAcceptableFunction = template['isAcceptable'] || function(x) {	// By default, we use the identity function
        return x;
    }
    
    this._canSelectItem = template['canSelect'];							// Whether the user can select items at all.
    this._canSelectMultipleItems = template['canSelectMultipleItems'];		// Whether the user can select multiple items.
    this._hasNoItemOption = template['hasNoItemOption'];					// Whether a 'no item' option should appear.
};

// Should be called before switching to the state.
Game.State.ItemListState.prototype.setup = function(player, items) {
    this._player = player;
    var count = 0;
    // Iterate over each item, keeping only the aceptable ones and counting the number of acceptable items.
    var that = this;
    this._items = items.map(function(item) {
        // Transform the item into null if it's not acceptable
        if (that._isAcceptableFunction(item)) {
            count++;
            return item;
        } else
            return null;
    });
    // Clean set of selected indices
    this._selectedIndices = {};
    return count;
};

Game.State.ItemListState.prototype.render = function() {
	var letters = 'abcdefghijklmnopqrstuvwxyz';
    var msg="<h1>"+this._title+"</h1>";
	
	if(this._caption!==undefined)
		msg +=this._caption+"<br/><br/>";
	
    if (this._hasNoItemOption)
        msg+="0 - no item<br/>";
    
	for (var i = 0; i < this._items.length; i++) {
        if (this._items[i]) {	// Do we have one?
            var letter = letters.substring(i, i + 1);
            var selectionState = (this._canSelectItem && this._canSelectMultipleItems && this._selectedIndices[i]) ? '+' : '-';
			var suffix = '';
            if (this._items[i] === this._player.getArmor())
                suffix = ' (wearing)';
            else if (this._items[i] === this._player.getWeapon())
                suffix = ' (wielding)';
            msg+=letter + ' ' + selectionState + ' ' + this._items[i].describeA(true) + suffix+"<br/>";
        }
    }
	
	if(this._canSelectMultipleItems)
		msg+="<br/><strong>Press Space to confirm selections<br>or Escape to cancel</strong>"
	else
		msg+="<br/><strong>Press Escape to cancel</strong>"
	Game.modalAnnounce(msg);
};

Game.State.ItemListState.prototype.executeOkFunction = function() {
    var selectedItems = {};							// Gather the selected items.
    for (var key in this._selectedIndices)
        selectedItems[key] = this._items[key];
    
	Game.closeModalAnnounce();							// Switch back to the play state.
    Game.State.PlayState.setSubState(undefined);
    
    if (this._okFunction(selectedItems))			// Call the OK function and end the player's turn if req'd
        this._player.getMap().getEngine().unlock();
};

Game.State.ItemListState.prototype.handleInput = function(inputType, inputData) {
    if (inputType === 'keydown') {
		// If the user hit escape, hit enter and can't select an item, or hit enter without any items selected, simply cancel out
        if (inputData.keyCode === ROT.VK_ESCAPE || (inputData.keyCode === ROT.VK_SPACE && (!this._canSelectItem || Object.keys(this._selectedIndices).length === 0))) {
            Game.closeModalAnnounce();
			Game.State.PlayState.setSubState(undefined);
        } else if (inputData.keyCode === ROT.VK_SPACE)		// Handle pressing return when items are selected
            this.executeOkFunction();
        else if (this._canSelectItem && this._hasNoItemOption && inputData.keyCode === ROT.VK_0) {	// Handle pressing zero when 'no item' selection is enabled
            this._selectedIndices = {};
            this.executeOkFunction();
        } else if (this._canSelectItem && inputData.keyCode >= ROT.VK_A && inputData.keyCode <= ROT.VK_Z) {	// Handle pressing a letter if we can select
            var index = inputData.keyCode - ROT.VK_A;		// Map keycode back to index
            if (this._items[index]) {
                if (this._canSelectMultipleItems) {			// If multiple selection is allowed, toggle the selection status, else select the item and exit the state
                    if (this._selectedIndices[index])
                        delete this._selectedIndices[index];
                    else
                        this._selectedIndices[index] = true;
                    Game.refresh();							// Redraw state
                } else {
                    this._selectedIndices[index] = true;
                    this.executeOkFunction();
                }
            }
        }
    }
};

Game.State.inventoryState = new Game.State.ItemListState({
    title: 'INVENTORY',
    canSelect: false
});

Game.State.pickupState = new Game.State.ItemListState({
	title: 'PICK UP',
    caption: 'What would you like to pick up?',
    canSelect: true,
    canSelectMultipleItems: true,
    
	ok: function(selectedItems) {
		// Try to pick up all items, messaging the player if they couldn't all be picked up.
        if (!this._player.pickupItems(Object.keys(selectedItems)))
            Game.sendMessage(this._player, "You're too encumbered to pick up everything selected.");
        return true;
    }
});

Game.State.dropState = new Game.State.ItemListState({
	title: 'DROP',
    caption: 'What would you like to drop?',
    canSelect: true,
    canSelectMultipleItems: false,
    ok: function(selectedItems) {
        // Drop the selected item
        var key = Object.keys(selectedItems)[0];
        var item = selectedItems[key];
        Game.sendMessage(this._player, "You "+Game.randomMessage(['drop','cast aside','put down','get rid of'])+" %s.", [item.describeThe(false)]);		
        this._player.dropItem(key);
        return true;
    }
});

Game.State.eatState = new Game.State.ItemListState({
	title: 'EAT',
    caption: 'What would you like to eat?',
    canSelect: true,
    canSelectMultipleItems: false,
    isAcceptable: function(item) {
        return item && item.hasMixin('Edible');
    },
    ok: function(selectedItems) {
        // Eat the item, removing it if there are no consumptions remaining.
        var key = Object.keys(selectedItems)[0];
        var item = selectedItems[key];
        Game.sendMessage(this._player, "You "+Game.randomMessage(['consume','eat','munch','chomp','digest'])+" %s.", [item.describeThe(false)]);
        item.eat(this._player);
        if (!item.hasRemainingConsumptions())
            this._player.removeItem(key);
			
        return true;
    }
});

Game.State.wieldState = new Game.State.ItemListState({
	title: 'WIELD',
    caption: 'What would you like to wield?',
    canSelect: true,
    canSelectMultipleItems: false,
    hasNoItemOption: true,
    isAcceptable: function(item) {
        return item && item.hasMixin('Equippable') && item.isWieldable();
    },
    ok: function(selectedItems) {
        // Check if we selected 'no item'
        var keys = Object.keys(selectedItems);
        if (keys.length === 0) {
            this._player.unwield();
            Game.sendMessage(this._player, "You are empty handed.")
        } else {
            // Make sure to unequip the item first in case it is the armor.
            var item = selectedItems[keys[0]];
            this._player.unequip(item);
            this._player.wield(item);
            Game.sendMessage(this._player, "You are wielding %s.", [item.describeA()]);
        }
        return true;
    }
});

Game.State.wearState = new Game.State.ItemListState({
	title: 'WEAR',
    caption: 'What would you like to wear?',
    canSelect: true,
    canSelectMultipleItems: false,
    hasNoItemOption: true,
    isAcceptable: function(item) {
        return item && item.hasMixin('Equippable') && item.isWearable();
    },
    ok: function(selectedItems) {
        // Check if we selected 'no item'
        var keys = Object.keys(selectedItems);
        if (keys.length === 0) {
            this._player.unwield();
            Game.sendMessage(this._player, "You are not wearing anthing.")
        } else {
            // Make sure to unequip the item first in case it is the weapon.
            var item = selectedItems[keys[0]];
            this._player.unequip(item);
            this._player.wear(item);
            Game.sendMessage(this._player, "You are wearing %s.", [item.describeA()]);
        }
        return true;
    }
});

Game.State.examineState = new Game.State.ItemListState({
	title: 'EXAMINE',
    caption: 'What would you like to examine?',
    canSelect: true,
    canSelectMultipleItems: false,
    ok: function(selectedItems) {
        var keys = Object.keys(selectedItems);
        if (keys.length > 0) {
            var item = selectedItems[keys[0]];
            Game.sendMessage(this._player, "It's %s (%s).", [item.describeA(),item.details()]);
        }
        return true;
    }
});
