import pygame, sys, math, random, vector, utils, statemanager, json
from pygame.locals import *
from tween import *
from ship import Ship
from asteroid import Asteroid
from star import Star
from planet import Planet
from fuel import Fuel
from entity import Entity
from config import Config

class GameInfo:

    def __init__(self):
        self.CurrentStageId = 0
        self.CurrentDistance = 0
        self.AsteroidsTouched = 0
        self.MaxSpeed = 0
        self.Time = 0
        self.IsNewRecord = False

class Game:

    def Init(self, _payload):
        self.Planet = Planet(Config.PLANETSHAPE1, Config.PLANETSHAPE2, Config.WINDOWHEIGHT, -1)

        self.Ship = Ship(Config.SHIPSTART)

        self.Fuel = Fuel()

        self.Asteroids = []
        for i in range(0,Config.ASTEROIDCOUNT):
            asteroid = Asteroid()
            self.Asteroids.append(asteroid)

        self.Stars = []
        for i in range(0,Config.STARCOUNT):
            star = Star()
            star.Spawn(
                    [random.randrange(0,Config.WINDOWWIDTH),Entity.Offset+random.randrange(0,Config.WINDOWHEIGHT)], #position
                    random.randrange(Config.STARMINRADIUS,Config.STARMAXRADIUS)) # scale
            self.Stars.append(star)

        self.TimeIsSlow = False

        # initialize font; must be called after 'pygame.init()' to avoid 'Font not Initialized' error
        self.Font = pygame.font.SysFont("monospace", 16)
        self.FontBig = pygame.font.SysFont("monospace", 32)

        self.GameInfo = GameInfo()

        self.SaveGame = json.loads(open("savegame.json").read())

        self.ShakeActive = False

        self.StageTime = self.GetStageTime(0)
        self.NextStageTimer = -1
        self.ShowNextStage = False

        self.UserInputStarted = False

        self.HitTimer = 0

        self.IsInitialized = True

    def ShowStageAnim(self):
        self.ShowNextStage = True
        self.NextStageTimer = 0

    def GameOver(self):

        if self.GameInfo.CurrentDistance > self.SaveGame["Record"]:
            self.SaveGame["Record"] = self.GameInfo.CurrentDistance
            open("savegame.json","w").write(json.dumps(self.SaveGame))
            self.GameInfo.IsNewRecord = True

        statemanager.Set(2, self.GameInfo)

    def Shake(self):
        self.ShakeActive = True
        self.ShakeTimer = 0
        Entity.Shake = [0,0]

    def Reset(self):

        Entity.Offset = 0

        del self.Planet
        del self.Ship
        del self.Fuel
        for asteroid in self.Asteroids:
            del asteroid

        self.IsInitialized = False

    def Input(self, _fTime, _screen):

        if not self.IsInitialized:
            return

        keys = pygame.key.get_pressed()

        if not self.Fuel.IsEmpty():
            self.TimeIsSlow = keys[pygame.K_SPACE]
            if self.TimeIsSlow:
                self.Fuel.TimeSlow(_fTime)

            if keys[pygame.K_UP]:
                self.Ship.Boost(_fTime, _screen)
                self.Fuel.Boost(_fTime)
                self.UserInputStarted = True
        else:
            self.TimeIsSlow = False

        if keys[pygame.K_LEFT]:
            self.Ship.RotateLeft(_fTime)
        if keys[pygame.K_RIGHT]:
            self.Ship.RotateRight(_fTime)

    def Update(self, _fTime):

        if not self.IsInitialized:
            return

        if not self.UserInputStarted:
            return

        if self.ShakeActive:
            self.ShakeTimer += _fTime * 2
            Entity.Shake[0] = mytween(easeOutElasticSmall, 30,0,self.ShakeTimer)
            Entity.Shake[1] = mytween(easeOutElasticSmall, 30,0,self.ShakeTimer)
            if self.ShakeTimer > 1:
                self.ShakeActive = False

        self.GameInfo.Time += _fTime

        if self.TimeIsSlow:
            _fTime *= Config.TIMESLOWMULTIPLIER

        self.Ship.Update(_fTime)
        currentSpeed = vector.length(self.Ship.Entity.Velocity)
        if self.GameInfo.MaxSpeed < currentSpeed:
            self.GameInfo.MaxSpeed = currentSpeed

        for asteroid in self.Asteroids:

            if asteroid.IsDead():
                asteroid.Spawn(
                    [random.randrange(0,Config.WINDOWWIDTH),Entity.Offset+random.randrange(Config.ASTEROIDMINSPAWN,Config.ASTEROIDMAXSPAWN)], #position
                    [random.randrange(-40,40),random.randrange(-40,40)], #velocity
                    random.randrange(Config.ASTEROIDMINRADIUS,Config.ASTEROIDMAXRADIUS), #scale
                    random.randrange(-4,4)) #rotationSpeed

            asteroid.Update(_fTime)
            asteroid.Entity.UpdateXLimit()

            if self.HitTimer <= 0:
                if utils.IsCollision(self.Ship.Entity, asteroid.Entity):
                    force = utils.Collide(self.Ship.Entity,asteroid.Entity)
                    self.Fuel.Collision(math.fabs(force))
                    self.GameInfo.AsteroidsTouched += 1
                    self.Shake()
                    self.HitTimer = 0.5

        if self.HitTimer > 0:
            self.HitTimer -= _fTime

        for star in self.Stars:
            if star.IsDead():
                star.Spawn(
                    [random.randrange(0,Config.WINDOWWIDTH),Entity.Offset+random.randrange(-Config.WINDOWHEIGHT,0)], #position
                    random.randrange(Config.STARMINRADIUS,Config.STARMAXRADIUS)) # scale

        stageId = self.GetStageId(self.GameInfo.CurrentDistance)
        if self.GameInfo.CurrentStageId < stageId:
            self.GameInfo.CurrentStageId = stageId
            self.StageTime += self.GetStageTime(stageId)
            self.Fuel.Refill()
            self.ShowStageAnim()

        if self.ShowNextStage:
            self.NextStageTimer += _fTime
            if self.NextStageTimer > 2:
                self.ShowNextStage = False

        self.StageTime -= _fTime;
        if self.StageTime < 0:
            self.StageTime = 0;

        if self.StageTime == 0:
            self.Fuel.TimeOver(_fTime)

        if self.Fuel.IsEmpty():
            self.Ship.Entity.Velocity = vector.multiply(self.Ship.Entity.Velocity,1-_fTime)
            if vector.length(self.Ship.Entity.Velocity) < 1:
                self.GameOver()

    def GetStageId(self, distance):
        return math.floor(distance/5000)

    def GetStageDistance(self, _id):
        return _id * 5000;

    def GetStageTime(self, _id):
        t = 20 - _id * 5
        if t < 5:
            return 5
        return t

    def DrawStageTimer(self, _screen, _shake):
        seconds = self.StageTime % 60
        secondsStr = str(seconds)
        if seconds < 10:
            secondsStr = "0"+str(seconds)

        minutes = int(math.floor(self.StageTime / 60))
        minutesStr = str(minutes)
        if minutes < 10:
            minutesStr = "0"+str(minutes)

        col = Config.WHITE
        if self.StageTime < 10:
            col = Config.RED

        text = minutesStr+":"+secondsStr
        label = self.FontBig.render(text, 1, col)
        _screen.blit(label, (Config.WINDOWWIDTH-96+_shake[0], 66+_shake[1]))

    def DrawSpeed(self, _screen, _shake):
        speed = int(vector.length(self.Ship.Entity.Velocity))
        speedStr = str(speed)
        if speed < 10:
            speedStr = "000"+speedStr
        elif speed < 100:
            speedStr = "00"+speedStr
        elif speed < 1000:
            speedStr = "0"+speedStr

        text = speedStr+"km/h"
        label = self.Font.render(text, 1, Config.WHITE)
        _screen.blit(label, (Config.WINDOWWIDTH-82+_shake[0], 128+_shake[1]))

    def Draw(self, _screen, _drawUi):

        if not self.IsInitialized:
            return

        if not self.TimeIsSlow:
            _screen.fill(Config.BLACK)
        else:
            _screen.fill(Config.DARKBLUE)

        for star in self.Stars:
            star.Draw(_screen)

        self.Planet.DrawFront(_screen)

        self.Ship.Draw(_screen)

        for asteroid in self.Asteroids:
            asteroid.Draw(_screen)
            asteroid.DrawMarker(_screen)

        self.Planet.DrawBack(_screen)

        self.GameInfo.CurrentDistance = int(-Entity.Offset)

        if not _drawUi:
            return

        self.Fuel.Draw(_screen)

        shake = Entity.Shake

        # draw stage line
        nextStagePosition = - Entity.Offset + Config.WINDOWHEIGHT*0.75 - self.GetStageDistance(self.GameInfo.CurrentStageId+1)
        pygame.draw.line(_screen, Config.RED, [0, nextStagePosition], [Config.WINDOWWIDTH,nextStagePosition], 5)

        # render text
        self.DrawStageTimer(_screen, shake)

        self.DrawSpeed(_screen, shake)

        if self.ShowNextStage:
            y = Config.WINDOWHEIGHT/2-16

            label = self.FontBig.render("Stage "+str(int(self.GameInfo.CurrentStageId+1)), 1, Config.WHITE)
            pos = mytween(easeOutElasticSmall, -200,y,self.NextStageTimer)
            if self.NextStageTimer > 1:
                pos = mytween(easeInElasticSmall, y,800,self.NextStageTimer-1)
            _screen.blit(label, (200, pos))