abstract class GameObject {
  
  public static final int TYPE_ASTEROID = 0;
  public static final int TYPE_SHIP = 1;
  public static final int TYPE_BULLET = 2;
  public static final int TYPE_EXPLOSION = 3;
  
  int id;
  int type;
  boolean visible = false;
  
  public GameObject(int type) {
    this.type = type;
  }
  
  public abstract boolean collide(float[][] polygon);
  public abstract void update(float fraction);
  public abstract void draw();
  public abstract String serialize();
}

class Explosion extends GameObject {
  
  float scale = 0.0f;
  float force = 4.0f;
  float[] position = new float[2];
  float[][] lines = new float[10][2];
  
  public Explosion() {
    super(GameObject.TYPE_EXPLOSION);
  }
  
  public String serialize() {
    return "e:" + str(id) + ":" + str(position[0]) + ":" + str(position[1]) + ":" + str(scale) + ":" + str(force); 
  }
  
  public void sync(float px, float py, float f) {
    this.position[0] = px;
    this.position[1] = py;
    this.force = f;
    this.scale = 0.0f;
    
    for (int i = 0; i < lines.length; i+= 2) {
      float rad = 2 * PI * random(1.0f);
      px = cos(rad);
      py = sin(rad);
      
      lines[i + 0][0] = px;
      lines[i + 0][1] = py;
      lines[i + 1][0] = px * 2;
      lines[i + 1][1] = py * 2;
    }
  }
  
  public boolean collide(float[][] polygon) {
    return false;
  }
  
  public void update(float fraction) {
    if (this.scale >= this.force) {
      this.visible = false;
      return;
    }
    
    this.scale += this.force * fraction;
  }
  
  public void draw() {
    stroke(255);
    for (int i = 0; i < lines.length; i+= 2) {
      line(position[0] + lines[i + 0][0] * scale, position[1] + lines[i + 0][1] * scale, position[0] + lines[i + 1][0] * scale, position[1] + lines[i + 1][1] * scale); 
    }
  }
}

class Bullet extends GameObject {
  
  static final float BULLET_SPEED = 250.0f;
  static final float BULLET_LIFE = 2.0f;
  
  float[] position = new float[2];
  float[] force = new float[2];
  float[] direction = new float[2];
  float size = 10.0f;
  float life = 0.0f;
  color entityColor = color(255);
  int shipId;
  
  public Bullet() {
    super(GameObject.TYPE_BULLET);
  }
  
  public String serialize() {
    return "b:" + str(id) + ":" + str(position[0]) + ":" + str(position[1]) + ":" + str(force[0]) + ":" + str(force[1]) + ":" + str(life) + ":" + str(entityColor) + ":" + str(shipId); 
  }
  
  public void sync(float px, float py, float fx, float fy, float life, int ec, int sid) {
    this.position[0] = px;
    this.position[1] = py;
    this.force[0] = fx;
    this.force[1] = fy;    
    float fl = sqrt( fx * fx + fy * fy);
    this.direction[0] = fx / fl * this.size;
    this.direction[1] = fy / fl * this.size;
    this.life = life;
    this.entityColor = ec;
    this.shipId = sid;
  }
  
  public boolean collide(float[][] polygon) {
    return pointInPolygon(position, polygon) || pointInPolygon(position[0] - direction[0], position[1] - direction[1], polygon);
  }
  
  public void update(float elapsed) {
    
    this.life -= elapsed;
    this.position[0] += this.force[0] * elapsed;
    this.position[1] += this.force[1] * elapsed;
    
    wrap(this.position, this.size);
  }
  
  public void draw() {    
    stroke(entityColor);
    line(position[0], position[1], position[0] - direction[0], position[1] - direction[1]);
  }
}

class Asteroid extends Entity {
  float[][] model = new float[][] {
    new float[] { -5.0, -7.5 },
    new float[] { +2.5, -7.5 },
    new float[] { +7.5, -2.5 },
    new float[] { +7.5, +2.5 },
    new float[] { +2.5, +7.5 },
    new float[] { -2.5, +7.5 },
    new float[] { -7.5, +2.5 },
    new float[] { -7.5, -2.5 },
  };
  float[][] modelTranslated = new float[8][2];
  
  public Asteroid() {
    super(GameObject.TYPE_ASTEROID);
    
    this.entityColor = color(130);
  }
  
  public boolean collide(float[][] polygon) {    
    for (int p = 0; p < modelTranslated.length; ++p) {
      if (pointInPolygon(modelTranslated[p], polygon)) {
        return true;
      }
    }
    return false;
  }
  
  public void update(float fraction) {
    super.update(fraction);
    
    for (int i = 0; i < model.length; ++i) {
      to_world(modelTranslated[i], direction, size, position, model[i]);
    }
  }
  
  public void draw() {
    stroke(entityColor);
    drawPolygon(modelTranslated);
  }
}

class Ship extends Entity {
  float[][] model1 = new float[][] {
    new float[] { 7 -7.32448, 0 -14.8034 },
    new float[] { 0 -7.32448, 20-14.8034 },
    new float[] { 3 -7.32448, 17-14.8034 },
    new float[] { 12-7.32448, 17-14.8034 },
    new float[] { 15-7.32448, 20-14.8034 }
  };
  float[][] model2 = new float[][] {
    new float[] { 4 -7.32448, 19-14.8034 },
    new float[] { 11-7.32448, 19-14.8034 }
  };
  float[][] model1Translated = new float[5][2];
  float[][] model2Translated = new float[2][2];
  
  public Ship() {
    super(GameObject.TYPE_SHIP);
  }
  
  public boolean collide(float[][] polygon) {
    for(int p = 0; p < model1Translated.length; ++p) {
      if (pointInPolygon(model1Translated[p], polygon)) {
        return true;
      }
    }
    return false;
  }
  
  public void update(float fraction) {
    super.update(fraction);
    
    for (int i = 0; i < model1.length; ++i) {
      to_world(model1Translated[i], direction, 1.0f, position, model1[i]);
    }
    
    for (int i = 0; i < model2.length; ++i) {
      to_world(model2Translated[i], direction, 1.0f, position, model2[i]);
    }
  }
  
  public void draw() {
    stroke(entityColor);
    drawPolygon(model1Translated);
    
    if (this.accleration) {
      stroke(128, 255, 255);
      drawPolygon(model2Translated);
    }
  }
}

abstract class Entity extends GameObject {
  
  final float TURN_SPEED = 3.0f;
  final float ACCLERATION = 100.0f;
  
  color entityColor = color(255);
  float size = 1.0f;
  float hsize = 0.5f;
  float direction = 0.0f;
  float rotation = 0.0f;
  float[] position = new float[] { 0, 0 };
  float[] force = new float[] { 0, 0 };
  boolean accleration = false;
  boolean turnLeft = false;
  boolean turnRight = false;
  
  public Entity(int type) {
    super(type);
  }
  
  public String serialize() {    
    return (type == TYPE_SHIP ? "s:" : "a:") + str(id) + ":" + str(position[0]) + ":" + str(position[1]) + ":" + str(force[0]) + ":" + str(force[1]) + ":" + str(size) + ":" + str(direction) + ":" + str(rotation) + ":" + str(accleration) + ":" + str(turnLeft) + ":" + str(turnRight) + ":" + str(entityColor); 
  }
  
  public void sync(float px, float py, float fx, float fy, float size, float d, float r, boolean a, boolean tl, boolean tr, color ec) {
    this.position[0] = px;
    this.position[1] = py;
    this.force[0] = fx;
    this.force[1] = fy;
    this.size = size;
    this.hsize = size / 2.0f;
    this.direction = d;
    this.rotation = r;
    this.accleration = a;
    this.turnLeft = tl;
    this.turnRight = tr;
    this.entityColor = ec;
  }
  
  public void update(float fraction) {
    
    if (accleration) {
      force[0] += ACCLERATION * fraction * cos(direction - PI / 2);
      force[1] += ACCLERATION * fraction * sin(direction - PI / 2);
      
      forceLimit(force, ACCLERATION);
    }
    
    if (turnLeft) {
      direction -= TURN_SPEED * fraction;
    }
    
    if (turnRight) {
      direction += TURN_SPEED * fraction;
    }
    
    direction += rotation * fraction;
    
    position[0] += force[0] * fraction;
    position[1] += force[1] * fraction;
    
    wrap(position, size);
  }
}