#!/usr/bin/python3

import json
import socket
from math import sqrt
from random import random
from time import time, sleep

# Constants
HOST = "127.0.0.1"
PORT = 54321
NAME = "ZargBot"
BOMB_RANGE = 3
BOMB_TICKS = 5
ENEMY = 2
STONE = 3
WALL = 4
TICK_TIME = 0.1
SAFE_THRESHOLD = 0.1
ATTACK_RANGE = BOMB_RANGE - 1

# Globals
sock = socket.socket()
data = {}
weightMap = []

def cross_itr(rad):
    rng = range(-rad, rad+1)
    return [(i,j) for i in rng for j in rng if i == 0 or j == 0]

def wm(x, y):
    if x < 0 or y < 0 or len(weightMap) < 1 or \
            x >= len(weightMap[0]) or y >= len(weightMap):
        return 1
    return weightMap[y][x]

def distance(x0, y0, x1, y1):
    return sqrt(pow(x1 - x0, 2) + pow(y1 - y0, 2))

def send(msg):
    sock.send((msg+'\n').encode())

def receive():
    msg = ""
    while True:
        try:
            msg += sock.recv(1024).decode().split('\n')[0]
            final = json.loads(msg)
            return final
        except ValueError:
            pass

def bomb_proximity(x, y):
    for bomb in data["bombs"]:
        i = bomb["x"]
        j = bomb["y"]
        if i == x and j == y:
            return True
        if distance(i, j, x, y) < BOMB_RANGE:
            return True
    return False

def safe_weight(x, y, dist):
    return max(0, min(wm(x, y), 1) - (1 - bomb_weight(dist)))

def is_safe(x, y, dist):
    return safe_weight(x, y, dist) <= SAFE_THRESHOLD

def can_escape(x, y):
    for i, j in cross_itr(1):
        if (i == 0 and j == 0):
            continue
        if escape_dir_weight(x, y, i, j) <= SAFE_THRESHOLD:
            return True
    return False

def escape_dir_weight(x, y, i, j):
    weight = 0
    if i == 0 and j == 0:
        return safe_weight(x + i, y + j, 0) + SAFE_THRESHOLD / 2.0
    steps = 0
    for n in range(1, BOMB_RANGE + 1):
        xx = x + i * n
        yy = y + j * n
        w = safe_weight(xx, yy, n)
        steps += 1
        if w > SAFE_THRESHOLD:
            weight += w
        elif n == BOMB_RANGE:
            break
        else:
            w = min(safe_weight(xx + abs(j), yy + abs(i), n + 1), \
                    safe_weight(xx - abs(j), yy - abs(i), n + 1))
            if w > SAFE_THRESHOLD:
                weight += w
                steps += 1
            else:
                break
    weight /= steps
    return weight
    
def attack(x, y, target):
    hinders = []
    if ATTACK_RANGE > 1:
        for i, j in cross_itr(ATTACK_RANGE):
            if wm(x + i, y + j) == WALL:
               hinders.append((i, j))
    for i, j in cross_itr(ATTACK_RANGE):
        hindered = False
        for h in hinders:
            if (i < h[0] < 0 or 0 < h[0] < i or \
                    j < h[1] < 0 or 0 < h[1] < j):
                hindered = True
        if not hindered and wm(x + i, y + j) == target and can_escape(x, y):
            send("BOMB")
            return True
    return False

def escape(x, y):
    dirs = {}
    for i, j in cross_itr(1):
        dirs[escape_dir_weight(x, y, i, j)] = (i, j)
    keys2 = list(dirs.keys())
    c = dirs[sorted(keys2)[0]]
    move(c[0], c[1])
    
def move(x, y):
    if x == 1:
        send("RIGHT")
    elif x == -1:
        send("LEFT")
    elif y == -1:
        send("UP")
    elif y == 1:
        send("DOWN")

def random_move(x, y):
    dirs = []
    for i, j in cross_itr(1):
        if is_safe(x + i, y + j, 1):
            dirs.append((i, j))
    i = int(random() * len(dirs))
    c = dirs[i]
    move(c[0], c[1])

def say(msg):
    send("SAY " + msg)

def bomb_weight(ticks):
    return 1 - 1 / BOMB_TICKS * max(1, min(ticks, BOMB_TICKS))

def genWeightMap():
    global weightMap
    weightMap = []
    for y in range(0, data["height"]):
        weightMap.append([])
        for x in range(0, data["width"]):
            weight = 0
            tile = data["map"][y][x]
            if tile == '+':
                weight = WALL
            if tile == '#':
                weight = STONE
            weightMap[y].append(weight)
    for player in data["players"]:
        x = player["x"]
        y = player["y"]
        weightMap[y][x] = ENEMY
    for bomb in data["bombs"]:
        x = bomb["x"]
        y = bomb["y"]
        for i, j in cross_itr(BOMB_RANGE - 1):
            xx = max(0, min(x + i, data["width"] - 1))
            yy = max(0, min(y + j, data["height"] - 1))
            if i == 0 and j == 0:
                weight = 1
            else:
                weight = distance(x, y, xx, yy) / BOMB_RANGE + bomb_weight(bomb["state"])
            weightMap[yy][xx] = max(weightMap[yy][xx], max(0, min(weight, 1)))

#connect & play
sock.connect((HOST, PORT))
send("JSON")
send("NAME " + NAME)
while True:
    data = receive()
    if data["type"] == "status update":
        genWeightMap()
        x = data["x"]
        y = data["y"]
        if bomb_proximity(x, y):
            print("escaping")
            escape(x, y)
        elif not attack(x, y, STONE):
            if not attack(x, y, ENEMY):
                print("rand")
                random_move(x, y)
            else:
                print("attacking enemy")
        else:
            print("excavating")
sock.close()

