#include <iostream>
#include <cstring>

#include "OdinAI/NetworkManager.h"
#include "OdinAI/DebugUtil.h"
#include "OdinAI/SharedDefs.h"

namespace OdinAI
{
	bool NetworkManager::Connect(const char *ip, const char *port, const AsyncRecvCallback &asyncRecvCallback)
	{
		m_asyncRecvCallback = asyncRecvCallback;

		addrinfo *pResult = NULL, *pCurrent = NULL, hints;
		int iResult;
		//Initialize WinSocket
		iResult = WSAStartup(MAKEWORD(2, 2), &m_wsaData);
		if(iResult != 0)
		{
	#ifdef DEBUG_NETWORK
			std::cout << "WSAStartup failed: " << iResult << std::endl;
	#endif
			return false;
		}

		memset(&hints, 0, sizeof(hints));
		hints.ai_family = AF_UNSPEC;
		hints.ai_socktype = SOCK_STREAM;
		hints.ai_protocol = IPPROTO_TCP;

		//Get address info
		iResult = getaddrinfo(ip, port, &hints, &pResult);
		if(iResult != 0)
		{
	#ifdef DEBUG_NETWORK
			std::cout << "getaddrinfo failed: " << iResult << std::endl;
	#endif
			WSACleanup();
			return false;
		}

		//Create socket for listening
		m_clientSocket = socket(pResult->ai_family, pResult->ai_socktype, pResult->ai_protocol);
		if(m_clientSocket == INVALID_SOCKET)
		{
	#ifdef DEBUG_NETWORK
			std::cout << "Error when executing socket(): " << WSAGetLastError() << std::endl;
	#endif
			freeaddrinfo(pResult);
			WSACleanup();
			return false;
		}

		//Bind the socket to default address
		iResult = connect(m_clientSocket, pResult->ai_addr, (int)pResult->ai_addrlen);
		if(iResult == SOCKET_ERROR)
		{
	#ifdef DEBUG_NETWORK
			std::cout << "Error when calling connect: " << WSAGetLastError() << std::endl;
	#else
			std::cout << "Unable to connect to server!" << std::endl;
	#endif
			freeaddrinfo(pResult);
			closesocket(m_clientSocket);
			WSACleanup();
			return false;
		}

		freeaddrinfo(pResult);

		if (m_clientSocket == INVALID_SOCKET)
		{
			std::cout << "Unable to connect to server!" << std::endl;
			WSACleanup();
			return false;
		}

		m_netEvent = WSACreateEvent();

		if(WSAEventSelect(m_clientSocket, m_netEvent, FD_READ | FD_WRITE))
		{
	#ifdef DEBUG_NETWORK
			std::cout << "Error when calling WSAEventSelect: " << WSAGetLastError() << std::endl;
	#endif
			WSACleanup();
			return false;
		}

		return true;
	}

	void NetworkManager::Update()
	{
		WSANETWORKEVENTS eventStatus;

		if(WSAEnumNetworkEvents(m_clientSocket, m_netEvent, &eventStatus))
		{
	#ifdef DEBUG_NETWORK
			std::cout << "Error when calling WSAEnumNetworkEvents: " << WSAGetLastError() << std::endl;
	#endif
		}

		//Check if we got a packet to process
		if(eventStatus.lNetworkEvents & FD_READ)
		{
			HandleReceive();
		}

		if(eventStatus.lNetworkEvents & FD_WRITE)
		{
			m_bCanSend = true;
            Send(m_outBuffer, m_currentOutPos);
            m_currentOutPos = 0;
		}
	}
	
	void NetworkManager::HandleReceive()
	{
		int iNumBytesReceived = recv(m_clientSocket, &m_netBuffer[m_currentInPos], NETWORK_BUFFER_SIZE - m_currentInPos, 0);

		//Close socket when we received 0 bytes.
		if(iNumBytesReceived == 0)
		{
			// We should exit...
		}

		m_asyncRecvCallback(m_netBuffer, m_currentInPos + iNumBytesReceived);
	}

	void NetworkManager::Send(const void *data, int size)
	{
		if(m_bCanSend)
		{
			int numBytesSendt = 0;
			if((numBytesSendt = send(m_clientSocket, static_cast<const char*>(data), size, 0)) == SOCKET_ERROR)
			{
				if(WSAGetLastError() == WSAEWOULDBLOCK)
				{
					//The internal socket buffer is full, so wait for the socket to be ready again.
					WSAWaitForMultipleEvents(1, &m_netEvent, FALSE, WSA_INFINITE, FALSE);
				}
				else
				{
		#ifdef _DEBUG
					std::cout << "Could not send message to server! Disconnecting..." << std::endl;
		#endif
				}
			}
		}
        else
        {
            //std::cout << "Couldn't send package." << std::endl;
            memcpy(m_outBuffer + m_currentOutPos, data, size);
            m_currentOutPos += size;
        }

		//memset(m_outBuffer, 0, bufferSize+1);
	}

	void NetworkManager::Close()
	{
		closesocket(m_clientSocket);
	
		WSACloseEvent(m_netEvent);
		WSACleanup();
	}
}
