#version 420

// 1 divided by width and height
uniform vec2 viewport_mul;

// G Buffer
uniform sampler2D normal_buffer;
uniform sampler2D position_buffer;
uniform sampler2D depth_buffer;
uniform sampler2D spot_depth_buffer;

// Light uniforms
uniform vec3 light_position;
uniform vec3 light_direction;
uniform vec4 light_color;
uniform float spot_cos_cutoff;
uniform float spot_cos_cutoff_start;
uniform float spot_exponent;

uniform mat4 spot_view_projection;

uniform vec3 camera_position;
uniform vec2 camera_near_far;

in vec2 out_TexCoord;

layout(location = 0) out vec4 diffuse;
layout(location = 1) out vec4 specular;

const float ts = 1 / 1024.0;

vec2 offsets[] = vec2[](vec2(0,0), vec2(ts,0), vec2(-ts,0), vec2(0,ts), vec2(0,-ts));

const float BIAS = 0.005;

float linearize_depth_value(float depth)
{
	float linear_depth = (2 * camera_near_far.x) / (camera_near_far.y + camera_near_far.x - depth * (camera_near_far.y - camera_near_far.x));
	return linear_depth;
}

void main()
{
	vec2 tex_coord = gl_FragCoord.xy * viewport_mul;

	vec4 normal_and_specular = texture2D(normal_buffer, tex_coord);

	vec3 normal = normalize(normal_and_specular.rgb);
	vec3 position = texture2D(position_buffer, tex_coord).rgb;
	vec3 incident = normalize ( light_position - position);

	float spot_effect = dot(normalize(light_direction), -incident);
	if (spot_effect < spot_cos_cutoff)
		discard;

	vec4 clip = spot_view_projection * vec4(position, 1.0);
	clip /= clip.w;
	clip = (clip + 1) / 2;

	float intensity = 1;

	for (int i = 0; i < 5; i++)
	{
		if ( linearize_depth_value(texture2D(spot_depth_buffer, clip.xy + offsets[i]).r) + BIAS < linearize_depth_value(clip.z))
		{
			intensity -= 0.2;
		}
	}

	float attenuation = 1;
	if (spot_effect < spot_cos_cutoff_start)
	{
		float diff = spot_cos_cutoff_start - spot_cos_cutoff;
		attenuation = (spot_effect - spot_cos_cutoff) / diff;
	}

	vec3 viewDir = normalize ( camera_position - position);
	
	vec3 halfDir = normalize ( incident + viewDir );
	float lambert = clamp ( dot ( incident , normal ) ,0.0 ,1.0);
	float rFactor = clamp ( dot ( halfDir , normal ) ,0.0 ,1.0);
	
	float sFactor = pow ( rFactor , normal_and_specular.a ); // Shininess in worldnormal.a

	diffuse  = vec4 ( light_color.xyz * intensity * lambert * attenuation, 1.0);
	specular = vec4 ( light_color.xyz * intensity * sFactor * 0.33 * attenuation,1.0);
}