import time

class CallWrapper(object):
	def __init__(self, wrapped, checker):
		self.wrapped = wrapped
		self.checker = checker
		self.cache = {}

	def __getattr__(self, name):
		func = self.cache.get(name)
		if func is None:
			att = getattr(self.wrapped, name)
			func = self.checker(self.wrapped, name, att)
			self.cache[name] = func
		return func

class CheckedCallFailure(Exception):
	pass

class CheckedGlFunc(object):
	GL_ERRORS = {
		0: "GL_NO_ERROR",
		0x500: "GL_INVALID_ENUM",
		0x501: "GL_INVALID_VALUE",
		0x502: "GL_INVALID_OPERATION",
		0x505: "GL_OUT_OF_MEMORY"
	}
	def __init__(self, gl, name, func):
		self.gl = gl
		self.name = name
		self.func = func

	def __call__(self, *a, **kw):
		er = self.err()
		retval = self.func(*a, **kw)
		er2 = self.err()
		if er2 != er:
			raise CheckedCallFailure(self.fmt_err(a, kw, er2))
		return retval

	def fmt_err(self, a, kw, er):
		as_ = ", ".join(map(repr, a))
		kws = ", ".join("%s=%s" % (k, repr(v)) for k, v in kw.items())
		if kws != "": kws = ", " + kws
		return "%s(%s%s): %s %s" % (self.name, as_, kws, er, self.GL_ERRORS.get(er, "?"))

	def err(self):
		return self.gl.glGetError()

class CheckedGl(CallWrapper):
	def __init__(self, wrapped):
		CallWrapper.__init__(self, wrapped, CheckedGlFunc)

class TimedFunc(object):
	def __init__(self, wrapped, name, func):
		self.name = name
		self.func = func
		self.ncalls = 0
		self.calldurations = []

	def __call__(self, *a, **kw):
		self.ncalls += 1
		stime = time.time()
		retval = self.func(*a, **kw)
		self.calldurations.append(time.time() - stime)
		return retval

	def dump(self):
		if self.ncalls == 0:
			return

		mind = 1000 * min(self.calldurations)
		maxd = 1000 * max(self.calldurations)
		tot = 1000 * sum(self.calldurations)
		avgd = tot / len(self.calldurations)

		print "%s: %d call(s), duration min/max/avg = %.3f/%.3f/%.3f ms, tot=%.3f ms" % (
				self.name, self.ncalls, mind, maxd, avgd, tot)

class TimedCalls(CallWrapper):
	def __init__(self, wrapped):
		CallWrapper.__init__(self, wrapped, TimedFunc)

	def dump(self):
		funcs = self.cache.values()
		funcs.sort(key=lambda x: -x.ncalls)
		for func in funcs:
			func.dump()
