package skyport;

import java.util.Iterator;
import java.util.List;

import skyport.algorithms.astar.Node;
import skyport.algorithms.astar.Nodemap;
import skyport.enums.Tile;
import skyport.json.MapBean;
import skyport.json.PlayerBean;

public class Game {

	private int turn;
	private List<PlayerBean> players;
	private Resources resources;
	private Tilemap tilemap;
	private Nodemap nodemap;

	public Game() {
		turn = 0;
		players = null;
		resources = null;
		tilemap = new Tilemap();
		nodemap = null;
	}

	public int getTurn() {
		return turn;
	}

	public void setTurn(int turn) {
		this.turn = turn;
	}

	public List<PlayerBean> getPlayers() {
		return players;
	}

	public void setPlayers(final List<PlayerBean> players) {
		this.players = players;
	}

	public Tilemap getTilemap() {
		return tilemap;
	}

	public void setTilemap(final Tilemap map) {
		this.tilemap = map;
	}

	public Resources getResources() {
		return resources;
	}

	public Nodemap getNodemap() {
		return nodemap;
	}

	public void updateMap(final MapBean mapBean) {
		tilemap.setMapData(mapBean);
	}

	public void initialize(final MapBean mapBean, final List<PlayerBean> players) {
		tilemap.setMapData(mapBean);
		Tile[][] tiles = tilemap.getTiles();
		nodemap = new Nodemap(tilemap);
		resources = new Resources(tiles, players);
	}

	public PlayerBean getCurrentPlayer() {
		return players.get(0);
	}

	public PlayerBean getPlayerByName(final String playerName) {
		PlayerBean player = null;

		for (PlayerBean currentPlayer : players) {
			if (currentPlayer.getName().equals(playerName)) {
				player = currentPlayer;
				break;
			}
		}

		return player;
	}

	public boolean opponentAt(final Coordinate pos) {
		boolean opponentAt = false;

		Iterator<PlayerBean> it = players.iterator();
		it.next(); // Skip current player

		while (it.hasNext() && !opponentAt) {
			PlayerBean player = it.next();
			opponentAt = pos.equals(player.getPosition());
		}

		return opponentAt;
	}

	public Coordinate getCoordinateForPlayer(int index) {
		return players.get(index).getPosition();
	}

	public PlayerBean getClosestPlayerFrom(final Coordinate from) {
		PlayerBean closestPlayer = null;
		int distance = Integer.MAX_VALUE;
		int lowestHp = Integer.MAX_VALUE;

		Iterator<PlayerBean> it = players.iterator();
		it.next(); // Skip the current player

		while (it.hasNext()) {
			PlayerBean player = it.next();
			Coordinate to = player.getPosition();
			List<Node> nodes = nodemap.getRoute(from, to);

			if (nodes != null) {
				int steps = nodes.size();
				int hp = player.getHealth();

				if (steps < distance || (steps == distance && hp < lowestHp)) {
					closestPlayer = player;
					distance = steps;
					lowestHp = hp;
				}
			}
		}

		return closestPlayer;
	}

	public Coordinate getCoordinateToClosestTileFrom(final Coordinate from,
			Tile tile) {

		Coordinate result = null;
		boolean score = false;
		int x = from.getX();
		int y = from.getY();
		int maxRange = Math.max(tilemap.getJLength(), tilemap.getKLength());

		for (int range = 0; range < maxRange; ++range) {
			int x1 = x - range;
			int x2 = x + range;

			for (int currentX = x1; currentX <= x2 && !score; ++currentX) {
				int diffX = x - currentX;
				int y1 = y - range - (diffX > 0 ? 0 : diffX);
				int y2 = y + range - (diffX < 0 ? 0 : diffX);

				for (int currentY = y1; currentY <= y2 && !score; ++currentY) {
					Coordinate pos = new Coordinate(currentX, currentY);

					if (tilemap.getTileAt(pos) == tile) {
						result = pos;
						score = true;
					}
				}
			}
		}

		return result;
	}
}
