#!/usr/bin/env python
import math
import random
import sys
import time

WIDTH = 80
HEIGHT = 24
FRAME_LEN = 0.05

class Frame(object):
    def __init__(self, cols, rows):
        self.rows = rows
        self.cols = cols
        self.pixels = [[" " for col in xrange(self.cols)] for row in xrange(self.rows)]

    def __getitem__(self, (x, y)):
        return self.pixels[y][x]
    
    def __setitem__(self, (x, y), v):
        if 0 <= y < self.rows and 0 <= x < self.cols:
            self.pixels[y][x] = v

secs = lambda s: s / FRAME_LEN

def escape(code):
    return chr(27) + "[" + code

def gfxmode(*codes):
    return escape(";".join(map(str, codes))) + "m"

def render(frame):
    height = frame.rows
    string = ""
    for y in xrange(height):
        last_line = y == height-1
        width = frame.cols - (1 if last_line else 0)
        for x in xrange(width):
            string += frame[x, y]
        if not last_line:
            string += "\n"
    sys.stdout.write(clear + string + reset)
    sys.stdout.flush()

def sine(frame, t_scale, x_scale, amplitude, y_offset, gfx = None, 
char_func = None):
    if char_func == None:
        char_func = lambda: "*"
    for x in xrange(WIDTH): 
        val = amplitude*math.sin(x_scale*x + t_scale*t)
        y = int(round(val+y_offset))
        pixel = char_func()
        if gfx != None:
            pixel = gfxmode(gfx) + pixel + reset
        frame[x, y] = pixel
    
#clear = escape("2J") + escape("H")
clear = escape("H")
reset = gfxmode("0")

def scale(x, min0, max0, min1, max1):
    return (float(x)-min0) / (max0-min0) * (max1-min1) + min1

def line(frame, x0, y0, x1, y1, char_func = None):
    if char_func == None:
        char_func = lambda: "*"
    steep = abs(y1-y0) > abs(x1-x0)
    if steep:
        temp = x0
        x0 = y0
        y0 = temp
        temp = x1
        x1 = y1
        y1 = temp
    if x0 > x1:
        temp = x0
        x0 = x1
        x1 = temp
        temp = y0
        y0 = y1
        y1 = temp
    dx = x1-x0
    dy = abs(y1-y0)
    error = dx/2
    y = y0
    ystep = 1 if y0 < y1 else -1
    for x in xrange(x0, x1+1):    
        if steep:
            frame[y, x] = char_func()
        else:
            frame[x, y] = char_func()
        error -= dy
        if error < 0:
            y += ystep
            error += dx

def text(frame, string, x0, y0, escape = None):
    if x0 < 0:
        string = string[-x0:]
        x0 = 0
    for x in xrange(x0, WIDTH):
        if string:
            frame[x, y0] = string[0]
            if escape and x == x0:
                frame[x, y0] = escape + frame[x, y0]
            string = string[1:]
        else:
            break

def matmul(m1, m2):
    m2_is_vector = not hasattr(m2[0], "__getitem__")
    if m2_is_vector:
        m2 = [[x] for x in m2]
    m1_rows = len(m1)
    m1_cols = len(m1[0])
    m2_rows = len(m2)
    m2_cols = len(m2[0])
    assert m1_cols == m2_rows
    m3_rows = m1_rows
    m3_cols = m2_cols
    m3 = [[sum(m1[row][k] * m2[k][col] for k in xrange(m1_cols)) for col in xrange(m3_cols)] for row in xrange(m3_rows)]
    if m2_is_vector:
        m3 = [x for [x] in m3]
    return m3

def rotx(theta):
    return [[1.0, 0.0, 0.0, 0.0], [0.0, math.cos(theta), -math.sin(theta), 0.0], [0.0, math.sin(theta), math.cos(theta), 0.0], [0.0, 0.0, 0.0, 1.0]]

def roty(theta):
    return [[math.cos(theta), 0.0, math.sin(theta), 0.0], [0.0, 1.0, 0.0, 0.0], [-math.sin(theta), 0.0, math.cos(theta), 0.0], [0.0, 0.0, 0.0, 1.0]]

def translate(x, y, z):
    return [[1.0, 0.0, 0.0, x], [0.0, 1.0, 0.0, y], [0.0, 0.0, 1.0, z], [0.0, 0.0, 0.0, 1.0]]

def perspective(d):
    return [[1.0, 0.0, 0.0, 0.0], [0.0, 1.0, 0.0, 0.0], [0.0, 0.0, 1.0, 0.0], [0.0, 0.0, 1.0/d, 0.0]]

def homog((x, y, z)):
    return (x, y, z, 1)

def dehomog((x, y, z, w)):
    return (x/w, y/w, z/w)

# SCENES

def blank(frame, t):
    pass

def fade(steps):
    grid = []
    for y in xrange(HEIGHT):
        row = []
        for x in xrange(WIDTH):
            row.append(random.randint(0, steps-1))
        grid.append(row)
    
    def _fade(frame, t):
        step = t
        for y in xrange(HEIGHT):
            for x in xrange(WIDTH):
                if grid[y][x] <= step:
                    frame[x, y] = gfxmode(47) + " " + reset
    
    return _fade

def welcome(frame, t):
    enter_duration = 20
    still_duration = 40
    exit_duration = 20
    
    center_x = WIDTH/2
    center_y = HEIGHT/2
    welcome = "WELCOME TO"
    welcome_still_x = center_x - len(welcome)/2
    welcome_still_y = center_y-1
    title = "S U P E R D E M O  by  Kaikkeus-Sankari"
    title_still_x = center_x - len(title)/2
    title_still_y = center_y  
    
    try:
        time = t
        if time < enter_duration:
            welcome_y = int(round(scale(time, 0, enter_duration, 0, 
           welcome_still_y)))
            welcome_x = welcome_still_x
            title_y = int(round(scale(time, 0, enter_duration, HEIGHT-1, 
            title_still_y)))
            title_x = title_still_x
            raise Exception()
        time -= enter_duration
        if time < still_duration:
            welcome_y = welcome_still_y
            welcome_x = welcome_still_x
            title_y = title_still_y
            title_x = title_still_x
            raise Exception()
        time -= still_duration
        welcome_y = welcome_still_y
        welcome_x = int(round(scale(time, 0, exit_duration, welcome_still_x, -len(welcome))))
        title_y = title_still_y
        title_x = int(round(scale(time, 0, exit_duration, title_still_x, 
        WIDTH-1)))
    except:
        pass
    finally:
        frame[0, 0] = gfxmode(30, 47) + " "
        text(frame, welcome, welcome_x, welcome_y)
        text(frame, title, title_x, title_y)

def pres(frame, t):
    duration0 = 10
    duration1 = 60
    duration2 = 110

    prodline = "a party production"
    codeline = "code    ahihi & Shieni"
    musicline = "music    would have been nice too!"
    
    frame[0, 0] = gfxmode(30, 47) + " "
    if t >= duration0:
        prodline_ = prodline[:t-duration0]
        text(frame, prodline_, WIDTH/2 - len(prodline)/2, HEIGHT/2-2)
    if t >= duration1:
        codeline_ = codeline[:t-duration1]
        text(frame, codeline_, WIDTH/2 - len(codeline)/2, HEIGHT/2)
    if t >= duration2:
        musicline_ = musicline[:t-duration2]
        text(frame, musicline_, WIDTH/2 - len(codeline)/2 - 1, HEIGHT/2+1)

def spectrum(frame, t):
    colors = (41, 43, 42, 46, 44, 45)
    color = colors[t % len(colors)]
    frame[0, 0] = gfxmode(color) + " "

def sines(frame, t):
    random_char = lambda: random.choice("*#%+")
    
    amplitude1 = 0 + 8*math.sin(0.09*t)
    x_scale3 = 0.2 + 0.16*math.sin(0.2*t)
    sine(frame, 0.5, x_scale3, 4, HEIGHT/2, "35", random_char)
    sine(frame, 0.2, 0.1, amplitude1, HEIGHT/2, "31", random_char)
    sine(frame, 0.35, 0.5, 6, HEIGHT/2, "32", random_char)

def waves(frame, t):
    def draw4(x, y, pixel):
        frame[x, y] = pixel()
        frame[x, HEIGHT-1-y] = pixel()
        frame[WIDTH-1-x, y] = pixel()
        frame[WIDTH-1-x, HEIGHT-1-y] = pixel()
    colors = (31, 33, 32, 36, 34, 35)
    sub_i = t / 10
    sub_len = 3
    sub = []
    for i in xrange(sub_i, sub_i+sub_len):
        sub.append(colors[i % len(colors)])
    for y in xrange(HEIGHT/2):
        for x in xrange(WIDTH/2):
            distance = math.sqrt((0.5*x)**2 + y**2)
            i = int(round(distance*0.7 + 0.5*t)) % len(sub)
            draw4(x, y, lambda: gfxmode(1, sub[i]) + random.choice("\\/"))

def cube(frame, t):
    duration1 = 120
    duration2 = 40

    def black(n):
        for y in xrange(n):
            frame[0, y] = reset + " "
            frame[0, frame.rows-1-y] = reset + " "
            for x in xrange(1, frame.cols):
                frame[x, y] = " "
                frame[x, frame.rows-1-y] = " "
        
    if t >= duration1:
        colors = (41, 43, 42, 46, 44, 45)
        for x in xrange(frame.cols):
            for y in xrange(frame.cols):
                i = int(round((y + 2*math.sin(0.13*x - 0.2*t)))) % len(colors)
                color = colors[i]
                frame[x, y] = gfxmode(color) + " "
        t1 = t - duration1
        if t1 < duration2:
            black_lines = int(round(scale(t1, 0, duration2, frame.rows/2, 0)))
            black(black_lines)
            
    # cube
    r = 10.0
    z_offset = -20.0
    d = -10.0
    cube_colors = (37, 30)
    cube_color = cube_colors[t % len(cube_colors)]
    front = [(-r, r, r), (r, r, r), (r, -r, r), (-r, -r, r)]
    back = [(x, y, -r) for (x, y, z) in front]
    corners = [homog(p) for p in front + back]
    rot = matmul(roty(t*0.2), rotx(t*0.06))
    trans = translate(0.0, 0.0, z_offset)
    mat = matmul(trans, rot)
    persp = perspective(d)
    points = []
    for p in corners:
        p1 = dehomog(matmul(mat, p))
        (x, y, z) = dehomog(matmul(persp, homog(p1)))
        sx = int(round(2*x)) + WIDTH/2
        sy = int(round(y)) + HEIGHT/2
        points.append((sx, sy))
    for a, b in ((0, 1), (1, 2), (2, 3), (3, 0), (4, 5), (5, 6), (6, 7), (7, 4), (0, 4), (1, 5), (2, 6), (3, 7)):
        x0, y0 = points[a]
        x1, y1 = points[b]
        line(frame, x0, y0, x1, y1, lambda: reset + gfxmode(1, cube_color) + random.choice("CUBE"))
    
    fill = WIDTH*" "
    message = "greets to:" + fill + "#p.l." + fill + "queer conspiracy" + fill + "jaffa" + fill + "pants^2" + fill + "ISO" + fill + "!dedededemo" + fill + "and all you drunks at Payback 2013!" + fill + "thanks for watching <3" + fill
    message_x = WIDTH-t+duration1+duration2
    if t >= duration1+duration2:
        for x in xrange(frame.cols):
            frame[x, HEIGHT-2] = reset + " "
    text(frame, " " + message + " ", message_x, HEIGHT-2, gfxmode(1, 36, 40))

scenes = [
    (secs(2), blank),
    (20, fade(20)),
    (80, welcome),
    (secs(1), spectrum),
    (180, pres),
    (secs(1), spectrum),
    (secs(8), sines),
    (secs(8), waves),
    (1000, cube),
    (secs(1), spectrum)
]

t = 0
while scenes:
    duration, scene = scenes[0]
    frame = Frame(WIDTH, HEIGHT)
    scene(frame, t)    
    render(frame)
    time.sleep(FRAME_LEN)
    t += 1
    if t >= duration:
        scenes = scenes[1:]
        t = 0

class DeadlineException(Exception): pass
raise DeadlineException()
