#define GL_GLEXT_PROTOTYPES
#include <GL/gl.h>
#include <math.h>
#include <stdlib.h>
#include "fractal.h"
#include "shader.h"
#include "matrix.h"

GLuint vao;
GLuint vbo[4];
GLfloat *vertices, *normals, *texcoords;
GLuint *indices;

GLuint fracshaders[2];
GLuint fracprg;

float *perspective;

unsigned int vc, nc, tc, ic;

/* To block non visible cubes we use this enum p meas positive, n means negative */
enum Block{zero,ppp,pnn,npn,nnp,ppn,pnp,npp,nnn};

void cube(float x, float y, float z, float size, int depth, int startdepth, int enddepth, enum Block block){
   float half, third;
   int i;
   float v[] = { -1.0f, -1.0f, -1.0f, -1.0f, -1.0f, +1.0f, +1.0f, -1.0f, +1.0f, +1.0f, -1.0f, -1.0f,
    -1.0f, +1.0f, -1.0f, -1.0f, +1.0f, +1.0f, +1.0f, +1.0f, +1.0f, +1.0f, +1.0f, -1.0f,
    -1.0f, -1.0f, -1.0f, -1.0f, +1.0f, -1.0f, +1.0f, +1.0f, -1.0f, +1.0f, -1.0f, -1.0f,
    -1.0f, -1.0f, +1.0f, -1.0f, +1.0f, +1.0f, +1.0f, +1.0f, +1.0f, +1.0f, -1.0f, +1.0f,
    -1.0f, -1.0f, -1.0f, -1.0f, -1.0f, +1.0f, -1.0f, +1.0f, +1.0f, -1.0f, +1.0f, -1.0f,
    +1.0f, -1.0f, -1.0f, +1.0f, -1.0f, +1.0f, +1.0f, +1.0f, +1.0f, +1.0f, +1.0f, -1.0f };
    float n[] = { 0.0f, -1.0f, 0.0f, 0.0f, -1.0f, 0.0f, 0.0f, -1.0f, 0.0f, 0.0f, -1.0f, 0.0f,
    0.0f, +1.0f, 0.0f, 0.0f, +1.0f, 0.0f, 0.0f, +1.0f, 0.0f, 0.0f, +1.0f, 0.0f,
    0.0f, 0.0f, -1.0f, 0.0f, 0.0f, -1.0f, 0.0f, 0.0f, -1.0f, 0.0f, 0.0f, -1.0f,
    0.0f, 0.0f, +1.0f, 0.0f, 0.0f, +1.0f, 0.0f, 0.0f, +1.0f, 0.0f, 0.0f, +1.0f,
    -1.0f, 0.0f, 0.0f, -1.0f, 0.0f, 0.0f, -1.0f, 0.0f, 0.0f, -1.0f, 0.0f, 0.0f,
    +1.0f, 0.0f, 0.0f, +1.0f, 0.0f, 0.0f, +1.0f, 0.0f, 0.0f, +1.0f, 0.0f, 0.0f };
    int offset[] = { 0, 2, 1, 0, 3, 2, 4, 5, 6, 4, 6, 7, 8, 9, 10, 8, 10, 11, 12, 15, 14, 12, 14, 13, 16, 17, 18, 16, 18, 19, 20, 23, 22, 20, 22, 21 };

   if(depth == enddepth) return;

   half  = size / 2.0;
   third = size / 3.0;

   if(depth >= startdepth){
      for(i = 0; i < 24; i++){
         vertices[vc++] = x + half * v[i * 3 + 0];  normals[nc++] = n[i * 3 + 0];
         vertices[vc++] = y + half * v[i * 3 + 1];  normals[nc++] = n[i * 3 + 1];
         vertices[vc++] = z + half * v[i * 3 + 2];  normals[nc++] = n[i * 3 + 2];
      }

      for(i = 0; i < 36; i++) indices[ic++] = vc/3 - (24 - offset[i]);
   }

   if(block != ppn) cube(x + half, y + half, z - half, third, depth+1, startdepth, enddepth, nnp);
   if(block != pnn) cube(x + half, y - half, z - half, third, depth+1, startdepth, enddepth, npp);
   if(block != nnn) cube(x - half, y - half, z - half, third, depth+1, startdepth, enddepth, ppp);
   if(block != npn) cube(x - half, y + half, z - half, third, depth+1, startdepth, enddepth, pnp);

   if(block != ppp) cube(x + half, y + half, z + half, third, depth+1, startdepth, enddepth, nnn);
   if(block != pnp) cube(x + half, y - half, z + half, third, depth+1, startdepth, enddepth, npn);
   if(block != nnp) cube(x - half, y - half, z + half, third, depth+1, startdepth, enddepth, ppn);
   if(block != npp) cube(x - half, y + half, z + half, third, depth+1, startdepth, enddepth, pnn);

}

void initFractalScene(float *pm){
   perspective = pm;

   vertices  = malloc(sizeof(GLfloat) * 32904); /*FIXME bigger than we need*/
   if(vertices == NULL) exit(0);
   normals   = malloc(sizeof(GLfloat) * 32904);
   if(normals == NULL) exit(0);
   texcoords = malloc(sizeof(GLfloat) * 16000);
   if(texcoords == NULL) exit(0);
   indices   = malloc(sizeof(GLuint)  * 16452);
   if(indices == NULL) exit(0);

   vc        = 0;
   nc        = 0;
   tc        = 0;
   ic        = 0;

   cube(0.0, 0.0, 0.0, 10.0, 0, 0, 4, zero);

   glGenVertexArrays(1, &vao);
   glBindVertexArray(vao);
   glGenBuffers(4, vbo);

   glBindBuffer(GL_ARRAY_BUFFER, vbo[0]);
   glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * vc, vertices, GL_DYNAMIC_DRAW);
   glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, 0);
   glEnableVertexAttribArray(0);

   glBindBuffer(GL_ARRAY_BUFFER, vbo[1]);
   glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * nc, normals, GL_DYNAMIC_DRAW);
   glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 0, 0);
   glEnableVertexAttribArray(1);

   glBindBuffer(GL_ARRAY_BUFFER, vbo[2]);
   glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * tc, texcoords, GL_DYNAMIC_DRAW);
   glVertexAttribPointer(2, 2, GL_FLOAT, GL_FALSE, 0, 0);
   glEnableVertexAttribArray(2);

   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, vbo[3]);
   glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(GLuint) * ic, indices, GL_STATIC_DRAW);

   glBindVertexArray(0);

   fracshaders[0] = loadShader(GL_VERTEX_SHADER, "shaders/frac.vs");
   fracshaders[1] = loadShader(GL_FRAGMENT_SHADER, "shaders/frac.fs");
   fracprg        = createProgram(2, fracshaders);

   glBindVertexArray(vao);
   bindVarToBuff(fracprg, "vertex", vbo[0], 3);
   bindVarToBuff(fracprg, "normal", vbo[1], 3);
   bindVarToBuff(fracprg, "texcoo", vbo[2], 2);
   glBindVertexArray(0);
}

void drawWall(double time){
   GLint loc;
   float i,j;
   float camera[16];
   float pos[]    = {0.0, 0.0, 7.5};
   float target[] = {0.0, -100.0, 0.0};
   float up[]     = {0.0, 0.0, -1.0};
   float tr[16], rot[16], res[16];

   pos[1] = time * -10.0;
   target[1] = pos[1] - 100.0;
   lookAt(camera, pos, target, up);
   time += 1.4;
   rotate(rot, -1.0 * time, time / 2.0, 0.0);

   glUseProgram(fracprg);
   loc = glGetUniformLocation(fracprg, "pmatrix");
   glUniformMatrix4fv(loc, 1, GL_FALSE, perspective);

   loc = glGetUniformLocation(fracprg, "camera");
   glUniformMatrix4fv(loc, 1, GL_FALSE, camera);

   loc = glGetUniformLocation(fracprg, "model");

   for(i = -5.0; i < 5.0; i++)
      for(j = -5.0; j < 5.0; j++){
      translate(tr, 14.0*i, -1970.0, 14.0*j);
      matrixMultiply4x4(rot, tr, res);

      glUniformMatrix4fv(loc, 1, GL_FALSE, res);

      glBindVertexArray(vao);
      glDrawElements(GL_TRIANGLES, ic, GL_UNSIGNED_INT, 0);
   }
}

void drawTunnel(double time){
   GLint loc;
   float i;
   float camera[16];
   float pos[]    = {0.0, 0.0, 7.5};
   float target[] = {0.0, -100.0, 0.0};
   float up[]     = {0.0, 0.0, -1.0};
   float tr[16];

   pos[1] = pos[1] - (time - 200.0) * 10.0;
   target[1] = pos[1] - 100.0;
   target[0] = sin(time) * 20.0;
   target[2] = cos(time) * 10.0;
   lookAt(camera, pos, target, up);

   glUseProgram(fracprg);
   loc = glGetUniformLocation(fracprg, "pmatrix");
   glUniformMatrix4fv(loc, 1, GL_FALSE, perspective);

   loc = glGetUniformLocation(fracprg, "camera");
   glUniformMatrix4fv(loc, 1, GL_FALSE, camera);

   loc = glGetUniformLocation(fracprg, "model");

   for(i = 0.0; i < 60.0; i++){
         translate(tr, 0.0, i * -14.0, 0.0);
         glUniformMatrix4fv(loc, 1, GL_FALSE, tr);
         glBindVertexArray(vao);
         glDrawElements(GL_TRIANGLES, ic, GL_UNSIGNED_INT, 0);

         translate(tr, 0.0, i * -14.0, 14.0);
         glUniformMatrix4fv(loc, 1, GL_FALSE, tr);
         glBindVertexArray(vao);
         glDrawElements(GL_TRIANGLES, ic, GL_UNSIGNED_INT, 0);
   }
}

void fractalScene(double time){

   if(time > 180.0 && time < 197.0){
      drawWall(time);
   }
   if(time > 197.0) drawTunnel(time);
}

void freeFractalScene(){
   glDeleteBuffers(4, vbo);
   glDeleteVertexArrays(1, &vao);
   free(vertices);
   free(normals);
   free(texcoords);
   free(indices);
}
