function math() {

    //Comparsion precision constant
    this.E = 0.0001;

    this.eq = function(a, b) {
        return Math.abs(a - b) < math.E;
    };

    this.rotate = function(a, b, angle) {
        return {
            x: (a.x - b.x) * Math.cos(angle) - (a.y - b.y) * Math.sin(angle) + b.x,
            y: (a.x - b.x) * Math.sin(angle) + (a.y - b.y) * Math.cos(angle) + b.y
        };
    };

    this.length = function(x, y, x1, y1) {
        var lx = 0;
        var ly = 0;
        switch (arguments.length) {
            case 2:
                lx = x;
                ly = y;
                break;
            case 4:
                lx = x1 - x;
                ly = y1 - y;
                break;
            default:
                return 0;
        }
        ;

        return Math.sqrt(lx * lx + ly * ly);
    };

    this.det = function(x, y, x1, y1) {
        return x * y1 - y * x1;
    };

    /* 
     * Rectangular bounds. Can check if point is in bounds even if rectangle is
     * degenerate (a line).
     */
    this.Bounds = function(seg) {
        this.a = {
            x: Math.min(seg.a.x, seg.b.x),
            y: Math.min(seg.a.y, seg.b.y)
        };

        this.b = {
            x: Math.max(seg.a.x, seg.b.x),
            y: Math.max(seg.a.y, seg.b.y)
        };

        this.add = function(bounds) {
            this.a = {
                x: Math.min(bounds.a.x, this.a.x),
                y: Math.min(bounds.a.y, this.a.y)
            };

            this.b = {
                x: Math.max(bounds.b.x, this.b.x),
                y: Math.max(bounds.b.y, this.b.y)
            };
        };

        this.contains = function(p) {
            if (math.eq(this.a.y, this.b.y)) {
                return p.x >= this.a.x && p.x <= this.b.x && math.eq(p.y, this.a.y);
            }
            if (math.eq(this.a.x, this.b.x)) {
                return p.y >= this.a.y && p.y <= this.b.y && math.eq(p.x, this.a.x);
            }
            return p.x >= this.a.x && p.y >= this.a.y &&
                    p.x <= this.b.x && p.y <= this.b.y;
        };

        this.getRadius = function() {
            return math.length(this.a.x, this.a.y, this.b.x, this.b.y);
        };

        this.toString = function() {
            return "[(" + this.a.x + ";" + this.a.y + "), (" + this.b.x + ";" + this.b.y + ")]";
        };
    };

    this.Polygon = function(points) {
        this.points = points;

        this.getBounds = function() {
            var points = this.points;

            if (this._bounds !== undefined) {
                return this._bounds;
            }
            var min = {x: points[0].x, y: points[0].y};
            var max = {x: points[0].x, y: points[0].y};

            for (var i = 1; i < points.length; i++) {
                var p = points[i];

                if (p.x < min.x) {
                    min.x = p.x;
                }
                if (p.y < min.y) {
                    min.y = p.y;
                }
                if (p.x > max.x) {
                    max.x = p.x;
                }
                if (p.y > max.y) {
                    max.y = p.y;
                }
            }
            this._bounds = new math.Bounds({
                a: min, b: max
            });

            return this._bounds;
        };

        this.rotate = function(point, angle) {
            var points = this.points;
            for (var i = 0; i < points.length; i++) {
                points[i] = points[i].rotate(point, angle);
            }
            this._bounds = undefined;
        };

        this.move = function(x, y) {
            var dx;
            var dy;
            if (arguments.length === 1) {
                dx = x.x;
                dy = x.y;
            } else if (arguments.length === 2) {
                dx = x;
                dy = y;
            }
            var points = this.points;
            for (var i = 0; i < points.length; i++) {
                points[i].x += dx;
                points[i].y += dy;
            }
        };

        this.scale = function(center, delta) {
            var points = this.points;
            for (var i = 0; i < points.length; i++) {
                var scaleVector = new math.Vector(points[i].x - center.x, points[i].y - center.y).norm().scale(delta, delta);
                points[i] = points[i].add(scaleVector);
            }
            this._bounds = undefined;
        };

        this.toString = function() {
            return "Polygon {points:" + this.points.length + ", bounds: " + this.getBounds().getRadius() + "}";
        };
    };

    this.regularPolygon = function(center, radius, n) {
        var alpha = 0;
        var delta = Math.PI * 2 / n;
        var points = new Array();

        for (var i = 0; i < n; i++) {
            points.push(new math.Vector(center.x + radius * Math.cos(alpha),
                    center.y + radius * Math.sin(alpha)));
            alpha += delta;
        }
        points.push(new math.Vector(center.x + radius * Math.cos(0),
                center.y + radius * Math.sin(0)));
        var p = new math.Polygon(points);
        p.radius = radius;
        return p;

    };

    /*
     * Vector in the geometric sence.
     * Addition, scaling, scalar multiplication, projection and reflection
     * by normal vector provided
     */
    this.Vector = function(x, y) {

        this.x = x;
        this.y = y;

        this.length = function() {
            if (!this.hasOwnProperty("_length")) {
                this._length = math.length(this.x, this.y);
            }

            return this._length;
        };

        this.norm = function() {
            if (!this.hasOwnProperty("_normal")) {
                if (this.length() === 0) {
                    return new math.Vector(0, 0);
                }
                ;

                this._normal = new math.Vector(this.x / this.length(),
                        this.y / this.length());
            }
            ;

            return this._normal;
        };

        this.add = function(x, y) {
            if (arguments.length === 1) {
                return new math.Vector(this.x + x.x, this.y + x.y);
            }
            ;
            return new math.Vector(this.x + x, this.y + y);
        };

        this.mul = function(v) {
            return this.x * v.x + this.y * v.y;
        };

        this.scale = function(kx, ky) {
            if (arguments.length === 1) {
                return new math.Vector(this.x * kx, this.y * kx);
            }
            ;
            return new math.Vector(this.x * kx, this.y * ky);
        };

        this.project = function(v) {
            return v.scale(this.mul(v));
        };

        this.reflect = function(n) {
            return this.add(this.project(n).scale(-2));
        };

        this.collinear = function(v) {
            return math.det(this.x, this.y, v.x, v.y);
        };

        this.toString = function() {
            return "(" + x + "; " + y + ")";
        };

        this.rotate = function(point, angle) {
            var x = math.rotate(this, point, angle);
            return new math.Vector(x.x, x.y);
        };
    };

    /*
     * A line. Defined by two points, stored in a canonical way (Ax + By + C = 0)
     * 
     * Supported operations: 
     * Directing vector, normal vector, distance to point,
     * line-line intersection check.
     */
    this.Line = function(x, y, x1, y1) {

        if (arguments.length === 4) {
            this.A = (y - y1);
            this.B = (x1 - x);
            this.C = (x * y1 - x1 * y);
        } else if (arguments.length === 3) {
            this.A = y - x1.y;
            this.B = x1.x - x;
            this.C = (x * x1.y - x1.x * y);
        }
        ;

        this.vector = function() {
            return new math.Vector(-this.B, this.A).norm();
        };

        this.norm = function() {
            return new math.Vector(this.A, this.B).norm();
        };

        this.distanceTo = function(p) {
            return Math.abs(p.x * this.A + p.y * this.B + this.C) /
                    Math.sqrt(this.A * this.A + this.B * this.B);
        };

        this.intersection = function(line) {
            if (Math.abs(math.det(this.A, this.B, line.A, line.B)) < math.E) {
                return undefined;
            }
            ;

            return {
                x: -math.det(this.C, this.B, line.C, line.B) /
                        math.det(this.A, this.B, line.A, line.B),
                y: -math.det(this.A, this.C, line.A, line.C) /
                        math.det(this.A, this.B, line.A, line.B)
            };
        };

        this.toString = function() {
            return "[" + this.A + ", " + this.B + ", " + this.C + "]";
        };
    };

    /*
     * A segment. Defined by two points, stored as a line with that points as 
     * borders. 
     * 
     * Supported operations:
     * length, middle point, segment-segment intersection check, random point 
     * picker.
     */
    this.Segment = function(x, y, x1, y1) {

        this.line = new math.Line(x, y, x1, y1);
        this.a = {
            x: x,
            y: y
        };
        this.b = {
            x: x1,
            y: y1
        };
        this.middle = function() {
            return {
                x: (this.a.x + this.b.x) / 2,
                y: (this.a.y + this.b.y) / 2
            };
        };

        this.norm = function() {
            return this.line.norm();
        };

        this.length = function() {
            return math.length(this.a.x, this.a.y, this.b.x, this.b.y);
        };

        this.intersects = function(seg) {

            var ip = this.line.intersection(seg.line);

            if (ip === undefined) {
                return false;
            }
            ;


            var segBounds = new math.Bounds(seg);
            var bounds = new math.Bounds(this);

            //alert (segBounds + " and " + bounds + " cc " + ip.x +":" + ip.y + ": ina = " + segBounds.contains(ip)  + ", inb=" + bounds.contains(ip));
            return (segBounds.contains(ip) && bounds.contains(ip)) ? ip : undefined;
        };

        this.move = function(delta) {
            this.a.x += delta.x;
            this.a.y += delta.y;
            this.b.x += delta.x;
            this.b.y += delta.y;

            this.line = new math.Line(this.a.x, this.a.y, this.b.x, this.b.y);
        };

        this.rotateBy = function(angle) {

            this.a = math.rotate(this.a, this.middle(), angle);
            this.b = math.rotate(this.b, this.middle(), angle);

            this.line = new math.Line(this.a.x, this.a.y, this.b.x, this.b.y);
        };

        this.rotateTo = function(pt) {
            var middle = this.middle();
            var length = this.length();
            var v = new math.Vector(pt.x - middle.x, pt.y - middle.y).norm();

            this.a = {
                x: middle.x + v.x * length / 2,
                y: middle.y + v.y * length / 2
            };

            this.b = {
                x: middle.x - v.x * length / 2,
                y: middle.y - v.y * length / 2
            };

            this.line = new math.Line(this.a.x, this.a.y, this.b.x, this.b.y);
        };

        this.randomPoint = function() {
            var px;
            var py;

            if (math.eq(this.a.x, this.b.x)) {
                px = this.a.x;
                py = Math.min(this.a.y, this.b.y) + Math.random() * Math.abs(this.a.y - this.b.y);
            } else if (math.eq(this.a.y, this.b.y)) {
                py = this.a.y;
                px = Math.min(this.a.x, this.b.x) + Math.random() * Math.abs(this.a.x - this.b.x);
            } else {
                px = Math.min(this.a.x, this.b.x) + Math.random() * Math.abs(this.a.x - this.b.x);
                py = -this.line.C / this.line.B - this.line.A / this.line.B * px;
            }
            ;

            return {
                x: px,
                y: py
            };
        };

        this.distanceTo = function(p) {
            var v = new math.Vector(this.b.x - this.a.x, this.b.y - this.a.y),
                    w = new math.Vector(p.x - this.a.x, p.y - this.a.y),
                    c1 = v.mul(w),
                    c2 = v.mul(v);

            if (c1 < 0) {
                return math.length(p.x, p.y, this.a.x, this.a.y);
            }
            ;

            if (c2 < c1) {
                return math.length(p.x, p.y, this.b.x, this.b.y);
            }
            ;

            var pb = v.scale(c1 / c2).add(this.a);

            return math.length(p.x, p.y, pb.x, pb.y);

        };

        this.getBounds = function() {
            return new math.Bounds(this);
        };

        this.toString = function() {
            return "[(" + this.a.x + ";" + this.a.y + "), (" + this.b.x + ";" + this.b.y + ")]";
        };
    };

    /*
     * Geometrical circle. 
     * 
     * Supported operations: normal and circle-segment intersection.
     */
    this.Circle = function(x, y, r) {

        this.x = x;
        this.y = y;
        this.r = r;

        this.intersects = function(seg) {
            return (math.length(this.x, this.y, seg.a.x, seg.a.y) >= this.r &&
                    math.length(this.x, this.y, seg.b.x, seg.b.y) <= this.r) ||
                    (math.length(this.x, this.y, seg.a.x, seg.a.y) <= this.r &&
                            math.length(this.x, this.y, seg.b.x, seg.b.y) >= this.r);
        };

        this.norm = function(p) {
            return new math.Vector(p.x - this.x, p.y - this.y).norm();
        };

        this.distanceTo = function(p) {
            return Math.abs(math.length(p.x, p.y, this.x, this.y) - this.r);
        };

        this.move = function(delta) {
            this.x += delta.x;
            this.y += delta.y;
        };

        this.contains = function(x, y) {
            return math.length(this.x, this.y, x, y) < this.r;
        };

        this.getBounds = function() {
            return new math.Bounds({
                a: {x: this.x - this.r, y: this.y - this.r},
                b: {x: this.x + this.r, y: this.y + this.r}
            });
        };
    };

    /*
     * A moving point. Defined by coordinates and vector of speed.
     */
    this.DPoint = function(x, y, v) {
        this.vector = v;
        this.x = x;
        this.y = y;

        this.seg = function(kl) {
            var k = arguments.length === 1 ? kl : 1;
            return new math.Segment(this.x, this.y, this.x + k * this.vector.x, this.y + k * this.vector.y);
        };

        this.toString = function() {
            return "(" + this.x + "; " + this.y + ")";
        };

        this.proceed = function() {
            this.x += this.vector.x;
            this.y += this.vector.y;
        };
    };
}
;

(function() {
    math = new math();

    //Simple check after initialization.
    var s = new math.Segment(0, 10, 10, 0);
    var s1 = new math.Segment(0, 0, 10, 10);
    var p = {
        x: 0,
        y: 0
    };
    var l = math.length(0, 0, s.middle().x, s.middle().y);

    if (!s.intersects(s1) || !math.eq(s.distanceTo(p), l) || !math.eq(s.distanceTo(p), s.line.distanceTo(p))) {
        alert("Test failed");
    }
    ;
})();