#define _WIN32_WINNT 0x400
#include <windows.h>
#include <d3d9.h>
#include <d3dx9.h>

#include <string>
#include <vector>
#include <sstream>
#include <list>

#include <time.h>
#include <algorithm>
#include <fstream>
#include <io.h>
#include <map>

#include "D3DApp.h"
#include "common_globals.h"

using std::list;
using std::string;
using std::vector;
using std::stringstream;

bool g_bAdvanceAndRenderEffects = true;

#include "ScrollArea.h"

#include "EffectLayout.h"
#include "effect.h"
#include "timeline.h"
#include "datafile.h"

#include "effectMeshView.h"
#include "effectKuva.h"
#include "EffectConv16bToBB.h"
#include "EffectDeferredMixer.h"
#include "EffectDeferredLight.h"
#include "EffectDeferredGPU.h"

#include "EffectCamera.h"
#include "EffectLight.h"
#include "EffectBlit.h"
#include "EffectFFT.h"


#include "EffectShade.h"

#include "EffectDof.h"
#include "EffectGodRay.h"

#include "EffectRobo.h"


#include "EffectGlow.h"

#include "EffectPixSim.h"
#include "EffectPixParticles.h"


#include "EffectGameProto.h"
#include "EffectGameCubicle.h"

#include "EffectOfs.h"


#include "MouseSensor.h"


#include "ContextData.h"
#include "DemoContextData.h"

#include "Shader.h"
#include "Deferred.h"

#include "fmod.h"
#include "music.h"


#include "../../demowork/sync/sync.h"


class TimelineInfoModeMouseSensor : public MouseSensor {
private:
  Timeline::ShowInfoMode::Enum m_showInfoMode;
  Timeline *m_pTimeline;
public:
  TimelineInfoModeMouseSensor(int al, int ar, int at, int ab, Timeline::ShowInfoMode::Enum showInfoMode, Timeline *pTimeline) : MouseSensor(al, ar, at, ab) {
    m_showInfoMode = showInfoMode;
    m_pTimeline = pTimeline;
    m_sensorType = 0;
  }
  bool OnAreaAction(int mouseX, int mouseY, bool lDown, bool lClick, bool lDblClick, bool rDown, bool rClick, bool rDlbClick);
};

bool TimelineInfoModeMouseSensor::OnAreaAction(int mouseX, int mouseY, bool lDown, bool lClick, bool lDblClick, bool rDown, bool rClick, bool rDlbClick) {
  if (lClick) {
    m_pTimeline->SetShowInfoMode(m_showInfoMode);
  }
  return true;
}



class TimelineChangeShowRTMouseSensor : public MouseSensor {
private:
  int m_iRtIndex;
  Timeline *m_pTimeline;

public:
  TimelineChangeShowRTMouseSensor(int al, int ar, int at, int ab, int rtIndex, Timeline *pTimeline) : MouseSensor(al, ar, at, ab) {
    m_iRtIndex = rtIndex;
    m_pTimeline = pTimeline;
    m_sensorType = 0;
  }
  bool OnAreaAction(int mouseX, int mouseY, bool lDown, bool lClick, bool lDblClick, bool rDown, bool rClick, bool rDlbClick);
};
bool TimelineChangeShowRTMouseSensor::OnAreaAction(int mouseX, int mouseY, bool lDown, bool lClick, bool lDblClick, bool rDown, bool rClick, bool rDlbClick) {
  if (lClick) {
    m_pTimeline->SetShowRtIndexWithToggleToMinus1(m_iRtIndex);
  }
  return true;
}


double PerfGetTickCount() {   // Return milliseconds
  
  static LARGE_INTEGER freq;
  static bool init = false;
  static bool perfcount = false;
  static double timeScale = 1.0;
  static __int64 timerStart = 0;

  _control87( _CW_DEFAULT, 0xfffff );
  
  if (!init) {
    init = true;
    perfcount = !!QueryPerformanceFrequency(&freq);
    if (perfcount) {
      // Factor in the conversion of freq to milliseconds
      timeScale *= 1000.0 / freq.QuadPart;
      LARGE_INTEGER count, count2;
      QueryPerformanceCounter(&count);
      timerStart = count.QuadPart;    

      // Check that QueryPerformanceCounter is monotonic
      // It is not on AMD64 X2 system
      DWORD test_start = GetTickCount();
      DWORD r;
      while((r = GetTickCount()) < test_start+250 && perfcount) {
        for (int i=0; i<1000; i++) {
          QueryPerformanceCounter(&count2);
          if (count2.QuadPart < count.QuadPart) {
            perfcount = false;
            timerStart = timeGetTime();    
            timeScale *= freq.QuadPart / 1000.0;
            break;
          }
          count.QuadPart = count2.QuadPart;
        }
      }
    } else {
      timerStart = timeGetTime();    
    }
  }
  if (!perfcount) {
    // Use timeGetTime
    return ((timeGetTime() - timerStart) * timeScale);
  } else {
    // User higher-resolution timing... 
    LARGE_INTEGER count;
    QueryPerformanceCounter(&count);
    return ((count.QuadPart - timerStart) * timeScale);
  }
}



Timeline::Timeline() {
  m_startTime = 0.0f;
  m_endTime = 0.0f;
  m_currentTime = 0.0f;
  m_loadedTimelineFileName = "";
  m_bFlushDeviceBetweenEffects = true; // enabled only when debug window mode is active
  m_bShowInactiveEffects = true;

  m_totalTime = 1.0f;

  m_showInfoMode = ShowInfoMode::EFFECTS;

  m_pScrollAreaFxList = new ScrollArea();
  m_pScrollAreaRtList = new ScrollArea();

  m_showRtIndex = -1;
}

Timeline::~Timeline() {
  // TBD: delete the effects
}

Effect *g_currentEffect = NULL;

Effect *GetCurrentEffect() {
  return g_currentEffect;
}


Effect* Timeline::GetNewDerivedEffect(string effectType) {
  
  // timeline to c++ code mapping

  UpdateLoadingFrame();


  if (effectType == "mesh") {
    return new EffectMeshView();
  } else if (effectType == "game_proto") {
    return new EffectGameProto();
  } else if (effectType == "game_cubicle") {
    return new EffectGameCubicle();
  } else if (effectType == "kuva") {
    return new EffectKuva();
  } else if (effectType == "hdr") {
    return new EffectConv16bToBB();
  } else if (effectType == "dfmix") {
    return new EffectDeferredMixer();
  } else if (effectType == "dfgpu") { // deffered light boxes from gpu position data
    return new EffectDeferredGPU();
  } else if (effectType == "dflight") {
    return new EffectDeferredLight();
  } else if (effectType == "camera") {
    return new EffectCamera();
  } else if (effectType == "offset") {
    return new EffectOffset();
  }  else if (effectType == "light") {
    return new EffectLight();
  } else if (effectType == "blit") {
    return new EffectBlit();
  } else if (effectType == "fft") {
    return new EffectFFT();
  } else if (effectType == "shade") {
    return new EffectShade();
  } else if (effectType == "dof") {
    return new EffectDof();
  } else if (effectType == "robo") {
    return new EffectRobo();
  } else if (effectType == "godray") {
    return new EffectGodRay();
  } else if (effectType == "glow") {
    return new EffectGlow();
  } else if (effectType == "pixpart") {
    return new EffectPixParticles();
  } else if (effectType == "pixsim") {
    return new EffectPixSim();
  } 

  char errMsg[4096];
  sprintf(errMsg, "Error:\nUnknown effect name \"%s\" in timeline.", effectType.c_str());
  MessageBox(NULL, errMsg, "Error in timeline", MB_ICONERROR);
  exit(-1);

  return NULL;
}


bool Timeline::AddEffect(string fileName, string effectType, float start, float end, vector<string> paramsIn, int clearValue, std::vector<RtBuffer> rt16bBufferNames, bool bDoubleBuffer) {

  Effect *effect = NULL;


  string layoutFile = fileName;
  EffectLayout* pEffectLayout = new EffectLayout();
  if (!pEffectLayout->readFromFile("", fileName, NULL)) {
    char errMsg[4096];
    sprintf(errMsg, "Error:\nEffect xml file \"%s\" not found or error in parsing.", fileName.c_str());
    MessageBox(NULL, errMsg, "Error in timeline", MB_ICONERROR);
    exit(-1);

    return false;
  }

  effectType = pEffectLayout->GetEffectType();
  std::string effectName = pEffectLayout->m_effectName;

  effect = GetNewDerivedEffect(effectType);

  if (effect == NULL)
    return false;

  effect->SetEffectLayout(pEffectLayout, fileName);
  pEffectLayout->SetBelongsToEffect(effect);
  effect->SetEnableSyncVariable(effectName+".ON");

  // TBD: add support to parse from multiple files

  effect->SetStartTime(start);
  effect->SetEndTime(end);
  effect->SetParams(paramsIn);
/*
  if (!effect->ParseFromFile(fileName)) {
  }
  */

  effect->SetFileAndType(fileName, effectType);

  effect->SetRT16bBufferNames(rt16bBufferNames);

//  effect->SetRT32bBufferName(rt32bBufferName);

  effect->SetDoubleBuffer(bDoubleBuffer);

  DWORD clearFlags = 0;
  if (clearValue&1)
    clearFlags+=D3DCLEAR_TARGET;
  if (clearValue&2)
    clearFlags+=D3DCLEAR_ZBUFFER;
  effect->SetClearFlags(clearFlags);
  m_effects.push_back(effect);
  effect->Init();
  return true;
}


void readEqualsStr(const char *str, const char *needle, char *result) {
  const char *posseNow;
  posseNow = strstr(str, needle);
  result[0] = 0;
  if (posseNow) {
    int ii=0;
    while (posseNow[ii] != '=') {
      ii++;
    }
    ii++;

    while ((posseNow[ii] == ' ') || (posseNow[ii] == '\n')) {
      ii++;
    }
    int iii=0;
    int posseEnd = 0;
    int inBraces = 0;
    while (!posseEnd && (posseNow[ii] != '\n') && (posseNow[ii] != 13) && (posseNow[ii] != ']') && (posseNow[ii] != 0)) {
      if (posseNow[ii] != '"') {
      //  if (inBraces || ((!inBraces) && (posseNow[ii] != ' '))) {
        if (posseNow[ii] != ' ') {
          result[iii] = posseNow[ii];
          iii++;
        }
      } else {
        if (inBraces) {
          break;
        } else {
          inBraces=1;
        }
        // inBraces = !inBraces;
      }
      if (!inBraces) {
        if (posseNow[ii] == ' ') {
          break;
        }
      }
      ii++;
    }
    result[iii] = 0;
  }
}


// reload the timeline
void Timeline::Reload(string timelineFile) {
  if (m_effects.size()) {
    for (vector<Effect*>::iterator it=m_effects.begin(); it!=m_effects.end(); it++) {
      delete *it;
    }
  }
  Init(timelineFile);
}


bool Timeline::Init(string timelineFile) {
// read config file
// etc.
// init the timeline & effects
  bool success = false;

  unsigned char *data=NULL;
  unsigned int dataSize=0;
  
  if (strcmp(m_loadedTimelineFileName.c_str(), timelineFile.c_str())!=0) {
    InsertLoadedFileTimeline(timelineFile);
  }

  if (strcmp(timelineFile.c_str(), "")==0) {
    timelineFile = m_loadedTimelineFileName;
  }

  if (getData(timelineFile.c_str(),  &data, &dataSize)) {

    m_loadedTimelineFileName = timelineFile;

    char str[2048];
    char s[2048];

    unsigned int dataIndex = 0;
    int i=0;
    char c;
    int underGlobalComment=0;
    std::vector<string> paramsIn;
    do {
      c = data[dataIndex];
      dataIndex++;
      if (dataIndex < dataSize) {
        if ((c != '\n') && (c != 13)) {
          str[i] = c;
          i++;
        } else {
          str[i] = 0;
          i=0;

          // remove comments and copy string
          {
            int underComment = 0;
            int sI=0;
            unsigned int i=0;
            while (str[i] != 0) {
              if ((str[i] != '\n') && (str[i] != 13)) {
                if ((str[i] == '/') && (str[i+1] == '/')) {
                  underComment = 1;
                }
                if ((str[i] == '/') && (str[i+1] == '*')) {
                  underGlobalComment = 1;
                }
                if ((str[i] == '*') && (str[i+1] == '/')) {
                  underGlobalComment = 0;
                }
              } else {
                underComment = 0;
              }
              if ((underComment+underGlobalComment)==0) {
                if ((str[i] != '\n') && (str[i] != 13)) {
                  s[sI] = str[i];
                  sI++;
                }
              }
              i++;
            }
            s[sI] = 0;
          }

          char strNow[512];

          readEqualsStr(s, "fx", strNow);
          if (strNow[0]) {
            char fileName[256];
            char effectName[256];
            sprintf(fileName, "%s", strNow);

            readEqualsStr(s, "effect", strNow);
            if (strNow[0]) {
              sprintf(effectName, "%s", strNow);
            }

            float start=0.0f;
            float end=10000.0f;
            float length=10000.0f;
            int clearValue=0;

            paramsIn.clear();

            readEqualsStr(s, "start", strNow);
            if (strNow[0]) {
                stringstream ss;
                ss << strNow;
                ss >> start;
            }

            readEqualsStr(s, "end", strNow);
            if (strNow[0]) {
                stringstream ss;
                ss << strNow;
                ss >> end;
            }

            readEqualsStr(s, "length", strNow);
            if (strNow[0]) {
                stringstream ss;
                ss << strNow;
                ss >> length;
            }

            readEqualsStr(s, "clear", strNow);
            if (strNow[0]) {
                stringstream ss;
                ss << strNow;
                ss >> clearValue;
            }

            /*
            readEqualsStr(s, "in", strNow);
            
            if (strNow[0]) {
              char delims[] = ",";
              char *result = NULL;
              result = strtok( strNow, delims );
              while( result != NULL ) {
              }
            }
            */
            
            readEqualsStr(s, "params", strNow);
            
            if (strNow[0]) {
              char delims[] = ";";
              char *result = NULL;
              result = strtok( strNow, delims );
              while( result != NULL ) {
              //  float floatti;
              //  std::stringstream ss;
              //  ss << result;
              //  ss >> floatti;
              //  if (!ss.fail()) {
              //    paramsIn.push_back(floatti);
              //  }
                paramsIn.push_back(std::string("$")+std::string(result));
                result = strtok( NULL, delims );
              }
            }

            readEqualsStr(s, "out", strNow);
            
            if (strNow[0]) {
            }
            std::vector<RtBuffer> rt16bBufferNames;

            readEqualsStr(s, "rt16b", strNow);
            if (strNow[0]) {
                string s;
                stringstream ss;
                ss << strNow;
                ss >> s;
                RtBuffer rtb;
                rtb.m_name = s;
                rtb.m_fmt = D3DFMT_A16B16G16R16F;
                rt16bBufferNames.push_back(rtb);
            }

            readEqualsStr(s, "rt32b", strNow);
            if (strNow[0]) {
                string s;
                stringstream ss;
                ss << strNow;
                ss >> s;
                RtBuffer rtb;
                rtb.m_name = s;
                rtb.m_fmt = D3DFMT_A32B32G32R32F;
                rt16bBufferNames.push_back(rtb);
            }


            for (int i=1; i<8; i++) {
              char rt16bname[512];
              sprintf(rt16bname, "rt16b%d", i);
              readEqualsStr(s, rt16bname, strNow);
              if (strNow[0]) {
                  string s;
                  stringstream ss;
                  ss << strNow;
                  ss >> s;
                  RtBuffer rtb;
                  rtb.m_name = s;
                  rtb.m_fmt = D3DFMT_A16B16G16R16F;
                  rt16bBufferNames.push_back(rtb);
              }
            }

            for (int i=1; i<8; i++) {
              char rt16bname[512];
              sprintf(rt16bname, "rt32b%d", i);
              readEqualsStr(s, rt16bname, strNow);
              if (strNow[0]) {
                  string s;
                  stringstream ss;
                  ss << strNow;
                  ss >> s;
                  RtBuffer rtb;
                  rtb.m_name = s;
                  rtb.m_fmt = D3DFMT_A32B32G32R32F;
                  rt16bBufferNames.push_back(rtb);
              }
            }

            bool bDoubleBuffer = false;

            readEqualsStr(s, "dbuf", strNow);
            if (strNow[0]) {
              if (strcmp(strNow, "true")==0) {
                bDoubleBuffer = true;
              }
            }

            if (end < -9999.0f && length < -9999.0f) {
              char errMsg[512];
              sprintf(errMsg, "Error:\nYou have to specify either end or length in timeline, missing for effect file \"%s\"", fileName);
              MessageBox(NULL, errMsg, "Timeline error", MB_ICONERROR);
              exit(0);
            }

            if (length > -9999.0f) {
              end = start + length;
            }

            AddEffect(fileName, effectName, start, end, paramsIn, clearValue, rt16bBufferNames, bDoubleBuffer);

            if ((end+1.0f) > m_endTime) {
              m_endTime = end+1.0f;
            }
          }
        }
      }
    } while(dataIndex<dataSize);
    success = true;
  }
  if (data != NULL) {
    delete [] data;
  }
  return success;
}

void Timeline::SetCurrentTime(float timeAbs, float timeStep) {
  m_currentTime = timeAbs;
  m_currentTimeStep = timeStep;

  for (vector<Effect*>::iterator it=m_effects.begin(); it!=m_effects.end(); it++) {
    (*it)->SetTimeValues(timeAbs, timeStep);
  }
}

bool g_reloadEffects = false;
bool g_reloadShaders = false;

bool TimelineMouseSensor::OnAreaActionTMS(int mouseX, int mouseY, bool lDown, bool lClick, bool lDblClick, bool rDown, bool rClick, bool rDlbClick) {
  if (lClick) {
    /*
    if (m_pEffect->IsEnabled()) {
      m_pEffect->SetDisabled();
    } else {
      m_pEffect->SetEnabled();
    }
    */
    if (m_pEffect && GetRocket()) {
      if (m_actionType==0) {
      set_track_on_focus(GetRocket(), (m_pEffect->GetName()).c_str(), 0);
      } else {
        char cmd[1024];
        sprintf(cmd, "start notepad++ %s", m_pEffect->GetFilename().c_str());
        int k = system(cmd);
      }
    }
  }
  return true;
}


bool BooleanSwitchMouseSensor::OnAreaActionBSMS(int mouseX, int mouseY, bool lDown, bool lClick, bool lDblClick, bool rDown, bool rClick, bool rDlbClick) {
  if (lClick) {
    if (*m_pBool) {
      *m_pBool = false;
    } else {
      *m_pBool = true;
    }
  }
  return true;
}


bool MoverMouseSensor::OnAreaActionMMS(int mouseX, int mouseY, bool lDown, bool lClick, bool lDblClick, bool rDown, bool rClick, bool rDlbClick) {

  ScrollArea *pScrollArea = (ScrollArea*)m_pContext;

  int mouseOrigX = mouseX;
  int mouseOrigY = mouseY;

  POINT cursorPos;
  GetCursorPos(&cursorPos);
  mouseX = cursorPos.x;
  mouseY = cursorPos.y;

  if (lClick) {
    if (this->IsHovering(mouseOrigX, mouseOrigY)) {
      pScrollArea->m_mouseDownAtX = mouseX;
      pScrollArea->m_moveXAtMouseDown = *m_pMoveX;
      pScrollArea->m_mouseDownAtY = mouseY;
      pScrollArea->m_moveYAtMouseDown = *m_pMoveY;
      pScrollArea->m_bScrollingWithBar = true;
    }
    g_bAdvanceAndRenderEffects = false;
  }
  if (lDown && pScrollArea->m_bScrollingWithBar) {
    *m_pMoveX = pScrollArea->m_moveXAtMouseDown + (float)(mouseX-pScrollArea->m_mouseDownAtX)*m_scaleX;
    *m_pMoveY = pScrollArea->m_moveYAtMouseDown + (float)(mouseY-pScrollArea->m_mouseDownAtY)*m_scaleY;
    if (*m_pMoveX < m_minX) {
      *m_pMoveX = m_minX;
    }
    if (*m_pMoveX > m_maxX) {
      *m_pMoveX = m_maxX;
    }
    if (*m_pMoveY < m_minY) {
      *m_pMoveY = m_minY;
    }
    if (*m_pMoveY > m_maxY) {
      *m_pMoveY = m_maxY;
    }
  }
  if (!lDown && pScrollArea->m_bScrollingWithBar) {
    pScrollArea->m_bScrollingWithBar = false;
  }
  if (!lDown && !lClick) {
    g_bAdvanceAndRenderEffects = true;
  }
  return true;
}




vector<TimelineMouseSensor> timelineMouseSensors;
vector<TimelineInfoModeMouseSensor> timelineInfoModeMouseSensors;
vector<TimelineChangeShowRTMouseSensor> timelineChangeShowRTMouseSensors;
vector<BooleanSwitchMouseSensor> booleanSwitchMouseSensors;
vector<MoverMouseSensor> moverMouseSensors;


bool Timeline::HandleInputs() {
  bool inputAvailable = true;
  for (vector<Effect*>::iterator it=m_effects.begin(); it!=m_effects.end(); it++) {
    if ((*it)->IsActive()) {
      if (inputAvailable) {
        inputAvailable = (*it)->HandleInput();
      }
    }
  }
  return inputAvailable;
}



void Timeline::AdvanceEffects() {

  ClearMouseSensors();
  timelineMouseSensors.clear();
  timelineInfoModeMouseSensors.clear();
  timelineChangeShowRTMouseSensors.clear();
  booleanSwitchMouseSensors.clear();
  moverMouseSensors.clear();

  if (!g_bAdvanceAndRenderEffects) {
    return;
  }

  ResetGlobalOfs();
  ResetGlobalCamera();
  ResetGlobalLights();

  g_numMeshesDrawn=0;
  g_numFacesDrawn=0;
  g_numVerticesDrawn=0;

  g_numMeshes=0;
  g_numFaces=0;
  g_numVertices=0;


  if (g_reloadEffects) {
    g_D3DApp->MarkAllMeshMaterialsToBeReloaded();
  //  Reload("");
  }

  bool bCheckFileTimes = false;
  static DWORD timeLastCheckForFileTimes = 0;
  if ((timeGetTime()-timeLastCheckForFileTimes)>200) {
    timeLastCheckForFileTimes = timeGetTime();
    bCheckFileTimes = true;
  }

  for (vector<Effect*>::iterator it=m_effects.begin(); it!=m_effects.end(); it++) {
    if ((*it)->IsActive()) {
      if (bCheckFileTimes) {
        if ((*it)->CheckShadersModifiedOnDisk()) {
          (*it)->ReloadShaders();
        }
     //   if ((*it)->CheckLayoutModifiedOnDisk()) {
     //     (*it)->ReloadLayout();
    //    }
      }
      if (g_reloadEffects) {
        (*it)->ReloadLayout();
        (*it)->ReloadEffect();
      }
      if (g_reloadShaders) {
        (*it)->ReloadShaders();
      }
      g_currentEffect = (*it);
      if ((*it)->IsEnabled()) {
        (*it)->SetAdvanceStartTime(PerfGetTickCount());
        (*it)->Advance();
        (*it)->SetAdvanceEndTime(PerfGetTickCount());
      }
    }
  }

  if (bCheckFileTimes) {
    if (GetDeferred()->CheckShadersModifiedOnDisk()) {
      GetDeferred()->ReloadShaders();
    }
  }


  if (g_reloadShaders) {
    GetDeferred()->ReloadShaders();
  }

  g_reloadEffects = false;
  g_reloadShaders = false;
}

bool bDoubleBufferRt16bs[16];
Texture *pCurrentRt16bs[16];
Texture *pCurrentPrevRt16bs[16];

void SetRt16b(std::string rtBufferName, bool bDoubleBuf, D3DFORMAT fmt, int rtIndex) {
  Texture *rt16b_buffer = NULL;
  Texture *rt16b_buffer_last = NULL;

  if (bDoubleBuf) {
    std::string bufNameLast = rtBufferName+"_prev";
    rt16b_buffer = g_D3DApp->createRenderTarget(g_D3DApp->m_windowWidth, g_D3DApp->m_windowHeight, rtBufferName, 0, fmt); 
    if (rt16b_buffer->bFresh) {
      // clear to zero
      g_D3DApp->clearRenderTarget(rt16b_buffer, D3DCOLOR_RGBA(0, 0, 0, 0));
    }
    rt16b_buffer_last = g_D3DApp->createRenderTarget(g_D3DApp->m_windowWidth, g_D3DApp->m_windowHeight, bufNameLast, 0, fmt); 
    if (rt16b_buffer_last->bFresh) {
      // clear to zero
      g_D3DApp->clearRenderTarget(rt16b_buffer_last, D3DCOLOR_RGBA(0, 0, 0, 0));
    }
    g_D3DApp->setRenderTarget(rt16b_buffer_last, rtIndex);
    pCurrentPrevRt16bs[rtIndex] = rt16b_buffer_last;
    bDoubleBufferRt16bs[rtIndex] = true;
  } else {
    rt16b_buffer = g_D3DApp->createRenderTarget(g_D3DApp->m_windowWidth, g_D3DApp->m_windowHeight, rtBufferName, 0, fmt); 
    g_D3DApp->setRenderTarget(rt16b_buffer, rtIndex);
    bDoubleBufferRt16bs[rtIndex] = false;
  }
  pCurrentRt16bs[rtIndex] = rt16b_buffer;

}

void UnsetRt16b(int rtIndex) {
  if (bDoubleBufferRt16bs[rtIndex]) {
    // double buffered unset -> flip the buffers
    DemoContextData *pDCD = (DemoContextData*)GetCurrentContextData();
    // trick is now to internally swap the textures of current and prev buffers
    Texture *rt16b_buffer = pCurrentRt16bs[rtIndex];
    Texture *rt16b_buffer_last = pCurrentPrevRt16bs[rtIndex];

    LPDIRECT3DTEXTURE9 pTex = rt16b_buffer->lpTexture;
    LPDIRECT3DSURFACE9 pDepth = rt16b_buffer->lpDepthSurface;

    rt16b_buffer->lpTexture = rt16b_buffer_last->lpTexture;
    rt16b_buffer->lpDepthSurface = rt16b_buffer_last->lpDepthSurface;

    rt16b_buffer_last->lpTexture = pTex;
    rt16b_buffer_last->lpDepthSurface = pDepth;

  }
//  if (pCurrentRt16bs[rtIndex] && rtIndex==0) {
 //   g_D3DApp->resetRenderTarget();
 // }
}


ScrollArea::ScrollArea() {
  m_posX = 0;
  m_posY = 0;
  m_bScrollingWithBar = false;
  ResetSize();
}

void ScrollArea::ResetSize() {
  m_visibleWidth = 0;
  m_visibleHeight = 0;
  m_contentWidth = 0;
  m_contentHeight = 0;

  m_contentMaxY = 0;
  m_contentMinY = 100000000;
}

void ScrollArea::SetVisibleSize(int width, int height) {
  m_visibleWidth = width;
  m_visibleHeight = height;
}

void ScrollArea::SetContentAt(int minX, int minY, int maxX, int maxY) {
  if (maxX > m_contentWidth) {
    m_contentWidth = maxX;
  }
  if (maxY > m_contentHeight) {
    m_contentHeight = maxY;
  }
  if (maxY > m_contentMaxY) {
    m_contentMaxY = maxY;
  }
  if (minY < m_contentMinY) {
    m_contentMinY = minY;
  }
}

int ScrollArea::GetPosY() {
  return m_posY;
}
void ScrollArea::SetPosY(int p) {
  m_posY = p;
}
int ScrollArea::GetMaxY() {
  return max(0, m_contentMaxY);
}
int ScrollArea::GetContentHeight() {
  return max(0, m_contentMaxY-m_contentMinY);
}
int ScrollArea::GetVisibleHeight() {
  return m_visibleHeight;
}

void ScrollArea::RenderContent(D3DApp *pD3DApp, int top, Texture *pContentTexture) {
  int scrollPosY = GetPosY();
  int scrollAreaHeight = GetVisibleHeight();

  D3DXMATRIXA16 wm;
  D3DXMATRIXA16 tm;
  D3DXMATRIXA16 temp;
  D3DXVECTOR3 trans(pD3DApp->m_windowWidth*0.5, top+scrollAreaHeight*0.5, 0.0f);
  D3DXVECTOR3 scale(pD3DApp->m_windowWidth*0.5, -scrollAreaHeight*0.5, 1.0f);
  pD3DApp->MakeWorldMatrix(&wm, &trans, &scale, 0);
  pD3DApp->m_pd3dDevice->SetTransform(D3DTS_WORLD, &wm);
  D3DXMatrixIdentity(&tm);
  D3DXMatrixScaling(&tm, 1.0f, scrollAreaHeight/(float)pContentTexture->height, 1.0f);
 // D3DXMatrixTranslation(&temp, 0.0f, (float)top/(float)pContentTexture->height, 0.0f);
 // D3DXMatrixMultiply(&tm, &temp, &tm);
  tm._32 += (float)top/(float)pContentTexture->height;
  pD3DApp->m_pd3dDevice->SetTransform(D3DTS_TEXTURE0, &tm);
  pD3DApp->m_pd3dDevice->SetTextureStageState(0, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT2);
  pD3DApp->setTexture(pContentTexture);
  
  pD3DApp->m_pd3dDevice->SetRenderState(D3DRS_LIGHTING, false);
  pD3DApp->m_pd3dDevice->LightEnable(0, false);
  pD3DApp->m_pd3dDevice->SetRenderState(D3DRS_ALPHABLENDENABLE, false);
  pD3DApp->m_pd3dDevice->SetRenderState(D3DRS_ZENABLE, false);
  pD3DApp->drawSquare(1.0, 1.0, 1.0, 1.0f);
  D3DXMatrixIdentity(&wm);
  pD3DApp->m_pd3dDevice->SetTransform(D3DTS_WORLD, &wm);
  pD3DApp->m_pd3dDevice->SetTransform(D3DTS_TEXTURE0, &wm);
  pD3DApp->m_pd3dDevice->SetTextureStageState(0, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_DISABLE);

}

void ScrollArea::RenderScrollBar(D3DApp *pD3DApp, int top) {
  int scrollPosY = GetPosY();
  int scrollMaxY = m_contentMaxY-m_contentMinY; // GetMaxY()+GetVisibleHeight();
  int scrollAreaHeight = GetVisibleHeight();
  int scrollBarHeight = scrollAreaHeight*((float)scrollAreaHeight/scrollMaxY);

  D3DXMATRIXA16 wm;
  D3DXVECTOR3 trans(pD3DApp->m_windowWidth-5, top+(float)scrollPosY/(float)(scrollMaxY-m_visibleHeight+1)*(scrollAreaHeight-scrollBarHeight)+scrollBarHeight*0.5f, 0.0f);
  D3DXVECTOR3 scale(5.0f, scrollBarHeight*0.5f, 1.0f);
  pD3DApp->MakeWorldMatrix(&wm, &trans, &scale, 0);
  pD3DApp->m_pd3dDevice->SetTransform(D3DTS_WORLD, &wm);
  Texture *pValkoTex = pD3DApp->addTexture("images/valko.png");
  if (pValkoTex) {
    pD3DApp->setTexture(pValkoTex);
  }
  pD3DApp->m_pd3dDevice->SetRenderState(D3DRS_LIGHTING, false);
  pD3DApp->m_pd3dDevice->LightEnable(0, false);
  pD3DApp->m_pd3dDevice->SetRenderState(D3DRS_ALPHABLENDENABLE, false);
  pD3DApp->m_pd3dDevice->SetRenderState(D3DRS_ZENABLE, false);
  pD3DApp->drawSquare(1.0, 1.0, 1.0, 1.0f);
  D3DXMatrixIdentity(&wm);
  pD3DApp->m_pd3dDevice->SetTransform(D3DTS_WORLD, &wm);

  float scaleMoveY = (float)(scrollMaxY-m_visibleHeight)/(scrollAreaHeight-scrollBarHeight);
  {
    MoverMouseSensor mSMS(pD3DApp->m_windowWidth-10, pD3DApp->m_windowWidth, top+(float)scrollPosY/(float)(scrollMaxY-m_visibleHeight+1)*(scrollAreaHeight-scrollBarHeight), top+(float)scrollPosY/(float)(scrollMaxY-m_visibleHeight+1)*(scrollAreaHeight-scrollBarHeight)+scrollBarHeight, &m_posX, &m_posY, 0, 0, 0, GetContentHeight()-m_visibleHeight, 0.0f, scaleMoveY, this);
    moverMouseSensors.push_back(mSMS);
  }
  MouseSensor *pMs = (MouseSensor*)&moverMouseSensors.back();
  AddMouseSensor(pMs);
}

void Timeline::RenderEffects() {
  HRESULT hr;
  LPDIRECT3DDEVICE9 pd3dDevice = g_D3DApp->m_pd3dDevice;

  if (g_bAdvanceAndRenderEffects) {
    pd3dDevice->Clear( 0L, NULL, D3DCLEAR_TARGET | D3DCLEAR_ZBUFFER, D3DCOLOR_RGBA(0, 0, 0, 0), 1.0f, 0 );


    vector<Effect*>::iterator itForClearAll = m_effects.begin();

  //  Texture *rt16b_buffer0 = g_D3DApp->createRenderTarget(g_D3DApp->m_windowWidth, g_D3DApp->m_windowHeight, "rt16b_buffer0", 0, D3DFMT_A32B32G32R32F); // D3DFMT_A16B16G16R16F); // D3DFMT_A16B16G16R16F);
    Texture *rt16b_buffer0 = g_D3DApp->createRenderTarget(g_D3DApp->m_windowWidth, g_D3DApp->m_windowHeight, "rt16b_buffer0", 0, D3DFMT_A16B16G16R16F); // D3DFMT_A16B16G16R16F); // D3DFMT_A16B16G16R16F);
    for (vector<Effect*>::iterator it=m_effects.begin(); it!=m_effects.end(); it++) {
      if ((*it)->GetClearFlags()==3) {
        itForClearAll = it;
      }
      if ((*it)->IsActive() && (*it)->IsEnabled()) {


        (*it)->SetRenderStartTime(PerfGetTickCount());
        
        if ((*it)->GetType() == "dfmix") {
          Deferred *pDeferred = GetDeferred();

          // re-render every effect after previous clear for every shadow-casting light source having the light source rt as rt
          vector<Light>* globalLights = GetGlobalLights();
          int iLight = 0;
          D3DXVECTOR3 origCameraPos = GetGlobalCameraPos();
          for (vector<Light>::iterator itLight = globalLights->begin(); itLight != globalLights->end(); itLight++, iLight++) {
            if (!(*itLight).m_bCastShadows) {
              continue;
            }
            std::string lightBufName;
            char tempStr[512];
            sprintf(tempStr, "rtLightDepth%d", iLight);
            lightBufName = tempStr;
            Texture *pRtDepthForLight = g_D3DApp->createRenderTarget(g_D3DApp->m_windowWidth, g_D3DApp->m_windowHeight, lightBufName, 0, D3DFMT_R32F);
            // 1.set pRtDepthForLight as the deferred depth buffer
            // 2.render all effects since itForClearAll until this effect
            // 3.reset rendertarget

            // 4. in the dfmix, use the light depth buffers to make shadows
            g_D3DApp->clearRenderTarget(pRtDepthForLight, D3DCOLOR_RGBA(255, 255, 255, 255));

            pd3dDevice->Clear( 0L, NULL, D3DCLEAR_TARGET | D3DCLEAR_ZBUFFER, D3DCOLOR_RGBA(0, 0, 0, 0), 1.0f, 0 );
            
            Texture *pOrigDeferredDepth = pDeferred->GetRT16bDepth();
            pDeferred->SetRT16bDepth(pRtDepthForLight);
            pDeferred->SetRenderOnlyDepth(true);
            D3DXVECTOR3 lightCamPos = D3DXVECTOR3((*itLight).m_position.x, (*itLight).m_position.y, (*itLight).m_position.z);
            D3DXVECTOR3 camLookAt = GetGlobalCameraLookAt();
            lightCamPos = (lightCamPos-camLookAt*0.0)*1.0+camLookAt*0.0;
            SetGlobalCameraPos(lightCamPos);
            g_D3DApp->CopyMatrixToFloat16(GetGlobalCameraView(), (*itLight).m_viewMatrix);
         //   (*itLight).m_viewMatrix = *GetGlobalCameraView();
            (*itLight).m_pRtDepth = pRtDepthForLight;
            for (vector<Effect*>::iterator itEffectForLight=itForClearAll; itEffectForLight!=it; itEffectForLight++) {
              if (!(*itEffectForLight)->IsActive() || !(*itEffectForLight)->IsEnabled()) {
                continue;
              }
              if (!(*itEffectForLight)->DoRenderForLights()) {
                continue;
              }
              g_D3DApp->m_pd3dDevice->SetPixelShader(NULL);
              g_D3DApp->m_pd3dDevice->SetVertexShader(NULL);
              (*itEffectForLight)->Render();
            }
            pDeferred->SetRT16bDepth(pOrigDeferredDepth);
            pDeferred->SetRenderOnlyDepth(false);

           // iLight++;
          }
          SetGlobalCameraPos(origCameraPos);
          pDeferred->SetView(GetGlobalCameraView());
        }
        
        

        g_currentEffect = *it;
        std::vector<RtBuffer> rt16bBufferNames = (*it)->GetRT16bBufferNames();
        Texture *rt16b_buffer = NULL;
        Texture *rt16b_buffer_last = NULL;

        bool bRt16bSet = false;

        for (int i=0; i<rt16bBufferNames.size(); i++) {
          if (rt16bBufferNames[i].m_name != "") {
            std::string bufName = rt16bBufferNames[i].m_name;
            
            if ((*it)->GetDoubleBuffer()) {
              std::string bufNameLast = rt16bBufferNames[i].m_name+"_prev";
              rt16b_buffer = g_D3DApp->createRenderTarget(g_D3DApp->m_windowWidth, g_D3DApp->m_windowHeight, bufName, 0, rt16bBufferNames[i].m_fmt); //D3DFMT_A16B16G16R16F);
              if (rt16b_buffer->bFresh) {
                // clear to zero
                g_D3DApp->clearRenderTarget(rt16b_buffer, D3DCOLOR_RGBA(0, 0, 0, 0));
              }
              rt16b_buffer_last = g_D3DApp->createRenderTarget(g_D3DApp->m_windowWidth, g_D3DApp->m_windowHeight, bufNameLast, 0, rt16bBufferNames[i].m_fmt); //D3DFMT_A16B16G16R16F);
              if (rt16b_buffer_last->bFresh) {
                // clear to zero
                g_D3DApp->clearRenderTarget(rt16b_buffer_last, D3DCOLOR_RGBA(0, 0, 0, 0));
              }
              g_D3DApp->setRenderTarget(rt16b_buffer_last, i);
            } else {
              rt16b_buffer = g_D3DApp->createRenderTarget(g_D3DApp->m_windowWidth, g_D3DApp->m_windowHeight, bufName, 0, rt16bBufferNames[i].m_fmt); //D3DFMT_A16B16G16R16F);
              if (rt16b_buffer) {
                g_D3DApp->setRenderTarget(rt16b_buffer, i);
              }
            }
            bRt16bSet = true; 
          }
        }
        if (!bRt16bSet) {
          g_D3DApp->setRenderTarget(0, 0);
        }
        DWORD clearFlags = (*it)->GetClearFlags();
        if (clearFlags) {
          pd3dDevice->Clear( 0L, NULL, clearFlags, D3DCOLOR_RGBA(0, 0, 0, 0), 1.0f, 0 );
        }
        g_D3DApp->m_pd3dDevice->SetPixelShader(NULL);
        g_D3DApp->m_pd3dDevice->SetVertexShader(NULL);
        (*it)->Render();

        for (int i=0; i<rt16bBufferNames.size(); i++) {
          if (rt16bBufferNames[i].m_name != "") {
            
            std::string bufName = rt16bBufferNames[i].m_name;
            
            if ((*it)->GetDoubleBuffer()) {
              std::string bufNameLast = rt16bBufferNames[i].m_name+"_prev";
              rt16b_buffer = g_D3DApp->createRenderTarget(g_D3DApp->m_windowWidth, g_D3DApp->m_windowHeight, bufName, 0, rt16bBufferNames[i].m_fmt); //D3DFMT_A16B16G16R16F);
              if (rt16b_buffer->bFresh) {
                // clear to zero
                g_D3DApp->clearRenderTarget(rt16b_buffer, D3DCOLOR_RGBA(0, 0, 0, 0));
              }
              rt16b_buffer_last = g_D3DApp->createRenderTarget(g_D3DApp->m_windowWidth, g_D3DApp->m_windowHeight, bufNameLast, 0, rt16bBufferNames[i].m_fmt);
              if (rt16b_buffer_last->bFresh) {
                // clear to zero
                g_D3DApp->clearRenderTarget(rt16b_buffer_last, D3DCOLOR_RGBA(0, 0, 0, 0));
              }
              DemoContextData *pDCD = (DemoContextData*)GetCurrentContextData();
              // trick is now to internally swap the textures of current and prev buffers
              LPDIRECT3DTEXTURE9 pTex = rt16b_buffer->lpTexture;
              LPDIRECT3DSURFACE9 pDepth = rt16b_buffer->lpDepthSurface;

              rt16b_buffer->lpTexture = rt16b_buffer_last->lpTexture;
              rt16b_buffer->lpDepthSurface = rt16b_buffer_last->lpDepthSurface;

              rt16b_buffer_last->lpTexture = pTex;
              rt16b_buffer_last->lpDepthSurface = pDepth;
            }
            //if (i!=0) {
             // g_D3DApp->setRenderTarget(0, i);
            //}
       //     UnsetRt16b(i);
          }
        }
        if (bRt16bSet) {
          g_D3DApp->resetRenderTarget();
        }
        if (g_D3DAppDebug && m_bFlushDeviceBetweenEffects) {
          g_D3DApp->FlushDevice();
        }
        (*it)->SetRenderEndTime(PerfGetTickCount());
      }
    }

   // return;
  }

  //return;

  int rowsWritten = 0;

  D3DCOLOR textColor = 0xFFFFFFFF;
  char rimpsu[512];

  double totalTime = 0.0;

  if (m_showRtIndex >= 0 && m_showRtIndex < g_D3DApp->m_pRenderTargets.size()) {
    HRESULT hr= g_D3DApp->m_pd3dDevice->BeginScene();
    if (!FAILED(hr)) {

      g_D3DApp->m_pd3dDevice->Clear( 0L, NULL, D3DCLEAR_TARGET | D3DCLEAR_ZBUFFER, D3DCOLOR_RGBA(0, 0, 0, 0), 1.0f, 0 );

      g_D3DApp->resetRenderTarget();

      g_D3DApp->setTexture(g_D3DApp->m_pRenderTargets[m_showRtIndex]);

      g_D3DApp->m_pd3dDevice->SetPixelShader(NULL);
      g_D3DApp->m_pd3dDevice->SetTexture(1, NULL);
      g_D3DApp->m_pd3dDevice->SetTexture(2, NULL);
      g_D3DApp->m_pd3dDevice->SetTexture(3, NULL);
      g_D3DApp->m_pd3dDevice->SetTexture(4, NULL);
    
      g_D3DApp->m_pd3dDevice->SetRenderState(D3DRS_LIGHTING, false);
      g_D3DApp->m_pd3dDevice->LightEnable(0, false);
      g_D3DApp->m_pd3dDevice->SetRenderState(D3DRS_ALPHABLENDENABLE, false);
      g_D3DApp->m_pd3dDevice->SetRenderState(D3DRS_ZENABLE, false);


      g_D3DApp->m_pd3dDevice->SetRenderState(D3DRS_CULLMODE, D3DCULL_CCW); 
      g_D3DApp->m_pd3dDevice->SetRenderState(D3DRS_LIGHTING, false);
      g_D3DApp->m_pd3dDevice->LightEnable(0, false);
      g_D3DApp->m_pd3dDevice->SetRenderState(D3DRS_ALPHABLENDENABLE, true);

      g_D3DApp->m_pd3dDevice->SetTextureStageState(0, D3DTSS_COLORARG1, D3DTA_TEXTURE);
      g_D3DApp->m_pd3dDevice->SetTextureStageState(0, D3DTSS_COLORARG2, D3DTA_CURRENT);
      g_D3DApp->m_pd3dDevice->SetTextureStageState(0, D3DTSS_TEXCOORDINDEX, 0);
      g_D3DApp->m_pd3dDevice->SetTextureStageState(0, D3DTSS_COLOROP, D3DTOP_MODULATE);
      g_D3DApp->m_pd3dDevice->SetTextureStageState(0, D3DTSS_ALPHAOP, D3DTOP_SELECTARG1);
      g_D3DApp->m_pd3dDevice->SetTextureStageState(0, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_DISABLE);
      g_D3DApp->m_pd3dDevice->SetTextureStageState(0, D3DTSS_COLORARG0, D3DTA_CURRENT);
     
      g_D3DApp->m_pd3dDevice->SetRenderState(D3DRS_TEXTUREFACTOR, 0xFFFFFFFF);
      g_D3DApp->m_pd3dDevice->SetRenderState(D3DRS_SRCBLEND, D3DBLEND_ONE);
      g_D3DApp->m_pd3dDevice->SetRenderState(D3DRS_DESTBLEND, D3DBLEND_ZERO);
      g_D3DApp->m_pd3dDevice->SetTextureStageState(0, D3DTSS_ALPHAARG1, D3DTA_DIFFUSE);

      g_D3DApp->m_pd3dDevice->SetSamplerState( 0, D3DSAMP_MINFILTER, D3DTEXF_LINEAR );
      g_D3DApp->m_pd3dDevice->SetSamplerState( 0, D3DSAMP_MAGFILTER, D3DTEXF_LINEAR );

      g_D3DApp->m_pd3dDevice->SetSamplerState(0, D3DSAMP_ADDRESSU, D3DTADDRESS_MIRROR);
      g_D3DApp->m_pd3dDevice->SetSamplerState(0, D3DSAMP_ADDRESSV, D3DTADDRESS_MIRROR);


      g_D3DApp->m_pd3dDevice->SetTextureStageState(0, D3DTSS_ALPHAOP, D3DTOP_BLENDDIFFUSEALPHA);
      g_D3DApp->m_pd3dDevice->SetTextureStageState(0, D3DTSS_ALPHAARG1, D3DTA_TEXTURE);

      D3DXMATRIXA16 wm;
      D3DXMatrixIdentity(&wm);
      g_D3DApp->m_pd3dDevice->SetTransform(D3DTS_WORLD, &wm);
      g_D3DApp->m_pd3dDevice->SetTransform(D3DTS_VIEW, &wm);
      g_D3DApp->m_pd3dDevice->SetTransform(D3DTS_PROJECTION, &wm);

      D3DXMATRIXA16 tm;
      D3DXMatrixIdentity(&tm);
    //  D3DXMatrixScaling(&tm, 1.0f-1.0f/(float)g_D3DApp->m_pRenderTargets[m_showRtIndex]->width, 1.0f-1.0f/(float)g_D3DApp->m_pRenderTargets[m_showRtIndex]->height, 1.0f);
      tm._31 += (float)0.5f/(float)g_D3DApp->m_pRenderTargets[m_showRtIndex]->width;
      tm._32 += (float)0.5f/(float)g_D3DApp->m_pRenderTargets[m_showRtIndex]->height;
      g_D3DApp->m_pd3dDevice->SetTransform(D3DTS_TEXTURE0, &tm);
      g_D3DApp->m_pd3dDevice->SetTextureStageState(0, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT2);
    
     
      g_D3DApp->drawSquare(1.0, 1.0, 1.0, 1.0f);


      D3DXMatrixIdentity(&tm);
      g_D3DApp->m_pd3dDevice->SetTransform(D3DTS_TEXTURE0, &tm);
      g_D3DApp->m_pd3dDevice->SetTextureStageState(0, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_DISABLE);


      g_D3DApp->m_pd3dDevice->EndScene();
    }
  }

  D3DApp *pD3DApp = g_D3DApp;
  if (g_D3DAppDebug) {
    pD3DApp = g_D3DAppDebug;
  } else {
    return; // (no debug infos when no debug device?)
  }

  int rowH = 12;
  int rowStart = 10+(rowsWritten+4)*rowH;

  bool bEnteredBeginScene = true;
  if (FAILED(pD3DApp->m_pd3dDevice->BeginScene())) {
    bEnteredBeginScene = false;
  }

  LPD3DXFONT theFont = pD3DApp->getFont(12, 1, 0, "Arial");
  if (theFont) {
   pD3DApp->beginText(theFont);
  }

  MouseSensor *pMs;

  int showInfoModeItemWidth = 100;

  if (m_showInfoMode==ShowInfoMode::EFFECTS) {
    textColor = 0xFFFFFFFF;
    sprintf(rimpsu, "EFFECTS");
  } else {
    textColor = 0xFF555555;
    sprintf(rimpsu, "EFFECTS");
  }
  rowStart = 10+(rowsWritten+4)*rowH;
  {
    TimelineInfoModeMouseSensor MS(5+showInfoModeItemWidth*0, 5+showInfoModeItemWidth*1, rowStart, rowStart+rowH, ShowInfoMode::EFFECTS, this);
    timelineInfoModeMouseSensors.push_back(MS);
  }
  pMs = (MouseSensor*)&timelineInfoModeMouseSensors.back();
  AddMouseSensor(pMs);
  pD3DApp->RenderDebugInfo(rimpsu, rowsWritten+4, textColor, true, 5+showInfoModeItemWidth*0);

  if (m_showInfoMode==ShowInfoMode::RENDERTARGETS) {
    textColor = 0xFFFFFFFF;
    sprintf(rimpsu, "RENDERTARGETS");
  } else {
    textColor = 0xFF555555;
    sprintf(rimpsu, "RENDERTARGETS");
  }
  rowStart = 10+(rowsWritten+4)*rowH;
  {
    TimelineInfoModeMouseSensor MS(5+showInfoModeItemWidth*1, 5+showInfoModeItemWidth*2, rowStart, rowStart+rowH, ShowInfoMode::RENDERTARGETS, this);
    timelineInfoModeMouseSensors.push_back(MS);
  }
  pMs = (MouseSensor*)&timelineInfoModeMouseSensors.back();
  AddMouseSensor(pMs);
  pD3DApp->RenderDebugInfo(rimpsu, rowsWritten+4, textColor, true, 5+showInfoModeItemWidth*1);

  rowsWritten++;
  rowsWritten++;


  if (m_showInfoMode == ShowInfoMode::EFFECTS) {


    rowStart = 10+(rowsWritten+4)*rowH;

    if (m_bFlushDeviceBetweenEffects) {
      textColor = 0xFFFFFFFF;
      sprintf(rimpsu, "flush between effects ON");
    } else {
     textColor = 0x999999FF;
      sprintf(rimpsu, "flush between effects OFF");
    }
    {
      BooleanSwitchMouseSensor bSMS(0, 300, rowStart, rowStart+rowH, &m_bFlushDeviceBetweenEffects);
      booleanSwitchMouseSensors.push_back(bSMS);
    }
    pMs = (MouseSensor*)&booleanSwitchMouseSensors.back();
    AddMouseSensor(pMs);
    pD3DApp->RenderDebugInfo(rimpsu, rowsWritten+4, textColor, true);
    rowsWritten++;


    if (m_bShowInactiveEffects) {
      textColor = 0xFFFFFFFF;
      sprintf(rimpsu, "show inactive effects ON");
    } else {
      textColor = 0x999999FF;
      sprintf(rimpsu, "show inactive effects OFF");
    }
    rowStart = 10+(rowsWritten+4)*rowH;
    {
      BooleanSwitchMouseSensor bSMS(0, 300, rowStart, rowStart+rowH, &m_bShowInactiveEffects);
      booleanSwitchMouseSensors.push_back(bSMS);
    }
    pMs = (MouseSensor*)&booleanSwitchMouseSensors.back();
    AddMouseSensor(pMs);
    pD3DApp->RenderDebugInfo(rimpsu, rowsWritten+4, textColor, true);

    rowsWritten++;
    rowsWritten++;
    rowsWritten++;
    rowsWritten++;
    rowsWritten++;

    int rowsWrittenTop = rowsWritten+1;



    rowStart = 10+(rowsWritten+4)*rowH;
    int rowStartTop = rowStart;

    int scrollX = 0;
    int scrollY = m_pScrollAreaFxList->GetPosY();

    m_pScrollAreaFxList->ResetSize();

    Texture *pInfoRT = NULL;
    if (pD3DApp->m_windowWidthResized<= 0 || pD3DApp->m_windowHeightResized<=0) {
      pInfoRT = pD3DApp->getTexture("rtInfoRT");
    } else {
      pInfoRT = pD3DApp->createRenderTarget(pD3DApp->m_windowWidthResized, pD3DApp->m_windowHeightResized, "rtInfoRT");
    }
    if (pInfoRT) {
      pD3DApp->clearRenderTarget(pInfoRT, D3DCOLOR_RGBA(0,0,0,0));
    }
    if (theFont) {
      pD3DApp->endText();
      pD3DApp->setRenderTarget(pInfoRT);
      pD3DApp->beginText(theFont);
    }

    for (vector<Effect*>::iterator it=m_effects.begin(); it!=m_effects.end(); it++) {
      if ((*it)->IsActive()) {

        if (!m_bShowInactiveEffects && !(*it)->IsEnabled()) {
          continue;
        }

        if ((*it)->IsTimelineTypeCamera())
          textColor = 0xFF9999FF;
        else if ((*it)->IsTimelineTypeLight())
          textColor = 0xFFFFFF99;
        else if ((*it)->IsTimelineTypeRender())
          textColor = 0xFFAAFFAA;
        else if ((*it)->IsTimelineTypePreProcess())
          textColor = 0xFFFF9999;
        else if ((*it)->IsTimelineTypeMixer())
          textColor = 0xFFFFFFFF;
        else if ((*it)->IsTimelineTypePostProcess())
          textColor = 0xFF99FFFF;
        else if ((*it)->IsTimelineTypeConversion())
          textColor = 0xFF999999;
      
        D3DCOLOR onOffTextColor = 0xAA22FF22;
        string onOffString = "ON";
        if (!(*it)->IsEnabled()) {
          onOffTextColor = 0xAAFF2222;
          onOffString = "OFF";
        }

        int artStart = 30;
        int artGap = 90;
        int percOfs = 60;
        sprintf(rimpsu, "a:%3.2fms", (float)(*it)->GetPrevAdvanceTime());
        pD3DApp->RenderDebugInfo(rimpsu, rowsWritten+4, textColor, true, artStart+artGap*0-scrollX, 10-scrollY);
        sprintf(rimpsu, "%2.0f%%", (float)((*it)->GetPrevAdvanceTime()/GetPrevTotalTime())*100.0f);
        pD3DApp->RenderDebugInfo(rimpsu, rowsWritten+4, textColor, true, artStart+artGap*0+percOfs-scrollX, 10-scrollY);

        sprintf(rimpsu, "r:%3.2fms", (float)(*it)->GetPrevRenderTime());
        pD3DApp->RenderDebugInfo(rimpsu, rowsWritten+4, textColor, true, artStart+artGap*1-scrollX, 10-scrollY);
        sprintf(rimpsu, "%2.0f%%", ((float)(*it)->GetPrevRenderTime()/GetPrevTotalTime())*100.0f);
        pD3DApp->RenderDebugInfo(rimpsu, rowsWritten+4, textColor, true, artStart+artGap*1+percOfs-scrollX, 10-scrollY);

        sprintf(rimpsu, "t:%3.2fms", (float)(*it)->GetPrevAdvanceTime()+(float)(*it)->GetPrevRenderTime());
        pD3DApp->RenderDebugInfo(rimpsu, rowsWritten+4, textColor, true, artStart+artGap*2-scrollX, 10-scrollY);
        sprintf(rimpsu, "%2.0f%%",((float)((*it)->GetPrevAdvanceTime()+(*it)->GetPrevRenderTime())/GetPrevTotalTime())*100.0f);
        pD3DApp->RenderDebugInfo(rimpsu, rowsWritten+4, textColor, true, artStart+artGap*2+percOfs-scrollX, 10-scrollY);

        sprintf(rimpsu, "%s", (*it)->GetFilename().c_str()); // , (float)(*it)->GetPrevAdvanceTime(), (float)(*it)->GetPrevRenderTime(), (float)(*it)->GetPrevAdvanceTime()+(float)(*it)->GetPrevRenderTime());
        pD3DApp->RenderDebugInfo(rimpsu, rowsWritten+4, textColor, true, artStart+artGap*3-scrollX, 10-scrollY);


        rowStart = 10+(rowsWritten+4)*rowH;
        TimelineMouseSensor tMS(0, 300, rowStart-scrollY, rowStart-scrollY+rowH, *it, 0);
        timelineMouseSensors.push_back(tMS);
        MouseSensor *ms = (MouseSensor*)&timelineMouseSensors.back();
        AddMouseSensor(ms);

        TimelineMouseSensor tMS2(300, 400, rowStart-scrollY, rowStart-scrollY+rowH, *it, 1);
        timelineMouseSensors.push_back(tMS2);
        ms = (MouseSensor*)&timelineMouseSensors.back();
        AddMouseSensor(ms);

        

        pD3DApp->RenderDebugInfo(onOffString.c_str(), rowsWritten+4, onOffTextColor, true, 5-scrollX, 10-scrollY);
        m_pScrollAreaFxList->SetContentAt(0, rowStart, 300, rowStart+rowH);

        rowsWritten++;

        if ((*it)->IsEnabled()) {
          totalTime += (*it)->GetPrevAdvanceTime()+(*it)->GetPrevRenderTime();
        }
      }
    }

    if (theFont) {
      pD3DApp->endText();
      pD3DApp->resetRenderTarget();
      pD3DApp->beginText(theFont);
    }


    int rowsEnd = 10+(rowsWritten+4)*rowH;
    m_pScrollAreaFxList->SetVisibleSize(pD3DApp->m_windowWidth, min(rowsEnd-rowStartTop, pD3DApp->m_windowHeight-rowStartTop));
    m_pScrollAreaFxList->RenderContent(pD3DApp, rowStartTop, pInfoRT);
    m_pScrollAreaFxList->RenderScrollBar(pD3DApp, rowStartTop);

    rowsWritten++;

    textColor = 0xFF88FFFF;
    sprintf(rimpsu, "total %3.2fms", (float)totalTime);
    pD3DApp->RenderDebugInfo(rimpsu, rowsWrittenTop, textColor, true, 5);
    rowsWritten++;

    textColor = 0xFFFFFFFF;

    sprintf(rimpsu, "meshes %d / %d   faces %d / %d   verts %d / %d", g_numMeshesDrawn, g_numMeshes, g_numFacesDrawn, g_numFaces, g_numVerticesDrawn, g_numVertices);
    pD3DApp->RenderDebugInfo(rimpsu, rowsWrittenTop+1, textColor, true, 5);
    rowsWritten++;
  }



  if (m_showInfoMode == ShowInfoMode::RENDERTARGETS) {

    int rowsWrittenTop=rowsWritten;
    rowsWritten++;
    rowStart = 10+(rowsWritten+4)*rowH;
    int rowStartTop = rowStart;
    int scrollX = 0;
    int scrollY = m_pScrollAreaRtList->GetPosY();

    m_pScrollAreaRtList->ResetSize();

    Texture *pInfoRT = pD3DApp->createRenderTarget(pD3DApp->m_windowWidthResized, pD3DApp->m_windowHeightResized, "rtInfoRT");
    pD3DApp->clearRenderTarget(pInfoRT, D3DCOLOR_RGBA(0,0,0,0));
    if (theFont) {
      pD3DApp->endText();
      pD3DApp->setRenderTarget(pInfoRT);
      pD3DApp->beginText(theFont);
    }

    int i=-1;
    int totalSizeInBytes = 0;

    for (vector<Texture*>::iterator it=g_D3DApp->m_pRenderTargets.begin(); it!=g_D3DApp->m_pRenderTargets.end(); ) {
      if (i==m_showRtIndex) {
        textColor = 0xFFFFFFFF;
      } else {
        textColor = 0xFF888888;
      }

      char tempStr[512];

      int bitsPerPixel = 8*4;
      int rtWidth = g_D3DApp->m_windowWidth;
      int rtHeight = g_D3DApp->m_windowHeight;

      std::string bitDepth = "8b RGBA";
      std::string reso;
      std::string sizeInMb;
      std::string bufferName = "backbuffer";

      if (i==-1) {
        sprintf(tempStr, "%dx%d", g_D3DApp->m_windowWidth, g_D3DApp->m_windowHeight);
        reso = tempStr;
      }
      if (i>=0) {
        sprintf(tempStr, "%dx%d", (*it)->width, (*it)->height);
        reso = tempStr;

        sprintf(tempStr, "Unknown bit depth: %d", (*it)->d3dFormat);
        bitDepth = tempStr;

        if ((*it)->d3dFormat == D3DFMT_A8R8G8B8) {
          bitDepth = "8b RGBA";
          bitsPerPixel = 8*4;
        }
        if ((*it)->d3dFormat == D3DFMT_A32B32G32R32F) {
          bitDepth = "32b f RGBA";
          bitsPerPixel = 32*4;
        }
        if ((*it)->d3dFormat == D3DFMT_A16B16G16R16F) {
          bitDepth = "16b f RGBA";
          bitsPerPixel = 16*4;
        }
        if ((*it)->d3dFormat == D3DFMT_R32F) {
          bitDepth = "32b f R";
          bitsPerPixel = 32*1;
        }
        if ((*it)->d3dFormat == D3DFMT_G32R32F) {
          bitDepth = "32b f RG";
          bitsPerPixel = 32*2;
        }
        bufferName = (*it)->name;
      }

      int sizeInBytes = rtWidth*rtHeight*bitsPerPixel/8;
      totalSizeInBytes += sizeInBytes;
      sprintf(tempStr, "%4.2fmb", (float)sizeInBytes/(1024.0f*1024.0f));
      sizeInMb = tempStr;

      pD3DApp->RenderDebugInfo(bufferName.c_str(), rowsWritten+4, textColor, true, 10-scrollX, 10-scrollY);
      pD3DApp->RenderDebugInfo(bitDepth.c_str(), rowsWritten+4, textColor, true, 150-scrollX, 10-scrollY);
      pD3DApp->RenderDebugInfo(reso.c_str(), rowsWritten+4, textColor, true, 240-scrollX, 10-scrollY);
      pD3DApp->RenderDebugInfo(sizeInMb.c_str(), rowsWritten+4, textColor, true, 300-scrollX, 10-scrollY);
      rowStart = 10+(rowsWritten+4)*rowH;

      TimelineChangeShowRTMouseSensor tMS(10, 340, rowStart-scrollY, rowStart-scrollY+rowH, i, this);
      timelineChangeShowRTMouseSensors.push_back(tMS);
      MouseSensor *ms = (MouseSensor*)&timelineChangeShowRTMouseSensors.back();
      AddMouseSensor(ms);
      m_pScrollAreaRtList->SetContentAt(10, rowStart, 300, rowStart+rowH);
      rowsWritten++;
      i++;
      if (i>0) {
        it++;
      }
    }
  

    if (theFont) {
      pD3DApp->endText();
      pD3DApp->resetRenderTarget();
      pD3DApp->beginText(theFont);
    }
    int rowsEnd = 10+(rowsWritten+4)*rowH;
    m_pScrollAreaRtList->SetVisibleSize(pD3DApp->m_windowWidth, min(rowsEnd-rowStartTop, pD3DApp->m_windowHeight-rowStartTop));
    m_pScrollAreaRtList->RenderContent(pD3DApp, rowStartTop, pInfoRT);
    m_pScrollAreaRtList->RenderScrollBar(pD3DApp, rowStartTop);

    textColor = 0xFF22FF22;
    sprintf(rimpsu, "total size %3.1fmb", (float)totalSizeInBytes/(1024.0f*1024.0f));
    pD3DApp->RenderDebugInfo(rimpsu, rowsWrittenTop+4, textColor, true);

  }


  SetPrevTotalTime(totalTime);

  if (theFont) {
    pD3DApp->endText();
  }

  if (bEnteredBeginScene) {
    pD3DApp->m_pd3dDevice->EndScene();
  }

}

