(function($){
	// A simple object cache to hold loaded textures and geometries
	var Cache = function(){
			this.geometry = {};
			this.mesh = {};
			this.texture = {};
			this.material = {};
			
			
			this.set = function(bin, name, object){
				this[bin][name] = object;
			};
			
			this.get = function(bin, name){
				return this[bin][name];
			};
		},
	
	
	
	// The demo class	
		Demo = function(){
			
			// Demo settings
			this.settings = {
				audioAvailableEvent: 'MozAudioAvailable',
				viewport: {
					width: 800,
					height: 450,
					aspect: 1
				}
			};
			
			// Timing information
			this.timing = {
				startTime: 0,
				elapsed: 0,
				bpm: 120,
				barLength: 4,
				barDuration: 0,
				barsElapsed: 0,
				sceneElapsed: 0
			};
			
			// Audio data
			this.audio = {
				object: null,
				frameBuffer: [],
				time: 0
			};
			
			// The end handler (OVERWRITE THIS)
			this.endHandler = function(demoInstance){};
			
			this.scenes = []; // Scene queue
			this.currentScene = null; // Current scene to be rendered
			this.cache = new Cache(); // Object cache
			this.renderer = null; // WebGL renderer
			this.running = true;
		},
		p = Demo.prototype;
	
	
	
	
	
	// Audio methods
	
	/**
	 * Loads an audio file (.ogg) and sets up the audio object
	 *
	 * @param string
	 *	Path to the audio file.
	 *
	 * @param function
	 *	Callback for the audioAvailable event.
	 */
	p.loadAudio = function(filePath, callback){
		var audioObject = new Audio();
		audioObject.src = filePath;
		audioObject.addEventListener(this.settings.audioAvailableEvent, callback, false);
		this.audio.object = audioObject;
	};
	
	
	/**
	 * Handles the audioAvailable event
	 *
	 * This does nothing ATM. Firefox throws a security exception if the event.frameBuffer
	 * is accessed.
	 *
	 * @param object
	 *	The event object.
	 */
	p.onAudioAvailable = function(event){
		/*
		this.audio.frameBuffer = event.frameBuffer;
		this.audio.time = event.time;
		*/
	};
	
	
	/**
	 * Initializes audio for playback (delegate)
	 *
	 * @param string
	 *	Path to the audio file.
	 */
	p.initAudio = function(filePath){
		this.loadAudio(filePath, this.onAudioAvailable);
	};
	
	
	
	
	
	
	
	// Scene methods
	
	/**
	 * Add a scene to the scene queue
	 *
	 * @param object
	 *	An object of the DemoScene class.
	 *	@see DemoScene.js
	 */
	p.pushScene = function(sceneObj){
		this.scenes.push(sceneObj);
	};
	
	
	/**
	 * Selects and draws a scene from the queue
	 */
	p.drawScene = function(){
		var that = this;
		
		// Main loop
		if(this.running){
			requestAnimationFrame(function(){
				that.drawScene();
			});
		}
		
		
		// Update timing information
		this.timing.elapsed = (new Date()).getTime() - this.timing.start;
		this.timing.barDuration = 60000 / this.timing.bpm * this.timing.barLength;
		this.timing.barsElapsed = Math.floor(this.timing.elapsed / this.timing.barDuration);
		
		
		// Update scene queue
		if(this.currentScene == null || this.currentScene.endBar <= this.timing.barsElapsed){
			// Do we have scenes left in the queue?
			if(this.scenes.length > 0){
				this.currentScene = this.scenes.shift(); // Get the first one in the queue
			}
			else{
				this.end(); // We've reached the end of the queue
			}
			
			this.currentScene.startTime = this.timing.elapsed;
			this.currentScene.endBar = this.timing.barsElapsed + this.currentScene.duration;
		}
		
		// Update current scene
		this.currentScene.update();
		
		// Render the current scene
		this.renderer.render(this.currentScene.scene, this.currentScene.camera);
	};
	
	
	/**
	 * Initializes the renderer
	 *
	 * @param string
	 *	Canvas wrapper element ID.
	 */
	p.initRenderer = function(elementId){
		var $emt = $('#' + elementId),
			width = $emt.innerWidth(),
			height = $emt.innerHeight();
		
		this.settings.viewport = {
			width: width,
			height: height,
			aspect: width / height
		};
		
		var renderer = new THREE.WebGLRenderer({
			antialias: true
		});
		renderer.setSize(width, height);
		renderer.shadowMapEnabled = true;
		renderer.shadowMapSoft = true;
		
		$emt.append(renderer.domElement);
		
		this.renderer = renderer;
	};
	
	
	
	
	
	
	// Main system methods
	
	/**
	 * Initializes the demo system
	 *
	 * @param string
	 *	ID of the canvas wrapper element.
	 *
	 * @param string
	 *	Audio file path.
	 */
	p.init = function(canvasWrapperId, audioFile){
		this.initAudio(audioFile);
		this.initRenderer(canvasWrapperId);
	};
	
	
	/**
	 * Starts the demo
	 */
	p.start = function(){
		this.audio.object.play();
		this.timing.start = (new Date()).getTime();
		this.drawScene();
	};
	
	
	/**
	 * Stops everything and calls the end handler
	 */
	p.end = function(){
		this.running = false;
		this.audio.object.pause();
		this.endHandler(this);
	}
	
	
	
	
	
	
	// Export objects
	window.Demo = Demo;
})(jQuery);