

#include "Constant.fx"
#include "ConstantBuffer.fx"
#include "HDR.fx"
#include "Deferred.fx"

struct VSIn
{
    float3 Pos		: POSITION;
    float2 Tex		: TEXCOORD;
};

struct VSOut
{
	float4 Pos			: SV_POSITION;
	float4 Tex			: TEXCOORD0;
};

struct PSIn
{
	float4 Pos			: SV_POSITION;
	float4 Tex			: TEXCOORD0;
};

struct PSOut
{
	float4 c0 : SV_TARGET0;
}; 
   
Texture2D	g_texBlur0;
Texture2D	g_texBlur1;
Texture2D	g_texBlur2;
Texture2D	g_texInput;
float4		g_texInputInvSize;
float4		g_texInputSize;

Texture2D	g_texDepth;
float4		g_texDepthInvSize;
float4		g_depthUVScale;
float		DepthRange	<	float defaultValue = 1000; >	= 1000;
float		DofNear		<	float defaultValue = 500; >		= 500;
float		DofFar		<	float defaultValue = 1500; >	= 1500;
float		DofSize		<	float defaultValue = 4; >		= 4;

Texture2D	Bitmap
<
	string UIName = "Bitmap";
	string Default = "DefaultWhite.png";	
>;

Texture2D	Bitmap2
<
	string UIName = "Bitmap2";
	string Default = "DefaultWhite.png";	
>;

Texture2D	Bitmap3
<
	string UIName = "Bitmap3";
	string Default = "DefaultWhite.png";	
>;

Texture2D	Bitmap4
<
	string UIName = "Bitmap4";
	string Default = "DefaultWhite.png";	
>;

Texture2D	Mask
<
	string UIName = "Mask";
	string Default = "DefaultWhite.png";	
>;

float Opacity		
<	
	float defaultValue = 1.0f;  
	string UIName = "Opacity";
 	string Default = "1.0";
> = 1.0f;

float OpacityBias
<	
	float defaultValue = 0.0f;  
	string UIName = "OpacityBias";
 	string Default = "0.0";
> = 0.0f;

float4 Color
<	
	string UIName = "Color";
 	string Default = "0;0;0;1";
>;

float Multiplier	
<	
	float defaultValue = 1.0f;  
	string UIName = "Multiplier";
 	string Default = "1.0";
> = 1.0f;

float Power
<	
	float defaultValue = 1.0f;  
	string UIName = "Power";
 	string Default = "1.0";
> = 1.0f;

float Bias
<	
	float defaultValue = 0.0f;  
	string UIName = "Bias";
 	string Default = "0.0";
> = 0.0f;

float DirX
<	
	float defaultValue = 1.0f;  
	string UIName = "DirX";
 	string Default = "1.0";
> = 0.0f;

float DirY
<	
	float defaultValue = 0.0f;  
	string UIName = "DirY";
 	string Default = "0.0";
> = 0.0f;

float OffsetU
<	
	float defaultValue = 0.0f;  
	string UIName = "OffsetU";
 	string Default = "0.0";
> = 0.0f;

float OffsetV
<	
	float defaultValue = 0.0f;  
	string UIName = "OffsetV";
 	string Default = "0.0";
> = 0.0f;

float AngleUV
<	
	float defaultValue = 0.0f;  
	string UIName = "AngleUV";
 	string Default = "0.0";
> = 0.0f;

float ScaleU
<	
	float defaultValue = 1.0f;  
	string UIName = "ScaleU";
 	string Default = "1.0";
> = 1.0f;

float ScaleV
<	
	float defaultValue = 1.0f;  
	string UIName = "ScaleV";
 	string Default = "1.0";
> = 1.0f;


#define UV_XFORM_ARGS "OffsetU;OffsetV;AngleUV;ScaleU;ScaleV;"

float4 g_cameraNearFar;	//x=near, y=far, z=near*far, w=far-near
float4 g_gbufferUVScale;
float4 g_eyePos;

//--------------------------------------------------------------------------------------
VSOut VS(VSIn IN)
{
	VSOut OUT;

	OUT.Pos	= float4(IN.Pos.xyz,1);

	float2 uv = IN.Tex.xy - 0.5f;
	uv = uv * float2(ScaleU, ScaleV);

	float s = sin(AngleUV);
	float c = cos(AngleUV);

	float u = c * uv.x - s * uv.y;
	float v = s * uv.x + c * uv.y;
	uv = float2(u, v);
	uv = uv + 0.5f;
	uv = uv + float2(OffsetU, OffsetV);

	OUT.Tex	= float4( IN.Tex.xy, uv ); 

	return OUT;
}

//--------------------------------------------------------------------------------------
PSOut PSCopy(PSIn IN)
{
	PSOut OUT;
	float4 rgba  = g_texInput.Sample(SamplerPointClamp, IN.Tex.xy );
	OUT.c0 = rgba;
	return OUT;
}

//--------------------------------------------------------------------------------------
technique11 Copy
{
    pass p0 
    {
        SetVertexShader( CompileShader( vs_4_0, VS() ) );        
        SetPixelShader( CompileShader( ps_4_0, PSCopy() ) );
		SetRasterizerState( CullNone );
		SetBlendState( DefaultBlending, float4(0,0,0,0), 0x0ffffffff );
		SetDepthStencilState( DisableDepth, 0 );
    }
}

 
// float3 Uncharted2Tonemap(float3 x)
// {
// 	float A = 0.15;
// 	float B = 0.50;
// 	float C = 0.10;
// 	float D = 0.20;
// 	float E = 0.02;
// 	float F = 0.30;
// 	return ((x*(A*x+C*B)+D*E)/(x*(A*x+B)+D*F))-E/F;
// }

//--------------------------------------------------------------------------------------
// PSOut PSLinearToGamma(PSIn IN)
// {
// 	float3 whiteScale	= 1.0f / Uncharted2Tonemap(11.2f);
// 	float  ExposureBias = 1.0f;
// 
// 	PSOut OUT;
// 
// 	float3 rgb  = g_texInput.Sample(SamplerPointClamp, IN.Tex.xy ).rgb; // * 2.0f;
// 	//rgb		= ExposureBias * Uncharted2Tonemap(rgb) * whiteScale;
//  
//  	OUT.c0.rgb	= rgb;
// // 	OUT.c0.rgb	= pow(rgb, 1.0f / 2.2f);
// 	OUT.c0.a	= 1;
// 	return OUT;
// }

//--------------------------------------------------------------------------------------
// technique11 LinearToGamma
// {
//     pass p0 
//     {
//         SetVertexShader( CompileShader( vs_4_0, VS() ) );        
//         SetPixelShader( CompileShader( ps_4_0, PSLinearToGamma() ) );
// 		SetRasterizerState( CullNone );
// 		SetBlendState( DefaultBlending, float4(0,0,0,0), 0x0ffffffff );
// 		SetDepthStencilState( DisableDepth, 0 );
//     }
// }

//--------------------------------------------------------------------------------------
//
// ColorBalance

float3 _ColorBalance(float3 v, float3 midtones)
{
	float3 mid = 0.667f * (1.0f - sqrt(abs(v - 0.5f)));
	v = (v + midtones * mid);
	return v;
}

// float CyanRed		< float defaultValue = 0; > = 0;
// float MagentaGreen	< float defaultValue = 0; > = 0;
// float YellowBlue	< float defaultValue = 0; > = 0;

float CyanRed 
< 
	string UIName = "CyanRed";
	string Default = "0";
>;

float MagentaGreen 
< 
	string UIName = "MagentaGreen";
	string Default = "0";
>;

float YellowBlue 
< 
	string UIName = "YellowBlue";
	string Default = "0";
>;

PSOut PSColorBalance(PSIn IN)
{
	PSOut OUT; OUT.c0 = 1;

	float3 col  = g_texInput.Sample(SamplerPointClamp, IN.Tex.xy).rgb;	
	col = _ColorBalance(col, float3(CyanRed, MagentaGreen, YellowBlue));
	OUT.c0.rgb = col.rgb;
	return OUT;
}

//--------------------------------------------------------------------------------------
technique11 ColorBalance
{
    pass p0 < string args="CyanRed;MagentaGreen;YellowBlue;"; >
    {
        SetVertexShader( CompileShader( vs_4_0, VS() ) );        
        SetPixelShader( CompileShader( ps_4_0, PSColorBalance() ) );
		SetRasterizerState( CullNone );
		SetBlendState( DefaultBlending, float4(0,0,0,0), 0x0ffffffff );
		SetDepthStencilState( DisableDepth, 0 );
    }
}

//--------------------------------------------------------------------------------------

#define MODE_ADD			1
#define MODE_SUB			2
#define MODE_LIGHTEN		3
#define MODE_DARKEN			4
#define MODE_LINEAR_LIGHT	5
#define MODE_SCREEN			6	
#define MODE_OVERLAY		7
#define MODE_SOFT_LIGHT		8
#define MODE_COLOR_DODGE	9
#define MODE_COLOR_BURN		10
#define MODE_VIVID_LIGHT	11
#define MODE_PIN_LIGHT		12
#define MODE_HARDMIX		13
#define MODE_REFLECT		14
#define MODE_PRODUCT		15
#define MODE_NORMAL			16

#define BlendAddf(base, blend) 			min(base + blend, 1.0)
#define BlendSubstractf(base, blend) 	max(base + blend - 1.0, 0.0)
#define BlendLightenf(base, blend) 		max(blend, base)
#define BlendDarkenf(base, blend) 		min(blend, base)
#define BlendScreenf(base, blend) 		(1.0 - ((1.0 - base) * (1.0 - blend)))
#define BlendOverlayf(base, blend) 		(base < 0.5 ? (2.0 * base * blend) : (1.0 - 2.0 * (1.0 - base) * (1.0 - blend)))
#define BlendSoftLightf(base, blend) 	((blend < 0.5) ? (2.0 * base * blend + base * base * (1.0 - 2.0 * blend)) : (sqrt(base) * (2.0 * blend - 1.0) + 2.0 * base * (1.0 - blend)))
#define BlendColorDodgef(base, blend) 	(min(base / max(0.01,(1.0 - blend)), 1.0))
#define BlendColorBurnf(base, blend) 	(max((1.0 - ((1.0 - base) / (blend+0.01))), 0.0))
#define BlendVividLightf(base, blend) 	((blend < 0.5) ? BlendColorBurnf(base, (2.0 * blend)) : BlendColorDodgef(base, (2.0 * (blend - 0.5))))
#define BlendPinLightf(base, blend) 	((blend < 0.5) ? BlendDarkenf(base, (2.0 * blend)) : BlendLightenf(base, (2.0 *(blend - 0.5))))
#define BlendHardMixf(base, blend) 		((BlendVividLightf(base, blend) < 0.5) ? 0.0 : 1.0)
#define BlendReflectf(base, blend) 		((blend == 1.0) ? blend : min(base * base / (1.0 - blend), 1.0))
#define BlendLinearDodgef 			BlendAddf
#define BlendLinearBurnf 			BlendSubstractf
#define BlendLinearLightf(base, blend) 	(blend < 0.5 ? BlendLinearBurnf(base, (2.0 * blend)) : BlendLinearDodgef(base, (2.0 * (blend - 0.5))))


float3 _Blend(float3 base, float3 blend, int mode)
{
	if (mode == MODE_ADD)
		return BlendAddf(base,blend);
	if (mode == MODE_SUB)
		return BlendSubstractf(base,blend);
	if (mode == MODE_LIGHTEN)
		return BlendLightenf(base,blend);
	if (mode == MODE_DARKEN)
		return BlendDarkenf(base,blend);
	if (mode == MODE_LINEAR_LIGHT)
		return BlendLinearLightf(base,blend);
	if (mode == MODE_SCREEN)
		return BlendScreenf(base,blend);
	if (mode == MODE_OVERLAY)
		return BlendOverlayf(base,blend);
	if (mode == MODE_SOFT_LIGHT)
		return BlendSoftLightf(base,blend);
	if (mode == MODE_COLOR_DODGE)
		return BlendColorDodgef(base,blend);
	if (mode == MODE_COLOR_BURN)
		return BlendColorBurnf(base,blend);
	if (mode == MODE_VIVID_LIGHT)
		return BlendVividLightf(base,blend);
	if (mode == MODE_PIN_LIGHT)
		return BlendPinLightf(base,blend);
	if (mode == MODE_HARDMIX)
		return BlendHardMixf(base,blend);
	if (mode == MODE_REFLECT)
		return BlendReflectf(base,blend);
	if (mode == MODE_PRODUCT)
		return base * blend;
	if (mode == MODE_NORMAL)
		return blend;

	return float3(1,0,1);	// pink screen, this should never happen.
}

PSOut PSPhotoshop(PSIn IN, uniform int mode)
{
	PSOut OUT; OUT.c0 = 1;

	float3 colBase = g_texInput.Sample(SamplerLinearClamp, IN.Tex.xy).rgb;	
	float3 colBlend = Bitmap.Sample(SamplerLinearWrap, IN.Tex.zw).rgb;		
	float mask = Mask.Sample(SamplerLinearWrap, IN.Tex.zw).r;
	OUT.c0.rgb = lerp(colBase, _Blend(colBase, colBlend, mode), saturate(Opacity * mask + OpacityBias));	
	return OUT;
}

//--------------------------------------------------------------------------------------
#define PS_TECHNIQUE( name, mode) \
technique11 name < string args="Bitmap;Mask;Opacity;OpacityBias" UV_XFORM_ARGS ; > \
{ \
    pass p0  \
    { \
        SetVertexShader( CompileShader( vs_4_0, VS() ) );        \
        SetPixelShader( CompileShader( ps_4_0, PSPhotoshop(mode) ) ); \
		SetRasterizerState( CullNone ); \
		SetBlendState( DefaultBlending, float4(0,0,0,0), 0x0ffffffff ); \
		SetDepthStencilState( DisableDepth, 0 ); \
    } \
}


PS_TECHNIQUE(PSAdd, MODE_ADD)
PS_TECHNIQUE(PSSub, MODE_SUB)
PS_TECHNIQUE(PSLighten, MODE_LIGHTEN)
PS_TECHNIQUE(PSDarken, MODE_DARKEN)
PS_TECHNIQUE(PSLinearLight, MODE_LINEAR_LIGHT)
PS_TECHNIQUE(PSScreen, MODE_SCREEN)
PS_TECHNIQUE(PSOverlay, MODE_OVERLAY)
PS_TECHNIQUE(PSSoftLight, MODE_SOFT_LIGHT)
PS_TECHNIQUE(PSColorDodge, MODE_COLOR_DODGE)
PS_TECHNIQUE(PSColorBurn, MODE_COLOR_BURN)
PS_TECHNIQUE(PSVividLight, MODE_VIVID_LIGHT)
PS_TECHNIQUE(PSPinLight, MODE_PIN_LIGHT)
PS_TECHNIQUE(PSHardMix, MODE_HARDMIX)
PS_TECHNIQUE(PSReflect, MODE_REFLECT)
PS_TECHNIQUE(PSProduct, MODE_PRODUCT)
PS_TECHNIQUE(PSNormal, MODE_NORMAL)

//--------------------------------------------------------------------------------------

PSOut PSShowZ(PSIn IN)
{
	PSOut OUT; OUT.c0 = 1;

	float z = g_texDepth.Sample(SamplerPointClamp, IN.Tex.xy * g_depthUVScale.xy).r;

	float cameraFar				= g_cameraNearFar.y;
	float cameraNearFar			= g_cameraNearFar.z;
	float cameraFarMinusNear	= g_cameraNearFar.w;

	float viewSpaceZ = -(cameraNearFar / (z * cameraFarMinusNear - cameraFar));
	OUT.c0.rgb = viewSpaceZ / DepthRange;

	return OUT;
} 

technique11 ShowZ 
{
	pass p0 < string args = "DepthRange;"; >
	{ 
		SetVertexShader( CompileShader( vs_4_0, VS() ) );        
		SetPixelShader( CompileShader( ps_4_0, PSShowZ() ) );
		SetRasterizerState( CullNone );
		SetBlendState( DefaultBlending, float4(0,0,0,0), 0x0ffffffff );
		SetDepthStencilState( DisableDepth, 0 );
	}
}

//--------------------------------------------------------------------------------------
PSOut PSDof(PSIn IN)
{
	PSOut OUT; OUT.c0 = 1;

	float dx = 1;
	float dy = 1;

	float4 v[12]; 
	v[0]  = float4(-0.326212f * dx, -0.40581f * dy, 0.0f, 0.0f); 
	v[1]  = float4(-0.840144f * dx, -0.07358f * dy, 0.0f, 0.0f);  
	v[2]  = float4(-0.695914f * dx, 0.457137f * dy, 0.0f, 0.0f); 
	v[3]  = float4(-0.203345f * dx, 0.620716f * dy, 0.0f, 0.0f); 
	v[4]  = float4(0.96234f * dx, -0.194983f * dy, 0.0f, 0.0f); 
	v[5]  = float4(0.473434f * dx, -0.480026f * dy, 0.0f, 0.0f); 
	v[6]  = float4(0.519456f * dx, 0.767022f * dy, 0.0f, 0.0f); 
	v[7]  = float4(0.185461f * dx, -0.893124f * dy, 0.0f, 0.0f); 
	v[8]  = float4(0.507431f * dx, 0.064425f * dy, 0.0f, 0.0f); 
	v[9]  = float4(0.89642f * dx, 0.412458f * dy, 0.0f, 0.0f); 
	v[10] = float4(-0.32194f * dx, -0.932615f * dy, 0.0f, 0.0f); 
	v[11] = float4(-0.791559f * dx, -0.59771f * dy, 0.0f, 0.0f); 

	float z = g_texDepth.Sample(SamplerPointClamp, IN.Tex.xy * g_depthUVScale.xy).r;

	float cameraFar				= g_cameraNearFar.y;
	float cameraNearFar			= g_cameraNearFar.z;
	float cameraFarMinusNear	= g_cameraNearFar.w;
	float viewSpaceZ			= -(cameraNearFar / (z * cameraFarMinusNear - cameraFar));
	float zfactor				= saturate( (viewSpaceZ-DofNear) / (DofFar-DofNear) );

	float3 rgb = g_texInput.Sample(SamplerPointClamp, IN.Tex.xy).rgb;

	float3 blurFinal = rgb;
	float totalContribution = 1;
	float step = 12;	
	
	float pi = 3.141592f;
	float angleStep = (2.0f * pi) / step;


	for (uint i=0; i<(uint)step; i++)
	{
		float cs = cos(angleStep * i);
		float ss = sin(angleStep * i);
		float2 off = float2(ss,cs) * (1 + (i % 2));	

		float zSample = g_texDepth.Sample(SamplerPointClamp, IN.Tex.xy * g_depthUVScale.xy + off * g_texDepthInvSize.xy * zfactor * DofSize ).r;
		float3 colorSample =  g_texInput.Sample(SamplerLinearClamp, IN.Tex.xy + off * g_texInputInvSize.xy * zfactor * DofSize).rgb;
			
		if (zSample > z)
		{
			blurFinal += colorSample;
			totalContribution += 1;
		}
	}
	blurFinal /= totalContribution;

	OUT.c0.rgb	= blurFinal; // lerp( rgb, b0, zfactor );
	OUT.c0.a	= 1;
	
	return OUT;
}

technique11 DOF
{
	pass p0 
		< 
			string args = "DofNear;DofFar;DofSize;"; 
			string UseBlur = "true"; 
		>
	{
		SetVertexShader( CompileShader( vs_4_0, VS() ) );        
		SetPixelShader( CompileShader( ps_4_0, PSDof() ) );
		SetRasterizerState( CullNone );
		SetBlendState( DefaultBlending, float4(0,0,0,0), 0x0ffffffff );
		SetDepthStencilState( DisableDepth, 0 );
	}
}

///////////////////////////////////////////////////////////////////////////////////////////////////
Texture3D ColorGradingLookUp 
< 
	string ResourceType = "3DLut"; 
	string TextureType = "3DLut"; 
 	string UIName = "ColorGradingLookUp";
 	string Default = "lut_neutral.png";
>;

float3 doColorGrading(float3 rgb)
{
	rgb *= 15.0f / 16.0f;
	float texelOffset = 0.5f / 16.0f;
	rgb = ColorGradingLookUp.Sample(SamplerLinearClamp, rgb.xyz + texelOffset.xxx).rgb;
	return rgb;
}

PSOut PSColorGrading(PSIn IN)
{     
	PSOut OUT;
	float3 rgb = g_texInput.Sample(SamplerPointClamp, IN.Tex.xy).rgb;
	float opaMask = Mask.Sample(SamplerPointClamp, IN.Tex.xy).r;
	OUT.c0.rgb	= lerp( rgb, doColorGrading(rgb), Opacity * saturate(opaMask + OpacityBias) );
	OUT.c0.a	= 1;
	return OUT;
}

technique11 ColorGrading
 		< 
 			string args = "ColorGradingLookUp;Opacity;OpacityBias;Mask";
 		>
{
	pass p0 
	{
		SetVertexShader( CompileShader( vs_4_0, VS() ) );        
		SetPixelShader( CompileShader( ps_4_0, PSColorGrading() ) );
		SetRasterizerState( CullNone );
		SetBlendState( DefaultBlending, float4(0,0,0,0), 0x0ffffffff );
		SetDepthStencilState( DisableDepth, 0 );
	}
}

///////////////////////////////////////////////////////////////////////////////////////////////////
float FogStart		< float defaultValue = 0; >					= 0.0f;
float FogEnd		< float defaultValue = 10000; >				= 10000.0f;
float3 FogColor		< float3 defaultValue = float3(1,1,1); >	= float3(1,1,1);
float FogDensity	< float defaultValue = 0.25f; >				= 0.25f;

Texture2D SSAONoiseTexture;
float SSAOTextureSize	< float defaultValue = 64.0f; > = 64.0f;
float SSAOSampleRadius	< float defaultValue = 32.0f; > = 32.0f;
float SSAODepthScale	< float defaultValue = 0.01f; > = 0.01f;
float SSAOAngleBias		< float defaultValue = 0.2f; >	= 0.2f;
float SSAOIntensity		< float defaultValue = 1.0f; >	= 1.0f;

//-----------------------------------------------------------------------------
float3 getPosition(float2 uv)
{
	GPixel gpixel = FetchGBuffer(uv * g_gbufferUVScale.xy);
	return gpixel.WorldPos.xyz;
}

//-----------------------------------------------------------------------------
float doAmbientOcclusion(in float2 tcoord, in float2 uv, in float3 p, in float3 cnorm) 
{ 
	float ssaoScale		= SSAODepthScale;
	float ssaoBias		= SSAOAngleBias;
	float ssaoIntensity = SSAOIntensity;

	float3 diff = getPosition(tcoord + uv) - p; 
	const float3 v = normalize(diff); 
	const float d = length(diff)*ssaoScale; 
	return max(0.0,dot(cnorm,v)-ssaoBias)*(1.0/(1.0+d))*ssaoIntensity; 
} 

float2 getRandom(in float2 uv) 
{ 
	return normalize(SSAONoiseTexture.Sample(SamplerLinearWrap, g_texInputSize.xy * uv / SSAOTextureSize.xx).xy * 2.0f - 1.0f); 
} 

PSOut PSUber(PSIn IN)
{
	PSOut OUT;

	float3 rgb = g_texInput.Sample(SamplerPointClamp, IN.Tex.xy).rgb;
	
	float z = g_texDepth.Sample(SamplerPointClamp, IN.Tex.zw).r;
	float cameraFar				= g_cameraNearFar.y;
	float cameraNearFar			= g_cameraNearFar.z;
	float cameraFarMinusNear	= g_cameraNearFar.w;
	float viewSpaceZ = -(cameraNearFar / (z * cameraFarMinusNear - cameraFar));

	GPixel gpixel = FetchGBuffer(IN.Tex.zw);

	// ----- SSAO -----
	float3 worldPos = gpixel.WorldPos.xyz;
	float3 worldNorm = gpixel.WorldNorm.xyz;

	const float2 ssaoOffsets[4] = { float2(1,0), float2(-1,0), float2(0,1), float2(0,-1) }; 

	float ao	= 0.0f; 
	float rad	= SSAOSampleRadius / viewSpaceZ;
	float3 rnd	= float3(getRandom(IN.Tex.xy), 0.0f);
	int iterations = 4; 
	for (int j = 0; j < iterations; ++j) 
	{ 
		float2 coord1 = reflect( float3(ssaoOffsets[j],0), rnd ).xy * rad; 
		float2 coord2 = float2(coord1.x*0.707 - coord1.y*0.707, coord1.x*0.707 + coord1.y*0.707); 
		ao += doAmbientOcclusion(IN.Tex.xy,coord1*0.25, worldPos, worldNorm); 
		ao += doAmbientOcclusion(IN.Tex.xy,coord2*0.5,	worldPos, worldNorm); 
		ao += doAmbientOcclusion(IN.Tex.xy,coord1*0.75, worldPos, worldNorm); 
		ao += doAmbientOcclusion(IN.Tex.xy,coord2,		worldPos, worldNorm); 
	} 

	ao /=(float)iterations*4.0; 
	rgb *= (1.0f - ao);

	// ----- DepthFog -----
	float fogFactor = saturate( FogDensity * (viewSpaceZ - FogStart) / (FogEnd - FogStart) );
	rgb = lerp( rgb, FogColor, fogFactor); 

	// ----- ColorGrading -----
	rgb = doColorGrading(rgb);

	OUT.c0.rgb	= rgb;
	OUT.c0.a	= 1;

	return OUT;
}

technique11 Uber 
{
	pass p0 
		< 
			string args = "ColorGradingLookUp;FogStart;FogEnd;FogColor;FogDensity;SSAONoiseTexture;SSAOTextureSize;SSAOSampleRadius;SSAODepthScale;SSAOAngleBias;SSAOIntensity;";
		>
	{
		SetVertexShader( CompileShader( vs_4_0, VS() ) );        
		SetPixelShader( CompileShader( ps_4_0, PSUber() ) );
		SetRasterizerState( CullNone );
		SetBlendState( DefaultBlending, float4(0,0,0,0), 0x0ffffffff );
		SetDepthStencilState( DisableDepth, 0 );
	}
}

///////////////////////////////////////////////////////////////////////////////

float4 PSFade(PSIn IN) : SV_TARGET0
{
	float3 rgb = g_texInput.Sample(SamplerPointClamp, IN.Tex.xy).rgb;
	rgb = lerp( rgb, Color.rgb, Opacity.xxx );
	return float4(rgb, 1);
}

technique11 Fade
 		< 
 			string args = "Color;Opacity;";
 		>
{
	pass p0 
	{
		SetVertexShader( CompileShader( vs_4_0, VS() ) );        
		SetPixelShader( CompileShader( ps_4_0, PSFade() ) );
		SetRasterizerState( CullNone );
		SetBlendState( DefaultBlending, float4(0,0,0,0), 0x0ffffffff );
		SetDepthStencilState( DisableDepth, 0 );
	}
}

///////////////////////////////////////////////////////////////////////////////

float BloomMultiplier;

Texture2D	Dirt
<
	string UIName = "Dirt";
	string Default = "DefaultBlack.png";	
>;

float DirtMultiplier;

float4 PSApplyBloom(PSIn IN) : SV_TARGET0
{
	float3 dirt = Dirt.Sample(SamplerLinearClamp, IN.Tex.xy).rgb;
	float3 background = g_texInput.Sample(SamplerLinearClamp, IN.Tex.xy).rgb;
	float3 bloom = Bitmap.Sample(SamplerLinearClamp, IN.Tex.xy).rgb;	
	float3 bloom2 = Bitmap2.Sample(SamplerLinearClamp, IN.Tex.xy).rgb;
	float3 bloom3 = Bitmap3.Sample(SamplerLinearClamp, IN.Tex.xy).rgb;
	float3 bloom4 = Bitmap4.Sample(SamplerLinearClamp, IN.Tex.xy).rgb;

	bloom = max(bloom, bloom2);
	bloom = max(bloom, bloom3);
	bloom = max(bloom, bloom4);

	background = pow( max(0,background), 2.2f );
	bloom = pow( max(0,bloom * (BloomMultiplier + dirt * DirtMultiplier)), 2.2f );

	return float4( pow(background + bloom, 1.0f / 2.2f), 1);
//	return float4(background + bloom * (BloomMultiplier + dirt * DirtMultiplier), 1);
}

technique11 ApplyBloom
{
	pass p0 
	{
		SetVertexShader( CompileShader( vs_4_0, VS() ) );        
		SetPixelShader( CompileShader( ps_4_0, PSApplyBloom() ) );
		SetRasterizerState( CullNone );
		SetBlendState( DefaultBlending, float4(0,0,0,0), 0x0ffffffff );
		SetDepthStencilState( DisableDepth, 0 );
	}
}



///////////////////////////////////////////////////////////////////////////////

PSOut PSVignette(PSIn IN)
{     
	PSOut OUT;

	float2 texCoords = IN.Tex.xy - 0.5f;
	float vignette = 1.0f - saturate( pow(dot(texCoords, texCoords), Power) * Multiplier );

	float3 rgb  = g_texInput.Sample(SamplerPointClamp, IN.Tex.xy).rgb;
	OUT.c0.rgb	= rgb * vignette;
	OUT.c0.a	= 1.0f;

	return OUT;
}

technique11 Vignette
 		< 
 			string args = "Multiplier;Power;";
 		>
{
	pass p0 
	{
		SetVertexShader( CompileShader( vs_4_0, VS() ) );        
		SetPixelShader( CompileShader( ps_4_0, PSVignette() ) );
		SetRasterizerState( CullNone );
		SetBlendState( DefaultBlending, float4(0,0,0,0), 0x0ffffffff );
		SetDepthStencilState( DisableDepth, 0 );
	}
}


///////////////////////////////////////////////////////////////////////////////
PSOut PSAberrationRGB(PSIn IN)
{     
	PSOut OUT;

	float2 texCoords = IN.Tex.xy - 0.5f;
	float d = saturate( dot(texCoords, texCoords) ) * Multiplier + Bias;

	float2 v = normalize(float2(DirX, DirY)) * d * 0.01f;

	float red = g_texInput.Sample(SamplerPointClamp, IN.Tex.xy + v).r;
	float green = g_texInput.Sample(SamplerPointClamp, IN.Tex.xy).g;
	float blue = g_texInput.Sample(SamplerPointClamp, IN.Tex.xy - v).b;

	OUT.c0.rgb	= float3(red,green,blue);
	OUT.c0.a	= 1.0f;

	return OUT;
}

technique11 AberrationRGB
 		< 
 			string args = "Multiplier;Bias;DirX;DirY";
 		>
{
	pass p0 
	{
		SetVertexShader( CompileShader( vs_4_0, VS() ) );        
		SetPixelShader( CompileShader( ps_4_0, PSAberrationRGB() ) );
		SetRasterizerState( CullNone );
		SetBlendState( DefaultBlending, float4(0,0,0,0), 0x0ffffffff );
		SetDepthStencilState( DisableDepth, 0 );
	}
}


//-----------------------------------------------------------------------------
float Contrast
<	
	float defaultValue = 1.0f;  
	string UIName = "Contrast";
 	string Default = "1.0";
> = 1.0f;

float Saturation
<	
	float defaultValue = 1.0f;  
	string UIName = "Saturation";
 	string Default = "1.0";
> = 1.0f;

float Brightness
<	
	float defaultValue = 1.0f;  
	string UIName = "Brightness";
 	string Default = "1.0";
> = 1.0f;

float3 ComputeCSB(float3 color, float brt, float sat, float con)
{
	float3 LumCoeff			= float3(0.2125, 0.7154, 0.0721);		
	float3 brtColor			= color * brt;
	float intensity			= dot(brtColor, LumCoeff);
	float3 satColor			= lerp(intensity.xxx, brtColor, sat);
	float3 conColor			= lerp(float3(0.5,0.5,0.5), satColor, con);
	return conColor;
}

PSOut PSConstrastSaturationBrightness(PSIn IN)
{     
	PSOut OUT;

	float3 rgb  = g_texInput.Sample(SamplerPointClamp, IN.Tex.xy).rgb;
	OUT.c0.rgb	= lerp( rgb, ComputeCSB(rgb, Brightness, Saturation, Contrast), Opacity );
	OUT.c0.a	= 1;

	return OUT;
}

technique11 CSB
 		< 
		string args = "Contrast;Saturation;Brightness;Opacity";
 		>
{
	pass p0 
	{
		SetVertexShader( CompileShader( vs_4_0, VS() ) );        
		SetPixelShader( CompileShader( ps_4_0, PSConstrastSaturationBrightness() ) );
		SetRasterizerState( CullNone );
		SetBlendState( DefaultBlending, float4(0,0,0,0), 0x0ffffffff );
		SetDepthStencilState( DisableDepth, 0 );
	}
}

///////////////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
#define FXAA_EDGE_THRESHOLD_MIN (1.0f / 16.0f)
#define FXAA_EDGE_THRESHOLD		(1.0f / 4.0f)
// FXAA_EDGE_THRESHOLD - The minimum amount of local contrast required 
//                       to apply algorithm.
//                       1.0/3.0  - too little
//                       1.0/4.0  - good start
//                       1.0/8.0  - applies to more edges
//                       1.0/16.0 - overkill

#define FXAA_SEARCH_THRESHOLD	(1.0f / 4.0f)
#define FXAA_SEARCH_STEPS		4
#define FXAA_SUBPIX_CAP         (0.75f)
#define FXAA_SUBPIX_TRIM        (1.0/4.0)
#define FXAA_SUBPIX_TRIM_SCALE	(1.0/(1.0 - FXAA_SUBPIX_TRIM))

//-----------------------------------------------------------------------------
float luma(float3 rgb)
{
	return dot(rgb, float3(0.3f, 0.6f, 0.1f));
}

//-----------------------------------------------------------------------------
float4 PSFXAA( PSIn IN ) : SV_TARGET0
{ 

	float2 rcpFrame = g_texInputInvSize.xy;

	float2 offsetN = float2(0, -g_texInputInvSize.y);
	float2 offsetS = float2(0, +g_texInputInvSize.y);
	float2 offsetW = float2(-g_texInputInvSize.x, 0);
	float2 offsetE = float2(+g_texInputInvSize.x, 0);
	float2 offsetNW = float2(-g_texInputInvSize.x, -g_texInputInvSize.y);
	float2 offsetNE = float2(+g_texInputInvSize.x, -g_texInputInvSize.y);
	float2 offsetSW = float2(-g_texInputInvSize.x, +g_texInputInvSize.y);
	float2 offsetSE = float2(+g_texInputInvSize.x, +g_texInputInvSize.y);


 	float3 rgbM = g_texInput.Sample(SamplerPointClamp, IN.Tex.xy).rgb;
 	float3 rgbN = g_texInput.Sample(SamplerPointClamp, IN.Tex.xy + offsetN).rgb;
 	float3 rgbS = g_texInput.Sample(SamplerPointClamp, IN.Tex.xy + offsetS).rgb;
 	float3 rgbW = g_texInput.Sample(SamplerPointClamp, IN.Tex.xy + offsetW).rgb;
 	float3 rgbE = g_texInput.Sample(SamplerPointClamp, IN.Tex.xy + offsetE).rgb;
 	
 	float3 rgbNW = g_texInput.Sample(SamplerPointClamp, IN.Tex.xy + offsetNW).rgb;
 	float3 rgbNE = g_texInput.Sample(SamplerPointClamp, IN.Tex.xy + offsetNE).rgb;
 	float3 rgbSW = g_texInput.Sample(SamplerPointClamp, IN.Tex.xy + offsetSW).rgb;
 	float3 rgbSE = g_texInput.Sample(SamplerPointClamp, IN.Tex.xy + offsetSE).rgb;

    float lumaN = luma(rgbN);
    float lumaS = luma(rgbS);
    float lumaW = luma(rgbW);
    float lumaE = luma(rgbE);

	float lumaNW = luma(rgbNW);
    float lumaNE = luma(rgbNE);
	float lumaSW = luma(rgbSW);
    float lumaSE = luma(rgbSE);

	float lumaM = luma(rgbM);

    float rangeMin = min(lumaM, min(min(lumaN, lumaW), min(lumaS, lumaE)));
    float rangeMax = max(lumaM, max(max(lumaN, lumaW), max(lumaS, lumaE)));
    float range = rangeMax - rangeMin;

    if(range < max(FXAA_EDGE_THRESHOLD_MIN, rangeMax * FXAA_EDGE_THRESHOLD)) 
	{
		return float4( rgbM, 1.0f );
	}

    float3 rgbL = (rgbN + rgbW + rgbM + rgbE + rgbS + rgbNE + rgbNW + rgbSW + rgbSE) / 9.0f;
    float lumaL = (lumaN + lumaW + lumaE + lumaS) / 4.0f;
    float rangeL = abs(lumaL - lumaM);

 	float blendL = max(0.0, (rangeL / range) - FXAA_SUBPIX_TRIM) * FXAA_SUBPIX_TRIM_SCALE; 
 	blendL = min(FXAA_SUBPIX_CAP, blendL);

	float edgeVert = 
        abs((0.25 * lumaNW) + (-0.5 * lumaN) + (0.25 * lumaNE)) +
        abs((0.50 * lumaW ) + (-1.0 * lumaM) + (0.50 * lumaE )) +
        abs((0.25 * lumaSW) + (-0.5 * lumaS) + (0.25 * lumaSE));
    float edgeHorz = 
        abs((0.25 * lumaNW) + (-0.5 * lumaW) + (0.25 * lumaSW)) +
        abs((0.50 * lumaN ) + (-1.0 * lumaM) + (0.50 * lumaS )) +
        abs((0.25 * lumaNE) + (-0.5 * lumaE) + (0.25 * lumaSE));
    bool horzSpan = edgeHorz >= edgeVert;

    float lengthSign = horzSpan ? -rcpFrame.y : -rcpFrame.x;
    if(!horzSpan) 
	{
		lumaN = lumaW;
		lumaS = lumaE;
	}
    float gradientN = abs(lumaN - lumaM);
    float gradientS = abs(lumaS - lumaM);
    lumaN = (lumaN + lumaM) * 0.5;
    lumaS = (lumaS + lumaM) * 0.5;

	bool pairN = gradientN >= gradientS;
	if(!pairN) 
	{
		lumaN = lumaS;
		gradientN = gradientS;
		lengthSign *= -1.0;
	}

    float2 posN;
    posN.x = IN.Tex.xy.x + (horzSpan ? 0.0 : lengthSign * 0.5);
    posN.y = IN.Tex.xy.y + (horzSpan ? lengthSign * 0.5 : 0.0);

    gradientN *= FXAA_SEARCH_THRESHOLD;

	float2 posP = posN;
    float2 offNP = horzSpan ? float2(rcpFrame.x, 0.0) : float2(0.0f, rcpFrame.y); 
    float lumaEndN = lumaN;
    float lumaEndP = lumaN;
    bool doneN = false;
    bool doneP = false;

    posN -= offNP;
    posP += offNP;

	[unroll]
	for(int i = 0; i < FXAA_SEARCH_STEPS; i++) 
	{
		if(!doneN) 
			lumaEndN = luma(g_texInput.Sample(SamplerLinearClamp, posN.xy).rgb);

		if(!doneP) 
			lumaEndP = luma(g_texInput.Sample(SamplerLinearClamp, posP.xy).rgb);

        doneN = doneN || (abs(lumaEndN - lumaN) >= gradientN);
        doneP = doneP || (abs(lumaEndP - lumaN) >= gradientN);

        if(doneN && doneP) 
			break;
        
		if(!doneN) 
			posN -= offNP;
        
		if(!doneP) 
			posP += offNP;
	}

	float2 pos = IN.Tex.xy;

	float dstN = horzSpan ? pos.x - posN.x : pos.y - posN.y;
    float dstP = horzSpan ? posP.x - pos.x : posP.y - pos.y;
    bool directionN = dstN < dstP;
    
    lumaEndN = directionN ? lumaEndN : lumaEndP;

	if(((lumaM - lumaN) < 0.0) == ((lumaEndN - lumaN) < 0.0)) 
        lengthSign = 0.0;

	float spanLength = (dstP + dstN);
    dstN = directionN ? dstN : dstP;
    float subPixelOffset = (0.5 + (dstN * (-1.0/spanLength))) * lengthSign;
	
    float3 rgbF = g_texInput.Sample(SamplerLinearClamp, float2(
		pos.x + (horzSpan ? 0.0 : subPixelOffset),
        pos.y + (horzSpan ? subPixelOffset : 0.0))).rgb;

	float3 rgb = lerp(rgbF, rgbL, blendL);

	return float4(rgb, 1);
}


//-----------------------------------------------------------------------------
//
// Techniques
//
//-----------------------------------------------------------------------------
technique11 FXAA
{
	pass p0 
	{
		SetVertexShader( CompileShader( vs_4_0, VS() ) );        
		SetPixelShader( CompileShader( ps_4_0, PSFXAA() ) );
		SetRasterizerState( CullNone );
		SetBlendState( DefaultBlending, float4(0,0,0,0), 0x0ffffffff );
		SetDepthStencilState( DisableDepth, 0 );
	}
}