#include "Constant.fx"
#include "ConstantBuffer.fx"
#include "HDR.fx"
#include "Deferred.fx"

struct VSIn
{
	float4 Pos		: POSITION;
	float2 Tex		: TEXCOORD0;
};

struct VSOut
{
	float4 Pos			: SV_POSITION;
	float4 Tex			: TEXCOORD0;
};

struct PSIn
{
	float4 Pos			: SV_POSITION;
	float4 Tex			: TEXCOORD0;
};    

struct PSOut  
{
	float4 c0 : SV_TARGET0;
};

Texture2D g_texSpotProj;
Texture2D g_texShadow;
TextureCube g_texCubeShadow;
Texture2D g_texShadowViewSpaceDepth;

float4	g_lightPos = float4(-1000,-1000,-1000,0);
float4	g_lightRange;
float4	g_lightColor;
float4	g_cones;
float4  g_spotDir; 
float4x4 g_spotViewProj;
float4x4 g_invSpotViewProj;
float4  g_UVScale = float4(1,1,1,1);


#define BLOCKER_SEARCH_NUM_SAMPLES 16 
#define PCF_NUM_SAMPLES		16 
//#define NEAR_PLANE			9.5 
// #define LIGHT_WORLD_SIZE	.5 
// #define LIGHT_FRUSTUM_WIDTH 3.75 
// Assuming that LIGHT_FRUSTUM_WIDTH == LIGHT_FRUSTUM_HEIGHT 
// #define LIGHT_SIZE_UV (LIGHT_WORLD_SIZE / LIGHT_FRUSTUM_WIDTH) 
//#define LIGHT_SIZE_UV 0.1333f
#define LIGHT_SIZE_UV (0.2f)
float NEAR_PLANE;

cbuffer POISSON_DISKS 
{ 
	float2 poissonDisk[16] = { 
		float2( -0.94201624, -0.39906216 ), 
		float2( 0.94558609, -0.76890725 ), 
		float2( -0.094184101, -0.92938870 ), 
		float2( 0.34495938, 0.29387760 ), 
		float2( -0.91588581, 0.45771432 ), 
		float2( -0.81544232, -0.87912464 ), 
		float2( -0.38277543, 0.27676845 ), 
		float2( 0.97484398, 0.75648379 ), 
		float2( 0.44323325, -0.97511554 ), 
		float2( 0.53742981, -0.47373420 ), 
		float2( -0.26496911, -0.41893023 ), 
		float2( 0.79197514, 0.19090188 ), 
		float2( -0.24188840, 0.99706507 ), 
		float2( -0.81409955, 0.91437590 ), 
		float2( 0.19984126, 0.78641367 ), 
		float2( 0.14383161, -0.14100790 ) 
	}; 
}; 

float PenumbraSize(float zReceiver, float zBlocker) //Parallel plane estimation 
{ 
	return (zReceiver - zBlocker) / zBlocker; 
} 

void FindBlocker(out float avgBlockerDepth,  
				 out float numBlockers, 
				 float2 uv, float zReceiver ) 
{ 
	//This uses similar triangles to compute what  
	//area of the shadow map we should search 
	float searchWidth = LIGHT_SIZE_UV * (zReceiver - NEAR_PLANE) / zReceiver; 

	float blockerSum = 0; 
	numBlockers = 0; 

	for( int i = 0; i < BLOCKER_SEARCH_NUM_SAMPLES; ++i ) 
	{ 
		float shadowMapDepth = g_texShadowViewSpaceDepth.SampleLevel(SamplerPointClamp, uv.xy + poissonDisk[i].xy * searchWidth.xx, 0).x; 
		if ( shadowMapDepth < zReceiver ) 
		{
			blockerSum += shadowMapDepth; 
			numBlockers++; 
		} 
	} 

	avgBlockerDepth = blockerSum / numBlockers; 
} 

float PCF_Filter( float2 uv, float zReceiver, float filterRadiusUV ) 
{ 
	float sum = 0.0f; 
	for ( int i = 0; i < PCF_NUM_SAMPLES; ++i ) 
	{ 
		float2 offset = poissonDisk[i] * filterRadiusUV; 
		sum += g_texShadowViewSpaceDepth.SampleCmpLevelZero(SamplerShadow, uv + offset, zReceiver); 		
		sum += g_texShadowViewSpaceDepth.SampleCmpLevelZero(SamplerShadow, uv + offset.yx, zReceiver); 		
	} 
	return sum / (PCF_NUM_SAMPLES * 2);
} 

float PCSS ( Texture2D shadowMapTex, float3 coords  ) 
{ 
	float2 uv = coords.xy; 
	float zReceiver = coords.z; // Assumed to be eye-space z in this code 

	// STEP 1: blocker search 
	float avgBlockerDepth = 0; 
	float numBlockers = 0; 
	FindBlocker( avgBlockerDepth, numBlockers, uv, zReceiver ); 

 	if( numBlockers < 1 )
		return 1.0f; 

	// STEP 2: penumbra size 
	float penumbraRatio		= PenumbraSize(zReceiver, avgBlockerDepth);     
	float filterRadiusUV	= penumbraRatio * LIGHT_SIZE_UV * NEAR_PLANE / coords.z; 

	// STEP 3: filtering 
	return PCF_Filter( uv, zReceiver, filterRadiusUV ); 
} 



//--------------------------------------------------------------------------------------
VSOut VS(VSIn IN)
{
	VSOut OUT;

	OUT.Pos	= float4(IN.Pos.xyz,1);
	OUT.Tex	= IN.Tex.xyxy * g_UVScale;

	return OUT;
}

//--------------------------------------------------------------------------------------
float3 ComputeLighting(GPixel g, bool isSpot, bool castShadow)
{
	bool isOmni = (!isSpot);

	float4 worldPos		= g.WorldPos;	// g_texGBuffer.Sample(SamplerPointClamp, float3(IN.Tex.xy,0) );
	float3 worldNorm	= g.WorldNorm;	// g_texGBuffer.Sample(SamplerPointClamp, float3(IN.Tex.xy,1) );
	float4 Kd			= g.Kd;			// pow(g_texGBuffer.Sample(SamplerPointClamp, float3(IN.Tex.xy,2) ), 2.2f);
	float4 Ks			= g.Ks;			// g_texGBuffer.Sample(SamplerPointClamp, float3(IN.Tex.xy,3) );
	float Ksss			= g.Ksss;




	
	float3 lightVec = g_lightPos.xyz - worldPos.xyz;
	float3 lightVecNormalized = normalize(lightVec);
	float3 eyeVec		= normalize(EyePos.xyz - worldPos.xyz);
	float3 halfVec		= normalize( (lightVecNormalized+eyeVec) );

	float ndl	= saturate(dot( worldNorm.xyz, lightVecNormalized ));
	float ndh	= saturate(dot( worldNorm.xyz, halfVec ));

	float attn	= 1 - saturate((distance(worldPos.xyz,g_lightPos.xyz) - g_lightRange.x) * g_lightRange.z);

	float3 lightColor = g_lightColor.rgb;
	float shadow = 1;
	if (isSpot)
	{		

		float dirContribution = dot(-g_spotDir.xyz, lightVecNormalized.xyz);
		lightColor *= 1.0f - saturate( (dirContribution - g_cones.x ) * g_cones.z );

		float4 uvp = mul( float4(worldPos.xyz,1), g_spotViewProj ); 
		float bias = distance(g_lightPos.xyz, worldPos.xyz) * 0.0001f;
		float4 projTex = float4(uvp.xyz,-bias) / uvp.w;	
		projTex.xy = projTex.xy * float2(0.5f,-0.5f) + 0.5f;

		if (castShadow)
		{
			//lightColor *= g_texShadow.SampleCmpLevelZero(SamplerShadow, projTex.xy, projTex.z ); 
			shadow *= PCSS(g_texShadowViewSpaceDepth, float3(projTex.xy,uvp.z) );
		}

		lightColor *= g_texSpotProj.SampleLevel(SamplerLinearWrap, projTex.xy, 0).rgb;	// ddx/ddy is wrong... force the lod to 0
	}

	if (isOmni && castShadow)
	{
		float distanceToShadedPixel = length(lightVec);
		float distanceToOccluder = g_texCubeShadow.SampleLevel( SamplerPointWrap, -lightVec.xzy, 0 ).x;
		if (distanceToOccluder < distanceToShadedPixel)
			shadow = 0.0f;
	}

	float3 lrgb = lightColor * Kd.rgb * attn;
	float3 diff = ndl * lrgb;

	float sssNdl = saturate( dot(eyeVec, -lightVecNormalized ) );
	diff += pow(sssNdl, 4) * lrgb * Ksss;

	float3 spec = pow( ndh, max(1.0f,Ks.a) ) * ((8 + Ks.a) / (8 * 3.141592f)) * lightColor * Ks.rgb * attn * saturate(ndl * 4.0f);
	float3 rgb = (diff + spec) * shadow;

	return rgb;
}


//--------------------------------------------------------------------------------------
PSOut PS(PSIn IN, uniform bool isSpot, uniform bool castShadow)
{
	PSOut OUT;
	OUT.c0 = 1;

	GPixel g	= FetchGBuffer(IN.Tex.xy);
	float3 rgb	= ComputeLighting(g, isSpot, castShadow);
	OUT.c0		= float4( rgb ,1);	

	return OUT;
}

//--------------------------------------------------------------------------------------
PSOut PSSelfIllum(VSOut IN)
{
	PSOut OUT;
	float4 Ksi = g_texGBuffer.Sample(SamplerPointClamp, float3(IN.Tex.xy,4) );
	OUT.c0 = Ksi;	
	return OUT;
}

//--------------------------------------------------------------------------------------
struct VSOmniIn
{
	float4 Pos		: POSITION;
};

struct VSOmniOut
{
	float4 Pos		: SV_POSITION;
	float4 Tex		: TEXCOORD0;
};

//--------------------------------------------------------------------------------------
VSOmniOut VSOmni(VSOmniIn IN)
{
	VSOmniOut OUT;

	OUT.Pos		= mul( IN.Pos, WorldViewProj );
	OUT.Pos.z	= min( OUT.Pos.z, OUT.Pos.w ) ;
	OUT.Tex		= OUT.Pos;

	return OUT;
}

//--------------------------------------------------------------------------------------
float4 PSOmniBase(VSOmniOut IN, bool shadow) 
{
	float2 uv = (IN.Tex.xy/IN.Tex.w) * float2(0.5f,-0.5f) + 0.5f;
	GPixel g = FetchGBuffer(uv);

	float2 screenXY	= (uv * 2.0f - 1.0f) * float2(1.0f, -1.0f);
	float2 proj		= float2(1.0f/Proj[0].x, 1.0f/Proj[1].y);

 	float cameraFar				= CameraNearFar.y;
 	float cameraNearFar			= CameraNearFar.z;
 	float cameraFarMinusNear	= CameraNearFar.w;
 
	float z = g_texDepthBuffer.Sample(SamplerPointClamp, uv).r;
 	float viewSpaceZ = -(cameraNearFar / (z * cameraFarMinusNear - cameraFar));
	float3 viewDir = float3(-screenXY * proj, 1.0f);
	viewDir = mul(viewDir, (float3x3)InvView); 

	float3 eyeToPos = viewDir.xyz * (-viewSpaceZ);
 	float4 WorldPos = float4(EyePos.xyz + eyeToPos, 1.0f);
	g.WorldPos = WorldPos;

	float3 rgb = ComputeLighting(g, false, shadow);

	return float4(rgb, 1.0f);
}

//--------------------------------------------------------------------------------------
float4 PSOmni(VSOmniOut IN) : SV_TARGET0
{
	return PSOmniBase(IN, false);	
}

//--------------------------------------------------------------------------------------
float4 PSOmniShadow(VSOmniOut IN) : SV_TARGET0
{
	return PSOmniBase(IN, true);	
}

//--------------------------------------------------------------------------------------
VSOmniOut VSSpot(VSOmniIn IN)
{
	VSOmniOut OUT;

	float4 localPos	= mul( IN.Pos, g_invSpotViewProj );
	localPos.xyz /= localPos.w;
	localPos.w = 1;
	OUT.Pos		= mul( localPos, WorldViewProj );
	OUT.Pos.z	= min( OUT.Pos.z, OUT.Pos.w ) ;
	OUT.Tex		= OUT.Pos;

	return OUT;
}

//--------------------------------------------------------------------------------------
float4 PSSpot(VSOmniOut IN) : SV_TARGET0
{
	float2 uv = (IN.Tex.xy/IN.Tex.w) * float2(0.5f,-0.5f) + 0.5f;
	GPixel g = FetchGBuffer(uv);
	float3 rgb = ComputeLighting(g, true, false);
	return float4(rgb, 1.0f); //  + float4(0.1,0,0,0);
}

//--------------------------------------------------------------------------------------
float4 PSSpotShadow(VSOmniOut IN) : SV_TARGET0
{
	float2 uv = (IN.Tex.xy/IN.Tex.w) * float2(0.5f,-0.5f) + 0.5f;
	GPixel g = FetchGBuffer(uv);
	float3 rgb = ComputeLighting(g, true, true);
	return float4(rgb, 1.0f); //  + float4(0.1,0,0,0);
}



//--------------------------------------------------------------------------------------
technique11 techSelfIllum
{
	pass p0
	{
		SetVertexShader( CompileShader( vs_4_0, VS() ) );        
		SetPixelShader( CompileShader( ps_4_0, PSSelfIllum() ) );
		SetRasterizerState( CullNone );
		SetBlendState( DefaultBlending, float4(0,0,0,0), 0x0ffffffff );
		SetDepthStencilState( DisableDepth, 0 );
	}
}

//--------------------------------------------------------------------------------------
technique11 TechOmni
{
	pass p0
	{
		SetVertexShader( CompileShader( vs_4_0, VSOmni() ) );        
		SetPixelShader( CompileShader( ps_4_0, PSOmni() ) );
		SetRasterizerState( CullBack );
		SetBlendState( AdditivBlending, float4(0,0,0,0), 0x0ffffffff );
		SetDepthStencilState( DisableDepth, 0 );
	}
}

//--------------------------------------------------------------------------------------
technique11 TechOmniShadow
{
	pass p0
	{
		SetVertexShader( CompileShader( vs_4_0, VSOmni() ) );        
		SetPixelShader( CompileShader( ps_4_0, PSOmniShadow() ) );
		SetRasterizerState( CullBack );
		SetBlendState( AdditivBlending, float4(0,0,0,0), 0x0ffffffff );
		SetDepthStencilState( DisableDepth, 0 );
	}
}

//--------------------------------------------------------------------------------------
technique11 TechSpot
{
	pass p0
	{
		SetVertexShader( CompileShader( vs_4_0, VSSpot() ) );        
		SetPixelShader( CompileShader( ps_4_0, PSSpot() ) );
		SetRasterizerState( CullBack );
		SetBlendState( AdditivBlending, float4(0,0,0,0), 0x0ffffffff );
		SetDepthStencilState( DisableDepth, 0 );
	}
}

//--------------------------------------------------------------------------------------
technique11 TechSpotShadow
{
	pass p0
	{
		SetVertexShader( CompileShader( vs_4_0, VSSpot() ) );        
		SetPixelShader( CompileShader( ps_4_0, PSSpotShadow() ) );
		SetRasterizerState( CullBack );
		SetBlendState( AdditivBlending, float4(0,0,0,0), 0x0ffffffff );
		SetDepthStencilState( DisableDepth, 0 );
	}
}