// 02.10.2011
#include "Constant.fx"
#include "BokehCommon.fx"

//--------------------------------------------------------------------------------------

struct BokehVSOutput
{
	float3 Position		: POSITION;
	float2 Size			: SIZE;
	float3 Color		: COLOR;
	float Depth : DEPTH;
};

struct BokehGSOutput
{
	float4 Position		: SV_Position;
	float2 Tex			: TEXCOORD;
	float3 Color		: COLOR;	
	float Depth : DEPTH;
};

struct VSIn
{
    float3 Pos			: POSITION;
    float2 Tex			: TEXCOORD;
};

struct VSOut
{
	float4 Pos			: SV_POSITION;
	float2 Tex			: TEXCOORD0;
};


//--------------------------------------------------------------------------------------
SamplerState PointSampler : register(s0);
SamplerState LinearSampler : register(s1);

AppendStructuredBuffer<BokehPoint> BokehPointBufferWrite : register(u1);
Texture2D LightedTexture : register(t0);
Texture2D DephtBlurTexture : register(t1);

StructuredBuffer<BokehPoint> BokehPointBufferRead : register(t0);

Texture2D BokehTexture : register(t0);


//--------------------------------------------------------------------------------------
VSOut VSGenerateBokeh(VSIn IN)
{
	VSOut OUT;

	OUT.Pos	= float4(IN.Pos.xyz,1);
    OUT.Tex	= IN.Tex.xy;

	return OUT;
}

//--------------------------------------------------------------------------------------
float4 PSGenerateBokeh(VSOut IN) : SV_TARGET0
{
	float2 centerCoord = IN.Tex;

	// Start with center sample color
	float3 centerColor = LightedTexture.Sample(PointSampler, centerCoord).xyz;
	float3 colorSum = centerColor;
	float totalContribution = 1.0f;

	// Sample the depth and blur at the center
	float2 centerDepthBlur	= DephtBlurTexture.Sample(PointSampler, centerCoord).xy;
	float centerDepth		= centerDepthBlur.x;
	float centerBlur		= centerDepthBlur.y;

	// We're going to calculate the average of the current 5x5 block of texels
	// by taking 9 bilinear samples
	const uint NumSamples = 9;
	const float2 SamplePoints[NumSamples] =
	{
		float2(-1.5f, -1.5f), float2(0.5f, -1.5f), float2(1.5f, -1.5f),
		float2(-1.5f, 0.5f), float2(0.5f, 0.5f), float2(1.5f, 0.5f),
		float2(-1.5f, 1.5f), float2(0.5f, 1.5f), float2(1.5f, 1.5f)
	};

	float3 averageColor = 0.0f;
	for(uint i = 0; i < NumSamples; ++i)
	{
		float2 sampleCoord = centerCoord + SamplePoints[i] * OneOverInputTextureSize.xy;
		float3 sample = LightedTexture.Sample(LinearSampler, sampleCoord).rgb;

		averageColor += sample;
	}
	averageColor /= NumSamples;

	// Calculate the difference between the current texel and the average
	float averageBrightness = dot(averageColor, 1.0f);
	float centerBrightness = dot(centerColor, 1.0f);
	float brightnessDiff = max(centerBrightness - averageBrightness, 0.0f);

	// If the texel is brighter than its neighbors and the blur amount
	// is above our threshold, then push the pixel position/color/blur amount into
	// our AppendStructuredBuffer containing bokeh points
	[branch]
	if(brightnessDiff >= BokehBrightnessThreshold && centerBlur > BokehBlurThreshold)
	{
		BokehPoint bPoint;		
		bPoint.Position = float3(IN.Pos.xy * OneOverInputTextureSize.xy, 0.0f/*centerDepth*/);
		bPoint.Color	= centerColor;
		bPoint.Blur		= centerBlur;

		BokehPointBufferWrite.Append(bPoint);
	}

	clip(-1);

	return float4(0,0,0,0);
}

//--------------------------------------------------------------------------------------
BokehVSOutput VSRenderBokeh(in uint VertexID : SV_VertexID)
{
	BokehVSOutput OUT;	

	BokehPoint bPoint	= BokehPointBufferRead[VertexID];
	OUT.Position		= bPoint.Position;
 	OUT.Position.xy		= bPoint.Position.xy * 2 - 1;
 	OUT.Position.y		*= -1;
 	// OUT.Color			= bPoint.Color; 	
 	OUT.Size			= bPoint.Blur * BokehMaxSize.xy; 
	OUT.Depth			= bPoint.Position.z;

	// Scale the color to conserve energy, by comparing new area vs. old area
	float cocRadius = bPoint.Blur * BokehMaxSize.z * 0.45f;
	float cocArea = cocRadius * cocRadius * 3.14159f;
	float falloff = pow(saturate(1.0f / cocArea), BokehFalloff);
	OUT.Color = bPoint.Color * falloff;

	return OUT;
}

//--------------------------------------------------------------------------------------
[maxvertexcount(4)]
void GSRenderBokeh(point BokehVSOutput input[1], inout TriangleStream<BokehGSOutput> SpriteStream)
{
	BokehGSOutput output;

	float2 Offsets[4] =
	{
		float2(-1, 1),
		float2(1, 1),
		float2(-1, -1),
		float2(1, -1)
	};

	float2 TexCoords[4] =
	{
		float2(0, 0),
		float2(1, 0),
		float2(0, 1),
		float2(1, 1)
	};

	// Emit 4 new verts, and 2 new triangles
	[unroll]
	for (int i = 0; i < 4; i++)
	{		
		output.Position = float4(input[0].Position.xy, 1.0f, 1.0f);
		output.Position.xy += Offsets[i].xy * input[0].Size.xy;
		
		output.Tex		= TexCoords[i];
		output.Color	= input[0].Color;
		output.Depth	= input[0].Depth;

		SpriteStream.Append(output);
	}

	SpriteStream.RestartStrip();
}

//--------------------------------------------------------------------------------------
float4 PSRenderBokeh(in BokehGSOutput input) : SV_Target0
{
	float3 sample = BokehTexture.Sample(LinearSampler, input.Tex).rgb * input.Color.rgb;
	return float4(sample, 1);
}

//--------------------------------------------------------------------------------------
technique11 GenerateBokeh
{
	pass p0
	{
	    SetVertexShader( CompileShader( vs_5_0, VSGenerateBokeh() ) );
        SetPixelShader( CompileShader( ps_5_0, PSGenerateBokeh() ) );
		SetRasterizerState( CullNone );
 		SetBlendState( DefaultBlending, float4(0,0,0,0), 0x0ffffffff );
 		SetDepthStencilState( DisableDepth, 0 );
	}
}

//--------------------------------------------------------------------------------------
technique11 RenderBokeh
{
	pass p0
	{
        SetVertexShader( CompileShader( vs_5_0, VSRenderBokeh() ) );        
        SetGeometryShader( CompileShader( gs_5_0, GSRenderBokeh() ) );
        SetPixelShader( CompileShader( ps_5_0, PSRenderBokeh() ) );
		SetRasterizerState( CullNone );
		SetBlendState( AdditivBlending, float4(0,0,0,0), 0x0ffffffff );
		SetDepthStencilState( DisableDepth, 0 );
	}
}

//--------------------------------------------------------------------------------------
