/**
 * Ecran/fentre d'affichage du temps de course du joueur,
 * et des meilleurs temps pour le circuit.
 *
 * S'affiche une fois la course finie, en superposition par rapport  la visu du jeu.
 *
 * Elle est dcoupe en deux phases, avec une boucle principale pour chaque phase :
 *  - affichage des meilleurs temps : boucle displayLoop()
 *  - choix de la suite : boucle choiceLoop()
 */
 
 
 /**
  * Cration de la fentre d'affichage des meilleurs temps.
  *
  *  - application : l'application (RacingGameApp)  qui rendre la main  l'issue
  *  - container : l'objet DIV qui contiendra les lments DHTML crs
  *  - fader : le fader sur le container (tat In, i.e. affichage normal, en entre)
  *  - imageFx : l'outil pour effectuer les oprations graphiques (texte dtour)
  */
function TimingWindow(application, container, fader, imageFx)
{
	this.application = application;
	this.container = container;
	this.fader = fader;
	this.imageFx = imageFx;
	
	this.width = 430;
	this.height = 190;
	this.transitionLineCount = 19; // nombre de lignes de persiennes
	this.transitionLineHeight = this.height / this.transitionLineCount;
	this.bestTimesCount = 6;
	
	this.respawnLayerCount = 10;
	

	this.bestTimesShown = false;
}

TimingWindow.prototype =  {


	/**
	 * Initialization / remise  zro des paramtres 
	 *  - road : le circuit juste jou
	 *  - raceTime : le temps de course  afficher (en 1/100e de seconde)
	 */
	initialize : function(road, raceResult) {
		this.userAcknowledged = true;
		this.timer = 0;
		this.raceResult = raceResult;
		this.raceTime = raceResult.raceTime;
		this.menuActive = false;
		
		this.road = road;
		this.bestTimesShown = false;
	},
	
	/**
	 * Renvoie l'option slectionne dans le menu, ou zro s'il n'a pas t cr
	 */
	getSelectedOption : function() {
		if (this.menuDriver) {
			return this.menuDriver.getSelectedOption();
		}
		return 0;
	},
	
	/**
	 * Rgle l'affichage des options du menu final en fonction de si le joueur
	 * a dverrouill le circuit suivant ou non.
	 *  - qualified : true si le joueur vient de qualifier (conditionne l'affichage d'un message et l'option "course suivante")
	 * 
	 * A n'appeler qu'aprs un launchScreen() (le menuDriver doit avoir t initialis)
	 */
	setQualified : function(qualified) {
		this.menuDriver.setOptionAvailable(0, qualified);
		this.menuDriver.setSelectedOption(qualified ? 0 : 1);
	},
 
	/**
	 * Cration et initialisation des lments d'affichage (DHTML)
	 * Branchement des gestionnaires d'vnements clavier
	 *
	 * Prrequis : fader In en entre (visu normale), on n'y touche pas
	 *
	 * - road : le circuit juste jou
	 * - raceResult : la description du rsultat de course
	 */
	launchScreen : function(road, raceResult) {
	
		// panel parent des div "persiennes", pendant l'affichage de celles-ci 
		this.transitionMaster = document.createElement('div');
		this.transitionMaster.setAttribute("class", "timing_master");
		this.container.appendChild(this.transitionMaster);

		// panel parent de l'affichage des temps, une fois les persiennes rassembles
		// en deux niveaux : le premier a une opacit  1 et contient les images des voitures,
		// le second a une opacit  0.5 et contient tout le reste. On procde ainsi car le rendu des
		// voitures avec 50% de transparence n'tait pas esthtique
		this.opaqueMaster = document.createElement('div');
		this.opaqueMaster.setAttribute("class", "timing_master");
		this.container.appendChild(this.opaqueMaster);
		this.permanentMaster = document.createElement('div');
		this.permanentMaster.setAttribute("class", "timing_display");
		this.opaqueMaster.appendChild(this.permanentMaster);

		this.lapRecordMessage = this.imageFx.createOutlinedTextPanel(this.permanentMaster, "timing_new_record", "loader_text", "white");
		this.illustrationLegend = this.imageFx.createOutlinedTextPanel(this.permanentMaster, "timing_legend", "loader_text", "white");
		
			
		this.illustration = document.createElement('img');
		this.illustration.setAttribute("class", "timing_illustration");
		this.opaqueMaster.appendChild(this.illustration);
		
		// animation en attente de la rponse Ajax
		this.hourglassAnimation = document.createElement('div');
		this.hourglassAnimation.setAttribute("class", "timing_hourglass");
		this.hourglassAnimation.appendChild(document.createTextNode(''));
		this.permanentMaster.appendChild(this.hourglassAnimation);

		// lignes de persiennes
		this.transitionLines = new Array(this.transitionLineCount);
		for (var line = 0 ; line < this.transitionLineCount ; ++line)
		{
			var currentLine = document.createElement('div');
			currentLine.setAttribute("class", "timing_transition_line");
			currentLine.style.top=(line*this.transitionLineHeight)+"px";
			this.transitionMaster.appendChild(currentLine);
			this.transitionLines[line] = currentLine;
		}
		
		// panel parent des "lames de verre", pendant l'affichage de celles-ci
		this.respawnMaster = document.createElement('div');
		this.respawnMaster.setAttribute("class", "timing_master");
		this.container.appendChild(this.respawnMaster);

		// "lames de verre"
		this.respawnLayers = new Array(this.respawnLayerCount);
		for (var index = 0 ; index < this.respawnLayerCount ; ++index)
		{
			var currentLayer = document.createElement('div');
			currentLayer.setAttribute("class", "timing_respawn_block");
			this.respawnMaster.appendChild(currentLayer);
			this.respawnLayers[index] = currentLayer;
		}
		
		this.menuDriver = new MenuDriver(this.respawnMaster, this.fader, this.imageFx, 3);
		this.menuDriver.initialize();
		this.menuDriver.setOptionText(0, _("TW001"));
		this.menuDriver.setOptionText(1, _("TW002"));
		this.menuDriver.setOptionText(2, _("TW003"));
		this.menuDriver.getControl(0).style.top = "25px";
		this.menuDriver.getControl(1).style.top = "85px";
		this.menuDriver.getControl(2).style.top = "145px";
		this.menuDriver.setEscapeOption(2);


		var control = this;
		document.onkeydown = function(event) { return control.onKeyDown(event); }
		document.onkeyup = function(event) { control.onKeyUp(event); }
		
		this.initialize(road, raceResult);
		
		this.displayLoop();
	},
 
	/**
	 * Boucle principale de gestion des vnements et d'affichage, pour la premire phase
	 * d'affichage des temps.
	 * Affiche progressivement (effet de persiennes) le fond semi-opaque
	 * puis met  jour avec les meilleurs temps.
	 */
	displayLoop : function() {
	
		var transitionCheckpoint = 2*this.transitionLineCount + this.transitionLineHeight;
		// transition initiale : on dploie le fond semi-opaque en persiennes
		if (this.timer < transitionCheckpoint)
		{
			for (var line = 0 ; line < this.transitionLineCount ; ++line)
			{
				
				var currentLineHeight = this.timer - 2*line;
				if (currentLineHeight>0 && currentLineHeight<= this.transitionLineHeight) {
					var currentLineStyle = this.transitionLines[line].style;
					currentLineStyle.height = currentLineHeight+"px";
					currentLineStyle.visibility = "visible";
				}
			}
		}
		else if (this.timer == transitionCheckpoint)
		{
			this.permanentMaster.style.opacity = 0.5;
			this.permanentMaster.style.backgroundColor = "black";
			this.transitionMaster.style.display = "none";

			var timeString = this.formatRaceTime(this.raceTime);
			this.permanentMaster.appendChild(document.createTextNode(timeString));
		}
		else if (!this.road.lapRecordUpdateCompleted) {
			// en attente de la rponse Ajax
			var line = "_ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _";
			var car = "'o-o-";
			var message = line.substr(1, this.timer%30)+car+line;
			message = message.substr(1, 30);
			this.hourglassAnimation.firstChild.data = message;
		}
		else if (this.road.lapRecordUpdateCompleted && !this.bestTimesShown)
		{
			this.hourglassAnimation.firstChild.data = '';
			if (this.userAcknowledged) {
				this.illustration.style.visibility="hidden";
				if (this.raceResult.messages.length > 0) {
					// Tant qu'il y a des messages, on les affiche un par un, en attendant une rponse 
					// utilisateur  chaque fois
					this.imageFx.outlinedWrite(this.lapRecordMessage, this.raceResult.messages[0].text);
					if (this.raceResult.messages[0].image != "") {
						this.illustration.setAttribute("src", this.raceResult.messages[0].image);
						this.illustration.style.visibility="visible";
					} 
					this.imageFx.outlinedWrite(this.illustrationLegend, this.raceResult.messages[0].legend);
					this.raceResult.messages.shift();
					
					// on informe l'application si un circuit a t dbloqu
					// todo :  refactorer dans un endroit plus appropri (pas l'affichage, mais c'est le
					// premier qui ait connaissance de l'application dans la chane de traitement
					// depuis le retour de la requte Ajax)
					if (this.raceResult.unlockedTrack>0) {
						this.application.trackUnlocked(this.raceResult.unlockedTrack);
					}
				} else {
					this.imageFx.outlinedWrite(this.lapRecordMessage, '');
					this.imageFx.outlinedWrite(this.illustrationLegend, '');
					this.displayBest();
					this.bestTimesShown = true;
				}
				this.userAcknowledged = false;
			}

		}
		
		++this.timer;
	
		
		if (this.userAcknowledged && this.bestTimesShown && this.timer > transitionCheckpoint) {
			this.setQualified(this.raceResult.unlockedTrack > 0);
			this.timer = 0;
			this.userAcknowledged = false;
			setTimeout("timingWindowChoiceLoop(lotus.timingWindow)", 40);
		} else {
			setTimeout("timingWindowDisplayLoop(lotus.timingWindow)", 40);
		}
	},
	
	/**
	 * Boucle principale de gestion des vnements et d'affichage, pour le menu de choix
	 * utilisateur (suite, recommencer ..) en fin de course.
	 * Fait disparatre l'affichge prcdent, le remplace par le nouveau
	 * et met  jour en fonction de la slection utilisateur
	 */
	choiceLoop : function() {
		this.menuActive = true;
		var vanishingPoint = 25, respawnPoint=vanishingPoint+35, completionPoint=respawnPoint+10;
		if (this.timer < vanishingPoint)
		{	// 1re phase, on fait disparatre le panel avec un effet de zoom 
			// qui le rend de plus en plus transparent, jusqu' disparatre compltement
			var margin = 25 - this.timer*this.timer;
			var opacity = 1-0.04*this.timer;
			this.opaqueMaster.style.opacity = opacity;
			this.opaqueMaster.style.left = margin+"px";
			this.opaqueMaster.style.top = margin+"px";
			this.opaqueMaster.style.right = margin+"px";
			this.opaqueMaster.style.bottom = margin+"px";
			if (margin<0) {
				var clip = (-margin)+"px";
				var clipBottom = (240-margin)+"px";
				var clipRight = (480-margin)+"px";
				this.opaqueMaster.style.clip = "rect("+clip+", "+clipRight+", "+clipBottom+", "+clip+")";
			}
		} else if (this.timer == vanishingPoint) {
			// fin de la 1re phase : on efface tous les lments qui y ont servi
			this.opaqueMaster.style.display = "none";
		} else if (this.timer < respawnPoint) {
			// 2e phase : le panel revient sous la forme d'une srie de "lames de verre" semi-transparents
			// qui viennent se mettre en place
			var motionEnd = 20;
			for (var index = 0 ; index < this.respawnLayerCount ; ++index) {
				var localTimer = this.timer - vanishingPoint - index;
				localTimer = (localTimer > motionEnd ? motionEnd : localTimer);
				if (localTimer > 0) {
					var c = Math.cos(1.57*localTimer/motionEnd);
					var x = Math.round(30*c*c);
					x = (x>25 ? 25 :x);
					var y = Math.round(90*c);
					var layerStyle = this.respawnLayers[index].style;
					layerStyle.visibility = "visible";
					layerStyle.left = -x+"px";
					layerStyle.right = x+"px";
					layerStyle.width = (2*x+430)+"px";
					layerStyle.top = y+"px";
					layerStyle.bottom = y+"px";
				}
			}
		} else if (this.timer <= completionPoint) {
			// 3e phase : le texte des options apparat par dessus le panel
			var opacity = 0.1*(this.timer-respawnPoint);
			this.menuDriver.getControl(0).style.opacity = opacity;
			this.menuDriver.getControl(1).style.opacity = opacity;
			this.menuDriver.getControl(2).style.opacity = opacity;
		} 
		
		//  partir de la 3e phase, l'utilisateur peut choisir une option avec les flches
		if (this.timer > respawnPoint) {
			this.menuDriver.repaint(this.timer - respawnPoint);
		}
		
		++this.timer;
	
		var keepOn = true;
		if (this.menuDriver.userAcknowledged) {
			document.onkeydown = null;
			document.onkeyup = null;
			if (!this.fader.fading) { // le rideau est  l'arrt
				if (this.fader.faded) { // soit il est dja tir
					this.terminateScreen();
					keepOn = false;
				} else { // soit on est au tout dbut et on doit le tirer
					this.fader.fadeOut();
				}
			}
		} 
		if (keepOn) {
			setTimeout("timingWindowChoiceLoop(lotus.timingWindow)", 40);
		}
	},

	
	/**
	 *	Ferme et dtruit les lments DHTML de l'cran
	 */
	terminateScreen : function() {
		document.onkeydown = null;
		document.onkeyup = null;
		for (var line = 0 ; line < this.transitionLineCount ; ++line)	{
			this.transitionMaster.removeChild(this.transitionLines[line]);
			this.transitionLines[line] = null;
		}
		for (var index = 0 ; index < this.respawnLayerCount ; ++index)	{
			this.respawnMaster.removeChild(this.respawnLayers[index]);
			this.respawnLayers[index] = null;
		}
		this.menuDriver.terminate();
		this.permanentMaster.removeChild(this.hourglassAnimation);
		this.permanentMaster.removeChild(this.illustrationLegend);
		this.permanentMaster.removeChild(this.lapRecordMessage);
		this.opaqueMaster.removeChild(this.permanentMaster);
		this.opaqueMaster.removeChild(this.illustration);
		this.container.removeChild(this.opaqueMaster);
		this.container.removeChild(this.respawnMaster);
		this.container.removeChild(this.transitionMaster);
		this.application.afterTimingWindow();
	},

	/**
	 * Affiche les meilleurs temps de la course
	 */
	displayBest : function() {
		for (var index=0; index<=this.road.lapRecordCount; ++index) {
			var playerLine = (index == this.road.lapRecordCount);

			var lineTop = (13+index*14+(playerLine?2:0))+"%";

			// numro dans le classement : 1  5
			if (!playerLine) {
				var rankPanel = document.createElement('div');
				rankPanel.appendChild(document.createTextNode(1+index));
				rankPanel.style.left = "5%";
				rankPanel.style.top = lineTop;
				this.permanentMaster.appendChild(rankPanel);
			}
			
			// nom du joueur
			var nameContents = (playerLine ? this.road.playerLapRecord : this.road.lapRecords[index]).name;
/*			if (nameContents.length > 20) {
				nameContents = nameContents.substr(, 0, 17)+"...";
			} else */ if (nameContents == "") {
				nameContents = "--------------------";
			}
			var namePanel = document.createElement('div');
			namePanel.appendChild(document.createTextNode(nameContents));
			namePanel.style.left = "10%";
			namePanel.style.top = lineTop;
			this.permanentMaster.appendChild(namePanel);
			
			// voiture avec laquelle a t fait le record
			var raceTime = (playerLine ? this.road.playerLapRecord : this.road.lapRecords[index]).time;
			if (raceTime > 0) {
				var carIndex = (playerLine ? this.road.playerLapRecord : this.road.lapRecords[index]).car;
				var carImage = document.createElement('div');
				carImage.style.backgroundImage = "url(cars/allCars24.png)";
				carImage.style.backgroundPosition = "0px "+(-24*(carIndex-1))+"px";
				carImage.style.left = "60%";
				carImage.style.top = lineTop;
				carImage.style.width = "60px";
				carImage.style.height = "24px";
				this.opaqueMaster.appendChild(carImage);
			}
			
			// temps mis
			
			var timeString = this.formatRaceTime(raceTime);
			var timePanel = document.createElement('div');
			timePanel.appendChild(document.createTextNode(timeString));
			timePanel.style.left = "80%";
			timePanel.style.top = lineTop;
			this.permanentMaster.appendChild(timePanel);
		}

	},
	
	/**
	 * Formate un temps en 1/100e de seconde vers une chane "mm:ss:cc",
	 * en rajoutant si ncessaire un zro devant les secondes ou 1/100e de seconde
	 * pour garder un format  deux chiffres.
	 */
	formatRaceTime : function(inTime) {
		inTime = parseInt(inTime);
		var timeString = "--:--:--";
		if (inTime > 0) {
			var cs = inTime%100;
			var seconds = ((inTime-cs)/100)%60;
			var minutes = Math.floor(inTime/6000);
			timeString = (minutes<10?"0":"")+minutes+":"+(seconds<10?"0":"")+seconds+":"+(cs<10?"0":"")+cs;
		}
		return timeString;
	},
	
	
	/**
	 * Handler des vnements onkeydown (touche clavier presse)
	 * Choix du mode s'il s'agit d'une flche verticale
	 * Renvoie true si l'vnement doit continuer  tre propag (i.e. non trait ici), false sinon
	 */
	onKeyDown : function(event) {
		if (this.menuActive) {
			return this.menuDriver.onKeyDown(event);
		} 
		var handled = this.keyControl(event, true);
		return !handled;
	},
	
	/**
	 * Handler des vnements onkeyup (touche clavier relche)
	 */
	onKeyUp : function(event) {
		if (this.menuActive) {
			this.menuDriver.onKeyUp(event);
		} else {
			this.keyControl(event, false);
		}
	},
	
	/**
	 * Handler dlgu pour les vnements clavier.
	 * Enregistre les appuis et les relchements des flches
	 * et de la touche Escape.
	 * Renvoie true si l'vnement est gr, false sinon
	 */
	keyControl : function(event, value)
	{
		var handled = true;
		var key = 0;
		if (window.event) { // IE
			key = window.event.keyCode;
		} else { // FF, Opera,...
			key = event.which;
		}

		switch (key) {
			case 27 : // escape
				this.userAcknowledged |= value;
				break;
			case 13 : // entre
			case 32 : // espace
				this.userAcknowledged |= value;
				break;
			default :
				handled = false;
		}
		return handled;
	}	
	
}

function timingWindowDisplayLoop(screen)
{
	screen.displayLoop();
}

function timingWindowChoiceLoop(screen)
{
	screen.choiceLoop();
}