import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.SwingUtilities;

import sun.java2d.loops.FillRect;

public class Debugger {
   
	private List<GameTurn> gameTurns = new ArrayList<GameTurn>();
    private GamePanel gamePanel = new GamePanel();
    private ControlsPanel controlsPanel = new ControlsPanel();
    private JTextArea logArea = new JTextArea(20, 20);
    private JFrame frame = new JFrame("CloudWars Debugger");
    private int currentIndex = 0;
    private DebugReader reader;
    
    public Debugger(final DebugReader reader) {
    	this.reader = reader;
    	SwingUtilities.invokeLater(new Runnable() {
    		public void run() {
    			createAndShowGUI(); 
    		}
    	});
    	gamePanel.setGameState(reader.GetTurn());
    }
    
    public static void main(String[] args){
    	File gamesdir = new File("games");
    	if(gamesdir.isDirectory()){
    		File [] files = gamesdir.listFiles();
    		Arrays.sort(files, new Comparator<File>() {
    			@Override
    			public int compare(File o1, File o2) {
    				return (int)(o2.lastModified() - o1.lastModified());
    			}
    		});
    		try {
				Debugger d = new Debugger(new DebugReader(files[0]));
			} catch (FileNotFoundException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
    	}
    }
	
	

    private void createAndShowGUI() {
        System.out.println("Created GUI on EDT? "+
        SwingUtilities.isEventDispatchThread());
        
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        JPanel rootPanel = new JPanel();
        rootPanel.setLayout(new BoxLayout(rootPanel, BoxLayout.X_AXIS));
        frame.add(rootPanel);
        
        rootPanel.add(gamePanel);
        JPanel sidePanel = new JPanel();
        sidePanel.setLayout(new BoxLayout(sidePanel, BoxLayout.Y_AXIS));
        rootPanel.add(sidePanel);
        sidePanel.add(logArea);
        logArea.append("Log output: \n");
        logArea.setEditable(false);
        JScrollPane scrollPane = new JScrollPane(logArea, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
        sidePanel.add(scrollPane);
        sidePanel.add(controlsPanel);
        frame.pack();
        frame.setVisible(true);
        
    }
    
    public void nextFrame(){
    	GameTurn turn = reader.GetTurn();
    	gamePanel.setGameState(turn);
    	gamePanel.paint(gamePanel.getGraphics());
    	//for(String s: turn.getLogMessages()){
    	//	logArea.append(s + "\n");
    	//}
    }
    
    class ControlsPanel extends JPanel{
    	private JButton nextButton;
    	public ControlsPanel() {
    		nextButton = new JButton("Next");
    		this.add(nextButton);
    		nextButton.addActionListener(new ActionListener() {
				
				@Override
				public void actionPerformed(ActionEvent e) {
					if(e.getSource().equals(nextButton)){
						nextFrame();
					}
				}
			});
		}
    }
        
    class GamePanel extends JPanel {
    	
    	private GameTurn gameState;
    	
    	public GamePanel() {
    		setBorder(BorderFactory.createLineBorder(Color.black));
    	}
    	
    	public Dimension getPreferredSize() {
    		return new Dimension(1280,720);
    	}
    	
    	public void paintComponent(Graphics g) {
    		super.paintComponent(g);
    		g.setColor(Color.WHITE);
    		g.fillRect(0, 0, 1280, 720);
    		if(gameState != null){
    			for(Cloud c: gameState.Rainclouds){
        			drawCloud(gameState, c, Color.GRAY, Color.RED, g);
        		}
    			for(int i = 0; i < gameState.Thunderstorms.size(); i++){
    				if(i == gameState.MeIndex){
    					drawMe(gameState, g);
    				}else{
    					drawCloud(gameState, gameState.Thunderstorms.get(i), Color.ORANGE, Color.RED, g);
    				}
    			}
    		}
    		
    	}
    	
    	private void drawMe(GameTurn gs, Graphics g){
    		Cloud me = gameState.Me();
    		
    		int cx = (int) me.Position.x;
			int cy = (int) me.Position.y;
			if(gs.target != null){
				int tx = (int) gs.target.x;
				int ty = (int) gs.target.y;
				g.setColor(Color.GREEN);
				g.drawLine(cx, cy, tx, ty );
			}
			drawCloud(gs, me, Color.BLUE, Color.RED, g);
			if(gs.wind != null){
				int wx = 5 * (int) gs.wind.x;
				int wy = 5 * (int) gs.wind.y;
				g.setColor(Color.MAGENTA);
				g.drawLine(cx, cy, cx+wx, cy+wy );
			}
    	}
    	
    	private void drawCloud (GameTurn gs, Cloud c, Color color, Color vcolor, Graphics g){
			int r = (int) c.Radius();
			int d = r+r;
			int cx = (int) c.Position.x;
			int cy = (int) c.Position.y;
			int x = (int) cx - r;
			int y = (int) cy - r;
			int vx = (int) c.Velocity.x;
			int vy = (int) c.Velocity.y;
			g.setColor(color);
			g.fillOval(x, y, d, d);
			if(r > 0){
				g.setColor(vcolor);
				g.drawLine(cx, cy, cx+ 3*vx, cy+ 3 *vy);
				if(r>5) {
					g.setColor(Color.BLACK);
					String vapor = "v" + Integer.toString(((int)c.Vapor));
					//String reward = "r" + Integer.toString((int)MyAi.reward(gs, c));
					//String cost = "c" + Integer.toString((int)MyAi.cost(gs, c.futurePos()));
					g.drawChars(vapor.toCharArray(), 0, vapor.length() , cx, y );
					//g.drawChars(reward.toCharArray(), 0, reward.length(), cx, y+10);
					//g.drawChars(cost.toCharArray(), 0, cost.length(), cx, y+20);
				}
			}
    	}

		public GameState getGameState() {
			return gameState;
		}

		public void setGameState(GameTurn gameState) {
			this.gameState = gameState;
		}  
    }
}

