#include <iostream>
#include <string>
#include <boost/asio.hpp>
#include <boost/format.hpp>
#include <vector>

#ifdef _WIN32
#define msleep(ms) Sleep(ms)
#else
#define msleep(ms) usleep(1000*(ms))
#endif

namespace asio = boost::asio;
using boost::asio::ip::tcp;

using boost::format;

class Cloud {
	public:
		float x, y, vx, vy, size;
		
		Cloud wind(float nx, float ny) {
			float strength = sqrtf(nx*nx + ny*ny);
			
			size -= strength;
			
			float radius = sqrtf(size);
			
			vx += 5 * nx / radius;
			vy += 5 * ny / radius;
			
			float nradius = sqrtf(strength);
			
			float wx = nx / strength;
			float wy = ny / strength;
			
			float distance = 1.1 * (radius + nradius);
			
			Cloud res = {
				x - wx * distance,
				y - wy * distance,
				-20 * nx / strength + vx,
				-20 * ny / strength + vy,
				strength
			};
			
			return res;
		}
		
		void move(int N) {
			float vd, d;
			if(N == 1) {
				vd = 1;
				d = 0.999;
			} else {
				vd = (powf(0.999, N) - 1) / (0.999 - 1);
				d = powf(0.999, N);
			}
			
			x += 0.1 * vx * vd;
			y += 0.1 * vy * vd;
			vx *= d;
			vy *= d;
			
			float radius = sqrtf(size);
			
			if(x - radius < 0) { x = radius; vx = fabs(vx) * 0.6; }
			if(y - radius < 0) { y = radius; vy = fabs(vy) * 0.6; }
			if(1280 - x - radius < 0) { x = 1280 - radius; vx = -fabs(vx) * 0.6; }
			if( 720 - y - radius < 0) { y =  720 - radius; vy = -fabs(vy) * 0.6; }
		}
		
		void consume(Cloud& other) {
			float dx = x - other.x;
			float dy = y - other.y;
			float dz = sqrtf(dx*dx + dy*dy);
			
			while(sqrtf(size) + sqrtf(other.size) >= dz && size >= 1.0 && other.size >= 1.0) {
				if(size > other.size) {
					size += 1.0;
					other.size -= 1.0;
				} else {
					size -= 1.0;
					other.size += 1.0;
				}
			}
		}
};
typedef std::vector<Cloud> Clouds;

//#include "viz.h"

class Client {
	private:
		asio::io_service io_service;
		tcp::socket socket;
		asio::streambuf recvbuf;
		std::istream nin;
	
		int iteration;
		int index;
		Clouds thunderstorms;
		Clouds clouds;
		Cloud* me;
		
		int itercomp;
		
	public:
		Client(const tcp::endpoint& ep) : socket(io_service), nin(&recvbuf) {
			std::cout << "Connecting." << std::endl;
			
			socket.connect(ep);
			
			std::cout << "Connected, waiting for START." << std::endl;
			
			write("NAME fisk\n");
			
			if(readline() != "START") {
				throw std::runtime_error("Received invalid input.");
			}
			
			itercomp = 0;
		}
		
		void write(const std::string& str) {
			asio::write(socket, asio::buffer(str.c_str(), str.size()));
		}
		
		void write(const format& fmt) {
			write(std::string(boost::str(fmt)));
		}
		
		std::string readline() {
			asio::read_until(socket, recvbuf, '\n');
			
			std::string line;
			std::getline(nin, line);
			return line;
		}
		
		void wind(float x, float y) {
			write(format("WIND %f %f\n") % x % y);
			std::cout << "Sent wind: " << readline() << std::endl;
		}
		
		void update_state() {
			thunderstorms.clear();
			clouds.clear();
			
			write("GET_STATE\n");
			while(1) {
				std::string line = readline();
				std::istringstream is(line);
				
				std::string cmd;
				is >> cmd;
				
				if(cmd == "BEGIN_STATE") {
					is >> iteration;
				} else if(cmd == "YOU") {
					is >> index;
				} else if(cmd == "THUNDERSTORM") {
					Cloud c;
					is >> c.x >> c.y >> c.vx >> c.vy >> c.size;
					thunderstorms.push_back(c);
				} else if(cmd == "RAINCLOUD") {
					Cloud c;
					is >> c.x >> c.y >> c.vx >> c.vy >> c.size;
					clouds.push_back(c);
				} else if(cmd == "END_STATE") {
					break;
				}
			}
			std::cout << "New state: " << iteration << ", " << thunderstorms.size() << ", " << clouds.size() << std::endl;
		}
		
		float simulate(float x, float y) {
			Clouds simulclouds(clouds);
			Clouds simulstorms(thunderstorms);
			
			bool send_wind = false;
			
			// Let out wind and create a corresponding raincloud.
			if(x*x + y*y > 0.99 && sqrtf(x*x + y*y) < simulstorms[index].size / 2) {
				//simulclouds.push_back(simulstorms[index].wind(x, y));
				//simulstorms[index].wind(x, y);
				send_wind = true;
			}
			
			const int step_size = 1;
			
			float score = 0;
			
			for(int sim_it = 0; sim_it < 300; sim_it += step_size) {
				// Send wind.
				if(send_wind && sim_it >= itercomp) {
					simulclouds.push_back(simulstorms[index].wind(x, y));
					send_wind = false;
				}
				
				// Move clouds.
				for(Clouds::iterator it = simulclouds.begin(); it != simulclouds.end(); it++) {
					it->move(step_size);
				}
				
				for(Clouds::iterator it = simulstorms.begin(); it != simulstorms.end(); it++) {
					it->move(step_size);
				}
				
				// Test intersection and consume smaller clouds.
				for(Clouds::iterator it = simulclouds.begin(); it != simulclouds.end(); it++) {
					for(Clouds::iterator it2 = it + 1; it2 != simulclouds.end(); it2++) {
						it->consume(*it2);
					}
					for(Clouds::iterator it2 = simulstorms.begin(); it2 != simulstorms.end(); it2++) {
						it->consume(*it2);
					}
				}
				for(Clouds::iterator it = simulstorms.begin(); it != simulstorms.end(); it++) {
					for(Clouds::iterator it2 = it + 1; it2 != simulstorms.end(); it2++) {
						it->consume(*it2);
					}
				}
				
				// Remove consumed clouds.
				for(Clouds::iterator it = simulclouds.begin(); it < simulclouds.end(); it++) {
					if(it->size < 1.0) {
						it = simulclouds.erase(it);
					}
				}
				if(simulstorms[index].size < 1.0) {
					return -10000;
				}
				if(simulstorms[!index].size < 1.0) {
					score += 100000;
				}
				score += simulstorms[index].size * (300 - sim_it);
			}
			//std::cout << "> " << x << "," << y << "," << simulstorms[index].size << " <" << std::endl;
			
			//viz(simulclouds, simulstorms, index);
			
			//return simulstorms[index].size;
			return score;
		}
		
		void run() {
			while(1) {
				update_state();
				
				time_t start = clock();
				
				float best_score = simulate(0, 0);
				float best_x = 0;
				float best_y = 0;
				
				std::cout << "Do nothing: " << best_score << std::endl;
				 
				const float smax = thunderstorms[index].size / 2 * (iteration < 10 ? 0.8 : 0.5);
				const float strengths[] = {smax, 0.75 * smax, 0.5 * smax, 0.25 * smax, 1.001};
				
				for(int angle = 0; angle < 16; angle++) {
					for(const float* strength = strengths; strength < strengths + sizeof(strengths); strength++) {
						float tx = sinf(angle * M_PI_4 * 0.5) * *strength;
						float ty = cosf(angle * M_PI_4 * 0.5) * *strength;
						float ts = simulate(tx, ty);
						if(ts > best_score) {
							best_score = ts;
							best_x = tx;
							best_y = ty;
						}
					}
				}
				
				if(best_x*best_x + best_y*best_y > 0.8) {
					wind(best_x, best_y);
				}
				time_t stop = clock();
				
				itercomp = 60.0 * (stop - start) / (CLOCKS_PER_SEC);
				
				std::cout << "Time elapsed: " << (1000000.0 * (stop - start) / (CLOCKS_PER_SEC)) << "us. itercomp: "
					<< itercomp << " Current size: "
					<< thunderstorms[index].size << " Best score: "
					<< best_score << "(" << best_x << ", " << best_y << ")" << std::endl;
				
				if((1000000.0 * (stop - start) / CLOCKS_PER_SEC) < 100000) {
					msleep(100.0 - (1000.0 * (stop - start) / CLOCKS_PER_SEC));
				}
			}
		}
};

int main(int argc, char* argv[]) {
	try {
		if (argc != 3) {
			std::cerr << "Usage: " << argv[0] << " <ip> <port>" << std::endl;
			return 1;
		}
		
		//viz_init();
		
		Client client(tcp::endpoint(boost::asio::ip::address::from_string(argv[1]), atoi(argv[2])));
		client.run();
		
	} catch (std::exception& e) {
		std::cout << "Exception: " << e.what() << std::endl;
	}
	return 0;
}