#version 330 core

uniform sampler2D dirt;
uniform sampler2D grass;
uniform sampler2D rock;
uniform sampler2D snow;
uniform sampler2D detail;
uniform sampler2D bump;

// .x = min height, .y = max height, .z = min slope, .w = max slope
uniform vec4 rockParams;
uniform vec4 dirtParams;
uniform vec4 snowParams;
uniform vec4 grassParams;
// .x=grass, .y=dirt, .z=rock, .w=snow
uniform vec4 mapStrength;

in vec2 pos_zw;
in vec3 terrain_normal;
in vec3 normal;
in vec3 ec_pos;
in vec3 uvw;
in float height;

layout (location = 0) out vec4 diffuse_specular;
layout (location = 1) out vec4 normal_depth;

#include <deferred/tangentspace_include.frag>

float computeWeight(float value, float minExtent, float maxExtent)
{
	if ( value >= minExtent && value <= maxExtent )
	{
		float range = maxExtent - minExtent;

		float weight = value - minExtent;

		weight *= 1.0 / range;
		weight -= 0.5;
		weight *= 2.0;

		weight *= weight;

		weight = 1.0 - abs(weight);
		weight = clamp(weight, 0.001, 1.0);

		return weight;
	}
	else
	{
		return 0.0;
	}
}


void main()
{
	// basic weights
	vec3 n = normalize(terrain_normal);
	float slope = 1.0 - n.y;

	vec4 weights;
	weights.x = computeWeight(height, rockParams.x, rockParams.y) * computeWeight(slope, rockParams.z, rockParams.w) * mapStrength.z;
	weights.y = computeWeight(height, dirtParams.x, dirtParams.y) * computeWeight(slope, dirtParams.z, dirtParams.w) * mapStrength.y;
	weights.z = computeWeight(height, snowParams.x, snowParams.y) * computeWeight(slope, snowParams.z, snowParams.w) * mapStrength.w;
	weights.w = computeWeight(height, grassParams.x, grassParams.y) * computeWeight(slope, grassParams.z, grassParams.w) * mapStrength.x;
	weights *= 1.0 / (weights.x + weights.y + weights.z + weights.w);

	// planar slope
	vec3 tpw = abs(n);
	tpw = (tpw - vec3(0.2)) * 7.0;
	tpw = max(tpw, vec3(0.0));
	tpw *= 1.0 / (tpw.x + tpw.y + tpw.z);


	// dirt
	vec4 temp = tpw.z * texture(dirt, uvw.xy);
	temp += tpw.x * texture(dirt, uvw.yz);
	temp += tpw.y * texture(dirt, uvw.xz);
	vec4 final = temp * weights.y;

	// grass
	temp = tpw.z * texture(grass, uvw.xy);
	temp += tpw.x * texture(grass, uvw.yz);
	temp += tpw.y * texture(grass, uvw.xz);
	final += temp * weights.w;

	// rock
	temp = tpw.z * texture(rock, uvw.xy);
	temp += tpw.x * texture(rock, uvw.yz);
	temp += tpw.y * texture(rock, uvw.xz);
	final += temp * weights.x;

	// snow
	temp = tpw.z * texture(snow, uvw.xy);
	temp += tpw.x * texture(snow, uvw.yz);
	temp += tpw.y * texture(snow, uvw.xz);
	final += temp * weights.z;

	// detail
	temp = tpw.z * texture(detail, uvw.xy*0.125);
	temp += tpw.x * texture(detail, uvw.yz*0.125);
	temp += tpw.y * texture(detail, uvw.xz*0.125);
	final += (temp*2.0 - vec4(1.0))*0.75;

	diffuse_specular.rgb = final.rgb;
	diffuse_specular.a = 0.0;

	vec3 tangent, binormal;
	per_fragment_tangent_space(ec_pos, uvw.xz, tangent, binormal);
	mat3 mr = mat3(normalize(tangent.xyz), normalize(binormal), normalize(normal));
	vec3 b = texture(bump, uvw.xz).xyz*2.0 - vec3(1.0);
	n = normalize(mr * b);

	normal_depth.xyz = n; //normalize(normal);
	normal_depth.w = pos_zw.x/pos_zw.y;
}
