using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.GamerServices;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Media;

using Game.Entidades; 
using Game.Servicios; 

namespace Game.Graficos
{
    /// <summary>
    /// This is a game component that implements IUpdateable.
    /// </summary>
    public class MatrizProcessor : AbstractProcessor
    {
        private Matrix world = Matrix.Identity;
        private Matrix centrado = Matrix.CreateTranslation(Parametros.TRANS_CENTRADO);

        private SpriteBatch spriteBatch; 
        
        private Viewport vista;

        private Plano planoSup;
        private Plano planoInf;
 
        private Contador contador; 

        private GameScene Main;
        private Matriz matriz;

        private ControlProcessor control;
        private PlayerIndex player;

        private Boolean controlGiroHabilitado; 

        // Lista de cubos a insertar en la matriz. 
        private List<Cubo> cubosEliminados = new List<Cubo>();
        
        public MatrizProcessor(DefaultScene game, PlayerIndex player)
            : base(game)
        {
            // clase base
            this.Main = (GameScene)game;
            // Jugador
            this.player = player; 
            // Control
            this.control = Main.control[(int)player]; 
            // Matriz de datos
            this.matriz = Main.matrices[(int)player]; 
            // Vista 
            this.vista = Main.vistas[(int)player];
            // Contador de efecto de detonadores
            this.contador = new Contador(50, 4);
            // Control de giro controlado por usuario
            this.controlGiroHabilitado = !(this.player == PlayerIndex.Two && (int)Main.Parameters[Parametros.PARAM_OPCION_SELECCIONADA] == Parametros.OPC_PLAYER_VS_CPU);
        }

        public override void Initialize()
        {
        }

        public override void LoadContent()
        {
            this.spriteBatch = new SpriteBatch(GraphicsDevice);
            this.planoInf = new Plano(GraphicsDevice, Recursos.suelo);
            this.planoSup = new Plano(GraphicsDevice, Recursos.suelo);
        }

        public override void UnloadContent()
        {
        }

        #region Logica

        public override void Update(GameTime gameTime)
        {
            Vector2 stick; 
            // Control de rotacion de la matriz
            stick = (controlGiroHabilitado) ? control.Vista : Vector2.UnitX * 0.25f; 
            matriz.rotacion += MathHelper.ToRadians(((90.0f / 1000.0f) * gameTime.ElapsedGameTime.Milliseconds) * stick.X);
            matriz.rotacion = this.matriz.rotacion % MathHelper.TwoPi;
            if (matriz.rotacion < 0) matriz.rotacion = matriz.rotacion + MathHelper.TwoPi;
            this.world = centrado * Matrix.CreateRotationY(matriz.rotacion);
            
            // Control de ejecucion de los cubos
            for (int X = 0; X < Parametros.MAX_X; X++)
            {
                for (int Z = 0; Z < Parametros.MAX_Z; Z++)
                {
                    List<Cubo> columna = matriz.coleccion[X,Z]; 
                    for (int indx = columna.Count - 1; indx >= 0; indx--)
                    {
                        Cubo cubo = columna[indx];
                        if (cubo.EsExplosion) 
                        {
                            Detonacion(columna, indx, X, Z);
                            cubosEliminados.Add(columna[indx]);
                            columna.RemoveAt(indx);

                        } 
                        else if (cubo.EsDetonador) 
                        {
                            if (cubo.EstaArmado)
                            {
                                // Comprobacion de detonacion por fin de tiempo
                                if (Detonacion(columna, indx, X, Z))
                                {
                                    this.Main.colorFondos[(int)this.player] = Parametros.CUBO_COLORES[cubo.Color];
                                    this.Main.audio.Play(Parametros.SONIDO_DETONACION);
                                    cubosEliminados.Add(columna[indx]);
                                    columna.RemoveAt(indx);
                                }
                            }
                            else
                            {
                                // Comprobacion de detonacion por colision entre detonadores
                                if (DetonacionDetonadores(columna, indx, X, Z))
                                {
                                    this.Main.colorFondos[(int)this.player] = Parametros.CUBO_COLORES[cubo.Color];
                                    this.Main.audio.Play(Parametros.SONIDO_DETONACION);
                                    cubosEliminados.Add(columna[indx]);
                                    columna.RemoveAt(indx);
                                }
                                else
                                {
                                    cubo.Temporizador(gameTime.ElapsedGameTime.Milliseconds);
                                }
                            }
                        }
                        // TODOS
                        UpdateCubo(cubo, (indx > 0) ? columna[indx - 1] : null, gameTime);
                    }
                }
            }

            // Actualizacion de cubos eliminados. 
            UpdateCubosEliminados();
        }

        #region Propagacion de detonaciones

        private Boolean DetonacionDetonadores(List<Cubo> columna, int indx, int x, int z)
        {
            Boolean detonacion = false;

            Cubo detonador = columna[indx];

            // Comprobacion detonacion en cubo inferior si tb es detonador del mismo color. 
            Cubo cuboInferior = (indx > 0) ? columna[indx - 1] : null;
            if (cuboInferior != null && cuboInferior.Color == detonador.Color && cuboInferior.EsDetonador)
            {
                if (cuboInferior.PosicionSuper == detonador.PosicionBase)
                {
                    cuboInferior.Explota(); detonacion = true;
                }
            }

            return detonacion;
        }

        private Boolean Detonacion(List<Cubo> columna, int indx, int x, int z)
        {
            Boolean detonacion = false; 

            Cubo detonador = columna[indx];

            // Comprobacion detonacion en cubo superior
            Cubo cuboSuperior = (indx < columna.Count - 1) ? columna[indx + 1] : null;
            if (cuboSuperior != null && cuboSuperior.Color == detonador.Color)
            {
                if (cuboSuperior.PosicionBase == detonador.PosicionSuper ) {
                    cuboSuperior.Explota(); detonacion = true; 
                }
            }

            // Comprobacion detonacion en cubo inferior
            Cubo cuboInferior = (indx > 0) ? columna[indx - 1] : null;
            if (cuboInferior != null && cuboInferior.Color == detonador.Color)
            {
                if (cuboInferior.PosicionSuper == detonador.PosicionBase)
                {
                    cuboInferior.Explota(); detonacion = true;
                }
            }

            // Comprobacion detonacion en cubos contiguos. 
            if (x > 0) {
                if (ComprobacionDetonacionColumna(detonador, matriz.coleccion[x - 1, z])) detonacion = true; 
            }
            if (z > 0) {
                if (ComprobacionDetonacionColumna(detonador, matriz.coleccion[x, z - 1])) detonacion = true; 
            }
            if (x < Parametros.MAX_X - 1)
            {
                if (ComprobacionDetonacionColumna(detonador, matriz.coleccion[x + 1, z])) detonacion = true;
            }
            if (z < Parametros.MAX_Z - 1)
            {
                if (ComprobacionDetonacionColumna(detonador, matriz.coleccion[x, z + 1])) detonacion = true;
            }
            return detonacion; 
        }

        private Boolean ComprobacionDetonacionColumna(Cubo detonador, List<Cubo> columna)
        {
            Boolean detonacion = false; 
            float detonadorBase = detonador.PosicionBase;
            float detonadorSuper = detonador.PosicionSuper; 
            foreach( Cubo cubo in columna ) 
            {
                if (cubo.PosicionBase == detonador.PosicionBase && cubo.Color == detonador.Color)
                {
                    cubo.Explota(); detonacion = true; 
                }
            }
            return detonacion; 
        }

        #endregion

        private void UpdateCubo(Cubo cubo, Cubo cuboBajo, GameTime gameTime)
        {
            float posicionSuper = (cuboBajo == null) ? 0.0f : cuboBajo.PosicionSuper; 
            if (cubo.PosicionBase > posicionSuper)
            {
                cubo.Caer(Parametros.CUBO_VELOCIDAD_CAIDA * gameTime.ElapsedGameTime.Milliseconds);
            }
            else if (cubo.PosicionBase < posicionSuper)
            {
                cubo.Posar(cuboBajo);
            }
        }

        private void UpdateCubosEliminados()
        {
            for (int indx = 0; indx <= cubosEliminados.Count - 1; indx++)
            {
                Cubo cubo = cubosEliminados[indx];
                if (cubo.Escala >= .0f)
                {
                    cubo.Escala -= .02f;
                }
                else
                {
                    // Cubo registrado en la lista de transferencia para el otro jugador
                    Main.transferencias[(int)this.player].Enviar(cubosEliminados[indx]);
                    // Cubo eliminado de la lista de eliminados. 
                    cubosEliminados.RemoveAt(indx);
                }
            }
        }

        #endregion

        #region Visualizacion

        public override void Draw(GameTime gameTime)
        {
            // Actualizacion de contador de efecto de detonadores
            this.contador.Update(gameTime);
            // Seleccion de vista
            GraphicsDevice.Viewport = vista; 
            // Proyeccion de fondo
            DrawFondo();
            // Bucle de proyeccion de todos los cubos en la matriz. 
            foreach (List<Cubo> columna in matriz.coleccion)
            {
                foreach (Cubo cubo in columna)
                {
                    DrawCubo(cubo);
                }
            }
            // Proyeccion de cubos eliminados
            foreach (Cubo cubo in cubosEliminados)
            {
                DrawCuboEliminados(cubo);
            }
            // Proyeccion de planos superiores e inferiores. 
            DrawPlanos();
        }

        private void DrawFondo()
        {
            spriteBatch.Begin();
            spriteBatch.Draw(Recursos.FondoJuego, new Rectangle(0, 0, vista.Width, vista.Height), Main.colorFondos[(int)this.player]);
            spriteBatch.End();
            GraphicsDevice.DepthStencilState = DepthStencilState.Default;
        }

        private void DrawPlanos()
        {
            planoInf.Draw(Main.camara,
                    Main.vistas[(int)player],
                    Matrix.CreateRotationX(-MathHelper.PiOver2) *
                    Matrix.CreateRotationY(matriz.rotacion));
            planoSup.Draw(Main.camara,
                    Main.vistas[(int)player],
                    Matrix.CreateRotationX(-MathHelper.PiOver2) *
                    Matrix.CreateRotationY(matriz.rotacion) *
                    Matrix.CreateTranslation(0.0f, Parametros.MAX_Y, 0.0f));
        }

        private void DrawCubo(Cubo cubo)
        {
            foreach (ModelMesh mesh in Recursos.CuboModelo.Meshes)
            {
                foreach (CubeEffect be in mesh.Effects)
                {
                    be.DiffuseTextureBase = Recursos.TDifusaCubo;
                    if ( cubo.EsDetonador ) {
                        if ( !cubo.EstaArmado ) {
                            be.DiffuseTextureOver = Recursos.TEmisionDetonador_Contando[ cubo.Lapso ]; 
                        } else {
                            be.DiffuseTextureOver = Recursos.TEmisionDetonador_Armado[1]; 
                        }
                    } else {
                        be.DiffuseTextureOver = Recursos.TEmisionCubo;
                    }
                    be.Color = ( cubo.EsDetonador ) ? Parametros.DETONADOR_COLORES[cubo.Color, contador.Valor ] :  Parametros.CUBO_COLORES[cubo.Color]; 
                    be.Projection = Main.camara.projection;
                    be.View = Main.camara.view;
                    be.World = Matrix.CreateScale(cubo.Escala) * Matrix.CreateTranslation(cubo.Posicion) * world;
                }
                mesh.Draw();
            }
            // Efecto de generacin de cubos recin transferidos
            if (cubo.Escala < 1.0f)
            {
                cubo.Escala += .05f;
            }
        }

        private void DrawCuboEliminados(Cubo cubo)
        {
            foreach (ModelMesh mesh in Recursos.CuboModelo.Meshes)
            {
                foreach (CubeEffect be in mesh.Effects)
                {
                    be.DiffuseTextureBase = Recursos.TDifusaCubo;
                    if (cubo.EsDetonador)
                    {
                        if (!cubo.EstaArmado)
                        {
                            be.DiffuseTextureOver = Recursos.TEmisionDetonador_Contando[cubo.Lapso];
                        }
                        else
                        {
                            be.DiffuseTextureOver = Recursos.TEmisionDetonador_Armado[1];
                        }
                    }
                    else
                    {
                        be.DiffuseTextureOver = Recursos.TEmisionCubo;
                    }
                    be.Color = Color.White; 
                    be.Projection = Main.camara.projection;
                    be.View = Main.camara.view;
                    be.World = Matrix.CreateScale( cubo.Escala ) * Matrix.CreateTranslation(cubo.Posicion) * world;
                }
                mesh.Draw();
            }
        }

        #endregion
    }
}
