﻿using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.GamerServices;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Media;

using FontContent;

using Game.Entidades;
using Game.Graficos;
using Game.Servicios; 

namespace Game
{
    class GameScene : DefaultScene
    {


        public Camara camara;

        public Matriz[] matrices = new Matriz[2];
        public Jugador[] jugadores = new Jugador[2];
        public ControlProcessor[] control = new ControlProcessor[2];
        public Viewport[] vistas = new Viewport[2];
        public Viewport vistaGeneral; 
        public Transferencia[] transferencias = new Transferencia[2];
        public Color[] colorFondos;

        public AudioService audio;
        public MusicaService musica; 

        private FontBatch fontBatch;
        private int timer = 9999;
        private int beepTimer = 0; 

        // Estado de juego
        bool pausa = false; 
        private enum EstadoJuego
        {
            Inicio, 
            Jugando,
            Final
        }
        EstadoJuego estadoJuego = EstadoJuego.Inicio;


        // Instancias de componentes de jugadores y matrices. 
        private JugadorProcessor[] jugadorComponentes = new JugadorProcessor[2];
        private MatrizProcessor[] matricesComponentes = new MatrizProcessor[2];
        private TransferenciaProcessor transferenciaComponente;

        public GameScene(Microsoft.Xna.Framework.Game game)
            :base(game)
        {
            // TODO: Construct any child components here
        }

#region Inicializacion 

        public override void Initialize()
        {
            // Inicializacion de Camara
            camara = new Camara(new Vector3(0.0f, 15.0f, 20.0f), new Vector3(0.0f, 2.0f, -10.0f), Vector3.Up, 
                Parametros.RESOLUCION_W/2.0f/ Parametros.RESOLUCION_H);

            // Inicializacion de Vistas
            vistas[0] = new Viewport(0, 0, Parametros.RESOLUCION_W / 2, Parametros.RESOLUCION_H);
            vistas[0].MinDepth = 0.0f; vistas[0].MaxDepth = 1.0f;
            vistas[1] = new Viewport(Parametros.RESOLUCION_W / 2, 0, Parametros.RESOLUCION_W / 2, Parametros.RESOLUCION_H);
            vistas[1].MinDepth = 0.0f; vistas[1].MaxDepth = 1.0f;
            vistaGeneral = new Viewport(0, 0, Parametros.RESOLUCION_W, Parametros.RESOLUCION_H);

            // Inicializacion de control de controles de entrada. 
            IniciarControles();
            // Inicializacion de matrices de cubos
            IniciarMatrices();
            // Inicializacion de jugadores
            IniciarJuegadores();
            // Inicializacion de transferencias
            IniciarTransferencias();
            // Inicializacion de interfaz
            IniciarHUDS();
            // Filtro de colores de fondo
            colorFondos = new Color[] { Color.White, Color.White }; 
 
            DetenerInteraccion();
            // Render de planos
            //Components.Add(new EscenarioComponente(this));

            base.Initialize();
        }

        private void IniciarControles()
        {
            this.control[0] = new ControlProcessor(this, PlayerIndex.One);
            this.control[1] = new ControlProcessor(this, PlayerIndex.Two);
            Components.Add(control[0]);
            Components.Add(control[1]);
        }

        private void IniciarMatrices()
        {
            matrices[0] = new Matriz();
            matrices[1] = new Matriz();
            matricesComponentes[0] = new MatrizProcessor(this, PlayerIndex.One);
            matricesComponentes[1] = new MatrizProcessor(this, PlayerIndex.Two);
            this.Components.Add(matricesComponentes[0]);
            this.Components.Add(matricesComponentes[1]);
        }

        private void IniciarJuegadores()
        {
            jugadores[0] = new Jugador();
            jugadores[1] = new Jugador();
            jugadorComponentes[0] = new JugadorProcessor(this, PlayerIndex.One);
            if ((int)this.Parameters[Parametros.PARAM_OPCION_SELECCIONADA] == Parametros.OPC_PLAYER_VS_PLAYER)
            {
                jugadorComponentes[1] = new JugadorProcessor(this, PlayerIndex.Two);
            }
            else
            {
                jugadorComponentes[1] = new JugadorProcessor(this, PlayerIndex.Two, (int)this.Parameters[Parametros.PARAM_IADIFICULTAD_SELECCIONADA]);
            }
            this.Components.Add(jugadorComponentes[0]);
            this.Components.Add(jugadorComponentes[1]);
        }

        private void IniciarTransferencias()
        {
            // Inicializacion de transferencias
            transferencias[0] = new Transferencia();
            transferencias[1] = new Transferencia();
            transferenciaComponente = new TransferenciaProcessor(this);
            Components.Add(transferenciaComponente);
        }

        private void IniciarHUDS()
        {
            this.Components.Add(new HUDProcessor(this, PlayerIndex.One));
            this.Components.Add(new HUDProcessor(this, PlayerIndex.Two));
        }

#endregion 

#region Carga contenidos

        public override void LoadContent()
        {
            this.fontBatch = new FontBatch(
                    new SpriteBatch(GraphicsDevice),
                    Content.Load<Fuente>(@"fuentes/ptrfuente"),
                    Content.Load<Texture2D>(@"fuentes/fuente"));
            LoadSonidos();
            LoadCubo();
            LoadObjetivo();
            LoadFondo();
            LoadSuelo();

            base.LoadContent();
        }

        private void LoadSonidos()
        {
            this.audio = new AudioService(new SoundEffect[] { Content.Load<SoundEffect>(@"sonidos/lanzado"), 
                                                                 Content.Load<SoundEffect>(@"sonidos/colision"), 
                                                                 Content.Load<SoundEffect>(@"sonidos/detonacion"),
                                                                 Content.Load<SoundEffect>(@"sonidos/recepcion"), 
                                                                 Content.Load<SoundEffect>(@"sonidos/preparado"), 
                                                                 Content.Load<SoundEffect>(@"sonidos/fin")});
            this.musica = new MusicaService( Content.Load<SoundEffect>(@"musica/musicajuego.com"));
        }


        private void LoadCubo()
        {
            // Texturas
            Recursos.TDifusaCubo = Content.Load<Texture2D>(@"cubo/difusa");
            Recursos.TEmisionCubo = Content.Load<Texture2D>(@"cubo/emision");
            Recursos.TEmisionDetonador_Contando = new Texture2D[10];
            Recursos.TEmisionDetonador_Contando[0] = Content.Load<Texture2D>(@"cubo/t_ex_0");
            for (int indx = 1; indx < 10; indx++)
            {
                Recursos.TEmisionDetonador_Contando[indx] = Content.Load<Texture2D>(@"cubo/t_" + indx.ToString());
            }
            Recursos.TEmisionDetonador_Armado = new Texture2D[2];
            Recursos.TEmisionDetonador_Armado[0] = Content.Load<Texture2D>(@"cubo/t_ex_0");
            Recursos.TEmisionDetonador_Armado[1] = Content.Load<Texture2D>(@"cubo/t_ex_1");
            // Sombreadores
            CubeEffect shader = new CubeEffect(Content.Load<Effect>(@"cubo/shaderCube"));
            // Modelos. 
            Recursos.CuboModelo = Content.Load<Model>(@"cubo/cubo");
            Recursos.CuboModelo.Meshes[0].MeshParts[0].Effect = shader;
        }

        private void LoadObjetivo()
        {
            Recursos.ObjetivoTexturas = new Texture2D[2];
            Recursos.ObjetivoTexturas[0] = Content.Load<Texture2D>(@"target/target1");
            Recursos.ObjetivoTexturas[1] = Content.Load<Texture2D>(@"target/target2");
            BasicEffect shader = new BasicEffect(this.GraphicsDevice);
            shader.VertexColorEnabled = false;
            shader.TextureEnabled = true;
            shader.LightingEnabled = false;
            Recursos.ObjetivoModelo = Content.Load<Model>(@"target/target");
            Recursos.ObjetivoModelo.Meshes[0].MeshParts[0].Effect = shader;
        }

        private void LoadFondo()
        {
            Recursos.FondoJuego = Content.Load<Texture2D>(@"fondos/fondo");
        }

        private void LoadSuelo()
        {
            Recursos.suelo = Content.Load<Texture2D>(@"matriz/suelo");
        }

#endregion 

#region Logica

        public override void Update(GameTime gameTime)
        {
            // Control de cierre
            if (GamePad.GetState(PlayerIndex.One).Buttons.Back == ButtonState.Pressed ||
                Keyboard.GetState().IsKeyDown(Keys.Escape))
            {
                this.RaiseExit();
            }
            // Control de salida a menu
            if (Keyboard.GetState().IsKeyDown(Keys.F12))
            {
                this.musica.Stop();
                this.RaiseNext();
            }
            // Control de distribucion logica. 
            switch (this.estadoJuego)
            {
                case EstadoJuego.Inicio:
                    Update_Inicio(gameTime);
                    break; 

                case EstadoJuego.Jugando:
                    Update_Ejecucion(gameTime);
                    break; 

                case EstadoJuego.Final:
                    Update_Final(gameTime);
                    break; 
            }
            // Recuperacion de filtros
            for (int indx = 0; indx < 2; indx++)
            {
                if (colorFondos[indx].R < 255) colorFondos[indx].R+=5;
                if (colorFondos[indx].G < 255) colorFondos[indx].G+=5;
                if (colorFondos[indx].B < 255) colorFondos[indx].B+=5; 
            }
            base.Update(gameTime);
        }

        private void Update_Inicio(GameTime gameTime)
        {
            beepTimer += gameTime.ElapsedGameTime.Milliseconds;
            if (beepTimer > 3000)
            {
                audio.Play(Parametros.SONIDO_PREPARADO);
                beepTimer = 0;
            }
            timer -= gameTime.ElapsedGameTime.Milliseconds;
            if (timer < 0)
            {
                this.estadoJuego = EstadoJuego.Jugando;
                ActivarInteraccion();
            }
        }

        private void Update_Ejecucion(GameTime gameTime)
        {
            musica.Update();
            // Control de pausa
            if (GamePad.GetState(PlayerIndex.One).Buttons.BigButton == ButtonState.Pressed ||
                Keyboard.GetState().IsKeyDown(Keys.Pause))
            {
                if (pausa)
                    ActivarInteraccion();
                else
                    DetenerInteraccion();
                pausa = !pausa;
            }
            // Control de fin de juego
            if (matrices[0].Rotura || matrices[1].Rotura)
            {
                DetenerInteraccion();
                this.estadoJuego = EstadoJuego.Final;
                musica.Stop();
                audio.Play(Parametros.SONIDO_FIN);
            }
        }

        private void Update_Final(GameTime gameTime)
        {
            // Control de vuelta al menu.
            timer += gameTime.ElapsedGameTime.Milliseconds;
            timer %= 1000; 
            if (GamePad.GetState(PlayerIndex.One).Buttons.Start == ButtonState.Pressed ||
                Keyboard.GetState().IsKeyDown(Keys.Enter))
            {
                this.RaiseNext();
            }
        }


        // Control el estado de interaccion de los jugadores.
        private void DetenerInteraccion()
        {
            jugadorComponentes[0].enabled = false;
            jugadorComponentes[1].enabled = false;
            jugadorComponentes[0].visible = false;
            jugadorComponentes[1].visible = false; 
            matricesComponentes[0].enabled = false;
            matricesComponentes[1].enabled = false;
            transferenciaComponente.enabled = false; 
        }

        private void ActivarInteraccion()
        {
            jugadorComponentes[0].enabled = true;
            jugadorComponentes[1].enabled = true;
            jugadorComponentes[0].visible = true;
            jugadorComponentes[1].visible = true;
            matricesComponentes[0].enabled = true;
            matricesComponentes[1].enabled = true;
            transferenciaComponente.enabled = true; 
        }

#endregion 

#region Visualizacion 

        public override void Draw(GameTime gameTime)
        {
            GraphicsDevice.Clear(Color.Blue);
            // TODO: Add your drawing code here
            base.Draw(gameTime);

            switch (this.estadoJuego)
            {
                case EstadoJuego.Inicio:
                    DrawInicio(gameTime);
                    break;
                case EstadoJuego.Final:
                    DrawFinal(gameTime);
                    break; 
            }
        }

        public void DrawInicio(GameTime gameTime)
        {
            int t1w, t1h, t2w, t2h; 
            foreach (Viewport v in vistas)
            {
                t1w = 7 * (v.Width / 24);
                t1h = 5 * (v.Height / 10);
                t2w = 11 * (v.Width / 24);
                t2h = 11 * (v.Height / 20); 

                GraphicsDevice.Viewport = v;
                this.fontBatch.DrawShadowText("PREPARATE!", t1w, t1h, 1.0f, Color.White, 0, 5);
                this.fontBatch.DrawShadowText((timer / 1000).ToString(), t2w, t2h, 2.0f, Color.White, 0, 12);
            }
 
        }

        public void DrawFinal(GameTime gameTime)
        {
            int t1h, t2w, t2h; 
            for (int player = 0; player < 2; player++)
            {
                Viewport v = vistas[player];
                t1h = v.Height / 3;
                t2w = v.Width / 4;
                t2h = v.Height / 2; 

                GraphicsDevice.Viewport = vistas[player];
                if (matrices[player].Rotura)
                {
                    this.fontBatch.DrawShadowText("TERMINADO", v.Width/16, t1h, 2.0f,Color.Salmon, 0, 12);
                }
                else
                {
                    this.fontBatch.DrawShadowText("VICTORIA", v.Width/6, t1h, 2.0f, Color.GreenYellow, 0, 12);
                }

                if (timer < 500)
                {
                    this.fontBatch.DrawShadowText("PULSA [ENTER]", t2w, t2h, 1.0f, Color.White, 0, 12);
                }
            }



        }

#endregion 

    }
}
