package org.shiftone.jrat.provider.tree;

import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import org.shiftone.jrat.core.Accumulator;
import org.shiftone.jrat.core.MethodKey;
import org.shiftone.jrat.provider.tree.ui.TreeOutputXmlViewBuilder;
import org.shiftone.jrat.util.StringUtil;
import org.shiftone.jrat.util.log.Logger;
import org.shiftone.jrat.util.time.Clock;
import org.shiftone.jrat.util.time.TimeUnit;

/**
 * Class StackNode
 * 
 * @author $Author: jeffdrost $
 * @version $Revision: 1.27 $
 */
public class StackNode extends Accumulator {

	private static final Logger LOG = Logger.getLogger(StackNode.class);
	public static final String VERSION = StringUtil.revision("$Revision: 1.27 $");
	public static final String VIEWER = TreeOutputXmlViewBuilder.class.getName();
	protected final MethodKey methodKey;
	protected final StackNode parent;
	protected final HashMap children = new HashMap();

	public StackNode() {

		// root node
		this.methodKey = null;
		this.parent = null;
	}

	public StackNode(MethodKey methodKey, StackNode treeNode) {

		this.methodKey = methodKey;
		this.parent = treeNode;
	}

	/**
	 * Method gets <b>AND CREATES IF NEEDED</b> the requested tree node
	 */
	public StackNode getChild(MethodKey methodKey) {

		StackNode treeNode = null;
		synchronized (children) {
			treeNode = (StackNode) children.get(methodKey);
			if (treeNode == null) {
				treeNode = new StackNode(methodKey, this);
				children.put(methodKey, treeNode);
			}
		}
		return treeNode;
	}

	public final StackNode getParentNode() {

		return parent;
	}

	public final boolean isRootNode() {

		return (methodKey == null);
	}

	public MethodKey getMethodKey() {

		return methodKey;
	}

	public void printXML(PrintWriter out) {

		LOG.info("printXML...");
		if (isRootNode()) {
			printRoot(out);
		} else {
			// this should never happen
			printNonRoot(out, 0);
		}
		LOG.info("printXML complete.");
		LOG.info("checkError = " + out.checkError());
	}

	private void printRoot(PrintWriter out) {

		long start = Clock.currentTimeNanos();
		LOG.info("printRoot...");
		out.println("<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>");
		out.println("<!-- generated by JRat 'call tree' " + VERSION + " provider on " + new Date() + "-->");
		out.println("<jrat-tree viewer=\"" + VIEWER + "\">");
		out.println("<view>");
		printChildren(out, 0);
		out.println("</view>");
		out.println("<!-- file written in " + (Clock.currentTimeNanos() - start) + " nanos -->");
		out.println("</jrat-tree>");
		LOG.info("printRoot complete.");
	}

	private void printNonRoot(PrintWriter out, int depth) {

		// out.print(StringUtil.bufferString(depth, ' '));
		out.print("<call");
		out.print(" c=\"" + methodKey.getClassName() + "\"");
		out.print(" m=\"" + methodKey.getMethodName() + "\"");
		out.print(" s=\"" + methodKey.getSignature() + "\"");
		out.print(" ent=\"" + getTotalEnters() + "\"");
		out.print(" xit=\"" + getTotalExits() + "\"");
		out.print(" err=\"" + getTotalErrors() + "\"");
		out.print(" dur=\"" + getTotalDurationNanos() + "\"");
		out.print(" sos=\"" + getSumOfSquares() + "\"");
		out.print(" mct=\"" + getMaxConcurrentThreads() + "\"");
		if (getTotalDurationNanos() != 0) {
			out.print(" min=\"" + getMinDurationNanos() + "\"");
			out.print(" max=\"" + getMaxDurationNanos() + "\"");
		}
		if (children.size() > 0) {
			out.println(">");
			printChildren(out, depth);
			// out.print(StringUtil.bufferString(depth, ' '));
			out.println("</call>");
		} else {
			out.println("/>");
		}
		out.flush();
	}

	private void printChildren(PrintWriter out, int depth) {

		// need to clone map - concurrency issues
		List list = new ArrayList();
		synchronized (children) {
			list.addAll(children.values());
		}
		for (int i = 0; i < list.size(); i++) {
			StackNode treeNode = (StackNode) list.get(i);
			treeNode.printNonRoot(out, depth + 1);
		}
	}
	// ---------------------------------------------------------------
}
