
varying vec4 eye;

uniform sampler2D tex;
uniform sampler2D noise;
uniform float xresm, yresm;
uniform float chroma;
uniform float hue;
uniform float saturation;
uniform float light;
uniform float contrast;

// Returns height of a pixel. Notice that the height isn't normalized (meaning that result is between 0.0 to 3.0)
float getHeight(vec2 texcoord) {
	vec4 col = texture2D(tex, texcoord);
	return col.r+col.g+col.b;
}

vec3 getNormal(vec2 tc) {
	float xp = xresm;
	float yp = yresm;

	float x0 = getHeight(tc-vec2(xp, 0.0))/3.0;
	float x2 = getHeight(tc+vec2(xp, 0.0))/3.0;
	float y0 = getHeight(tc-vec2(0.0, yp))/3.0;
	float y2 = getHeight(tc+vec2(0.0, yp))/3.0;

	float dh = x0-x2;
	float dv = y0-y2;

	// no black
	if (dh == 0.0) dh = 0.0001;

	return normalize(vec3(dh, dv, 0.025));
}


vec3 RGBtoHSL(vec3 color)
{
	vec3 hsl; // init to 0 to avoid warnings ? (and reverse if + remove first part)

	float fmin = min(min(color.r, color.g), color.b);    //Min. value of RGB
	float fmax = max(max(color.r, color.g), color.b);    //Max. value of RGB
	float delta = fmax - fmin;             //Delta RGB value

	hsl.z = (fmax + fmin) / 2.0; // Luminance

	if (delta == 0.0)		//This is a gray, no chroma...
	{
		hsl.x = 0.0;	// Hue
		hsl.y = 0.0;	// Saturation
	}
	else                                    //Chromatic data...
	{
		if (hsl.z < 0.5)
			hsl.y = delta / (fmax + fmin); // Saturation
		else
			hsl.y = delta / (2.0 - fmax - fmin); // Saturation

		float deltaR = (((fmax - color.r) / 6.0) + (delta / 2.0)) / delta;
		float deltaG = (((fmax - color.g) / 6.0) + (delta / 2.0)) / delta;
		float deltaB = (((fmax - color.b) / 6.0) + (delta / 2.0)) / delta;

		if (color.r == fmax )
			hsl.x = deltaB - deltaG; // Hue
		else if (color.g == fmax)
			hsl.x = (1.0 / 3.0) + deltaR - deltaB; // Hue
		else if (color.b == fmax)
			hsl.x = (2.0 / 3.0) + deltaG - deltaR; // Hue

		if (hsl.x < 0.0)
			hsl.x += 1.0; // Hue
		else if (hsl.x > 1.0)
			hsl.x -= 1.0; // Hue
	}

	return hsl;
}

float HUEtoRGB(float f1, float f2, float hue)
{
	if (hue < 0.0)
		hue += 1.0;
	else if (hue > 1.0)
		hue -= 1.0;
	float res;
	if ((6.0 * hue) < 1.0)
		res = f1 + (f2 - f1) * 6.0 * hue;
	else if ((2.0 * hue) < 1.0)
		res = f2;
	else if ((3.0 * hue) < 2.0)
		res = f1 + (f2 - f1) * ((2.0 / 3.0) - hue) * 6.0;
	else
		res = f1;
	return res;
}

vec3 HSLtoRGB(vec3 hsl)
{
	vec3 rgb;

	if (hsl.y == 0.0)
		rgb = vec3(hsl.z); // Luminance
	else
	{
		float f2;

		if (hsl.z < 0.5)
			f2 = hsl.z * (1.0 + hsl.y);
		else
			f2 = (hsl.z + hsl.y) - (hsl.y * hsl.z);

		float f1 = 2.0 * hsl.z - f2;

		rgb.r = HUEtoRGB(f1, f2, hsl.x + (1.0/3.0));
		rgb.g = HUEtoRGB(f1, f2, hsl.x);
		rgb.b = HUEtoRGB(f1, f2, hsl.x - (1.0/3.0));
	}

	return rgb;
}



void main() {
	vec2 xy = gl_TexCoord[0].xy;
	vec4 color = texture2D(tex, xy);

	if (chroma > 0.0) {
		vec4 e = normalize(eye);
		vec3 normal = getNormal(xy);
		float r = texture2D(tex, xy+chroma*0.01*(refract(normal, e.xyz, 0.5).xy)).r;
		float g = texture2D(tex, xy+chroma*0.01*(refract(normal, e.xyz, 0.8).xy)).g;
		float b = texture2D(tex, xy+chroma*0.01*(refract(normal, e.xyz, 0.2).xy)).b;
		color.xyz = vec3(r, g, b);
	}

	vec3 hsl = RGBtoHSL(color.rgb);
	hsl.r += hue;
	if (hsl.r >= 1.0) hsl.r -= 1.0;
	if (hsl.r < 0.0) hsl.r += 1.0;

	hsl.g *= saturation;
	hsl.b *= light;

	color.rgb = HSLtoRGB(hsl);

	color.rgb = (color.rgb-0.5)*contrast+0.5;

	gl_FragColor = color;
}
