#ifndef UI_UI_STACK_HPP
#define UI_UI_STACK_HPP

#include "ui/fps_counter.hpp"
#include "ui/event_key.hpp"
#include "ui/event_misc.hpp"
#include "ui/event_mouse_button.hpp"
#include "ui/event_mouse_motion.hpp"

namespace gfx
{
	class SurfaceScreen;
}

namespace ui
{
	class UiState;

	/** \brief The UI state stack.
	 *
	 * All new UI state handlers enter this stack, preferably on top of it. They
	 * remain there until they decide to die by their own volition.
	 */
	class UiStack
	{
    public:
      /** Handy typedef */
      typedef boost::shared_ptr<UiState> value_type;

		private:
			/** List (stack) of all UI states. */
			std::list<value_type> _state_list;

			/** Screen that's being updated. */
			gfx::SurfaceScreen &_screen;

			/** FPS calulator / bookkeeper. */
			FpsCounter _fps_counter;

		public:
			/** \brief Default constructor.
			 *
			 * @param scr Screen to attach to.
			 * @param fps Maximum response frequency (Hz).
			 */
			UiStack(gfx::SurfaceScreen & scr, int fps);

			/** \brief Destructor.
			 */
			~UiStack() { }

		private:
			/** \brief Get all pending events into the event lists.
			 *
			 * Will handle the events on the states, topmost first.
			 */
			void handleEvents();

			/** \brief Reinstate first UI state in list.
			 *
			 * @return True if there was something to reinstate, false if not.
			 */
			bool reinstateFirst();

		public:
			/** \brief Kill the topmost state.
			 *
			 * This state will stand down after the next update() call.
			 */
			void killTopState();

			/** \brief Push a state onto the stack.
			 *
			 * @param state State to push.
			 */
			void pushState(value_type &state);

			/** \brief Push a state after the current one onto the stack.
			 *
			 * States can use this to push new UI states to the stack to be executed
			 * immediately after they themselves exit.
			 *
			 * @param state State to push.
			 */
			void pushStateAfter(value_type &state);

			/** \brief Run until the last UI state is removed.
			 *
			 * All UI states run in a three-phase loop wherein they first wait for
			 * input, then are updated, and then rendered if there's still time.
			 *
			 * The call order of events is: misc, mouse motion, key, mouse button.
			 *
			 * The states may decide to quit at any point of this cycle by either
			 * returning false (from Update or Render) or setting their alive state
			 * to false (input). They will be removed before the next phase would
			 * begin.
			 */
			void run();

			/** \brief Switch places of two top states.
			 *
			 * Useful when switching between, for example, a menu and a game state.
			 */
			void switchTopStates();

		public:
			/** \brief Framerate accessor.
			 *
			 * @return Current framerate.
			 */
			inline int getFps() const
			{
				return _fps_counter.getCurrentFps();
			}

			/** \brief Frame count accessor.
			 *
			 * @return Current frame count.
			 */
			inline int64_t getFrameCount() const
			{
				return _fps_counter.getFrameCount();
			}

			/** \brief Get the number of states.
			 *
			 * @return Number of states currently.
			 */
			inline unsigned getNumStates() const
			{
				return static_cast<unsigned>(_state_list.size());
			}

			/** \brief Get the topmost state.
			 *
			 * @return Topmost state pointer.
			 */
			inline const UiState* getStateTop() const
			{
				return _state_list.empty() ? NULL : _state_list.front().get();
			}

			/** \brief Wrapper for push state after.
			 *
			 * @param state State to push.
			 */
			inline void pushStateAfter(UiState *state)
			{
				value_type container(state);
				this->pushStateAfter(container);
			}

			/** \brief Wrapper for push state.
			 *
			 * @param state State to push.
			 */
			inline void pushState(UiState *state)
			{
				value_type container(state);
				this->pushState(container);
			}

			/** \brief Wrapper for run.
			 *
			 * @param state Initial state to add.
			 */
			inline void run(value_type &state) 
			{
				this->pushState(state);
				this->run();
			}

			/** \brief Wrapper for run.
			 *
			 * @param state Initial state to add.
			 */
			inline void run(UiState *state) 
			{
				this->pushState(state);
				this->run();
			}
	};

	/** Only one UI stack should exist anyway. */
	extern UiStack ui_stack;
}

#endif
