#include "ui/ui_stack.hpp"

#include "gfx/generic.hpp"
#include "thr/dispatch.hpp"
#include "ui/ui_state.hpp"

#include <sstream>

using namespace ui;

UiStack::UiStack(gfx::SurfaceScreen & scr, int fps) :
	_screen(scr),
	_fps_counter(fps) { }

void UiStack::handleEvents()
{
	std::list<EventKey> events_key;
	std::list<EventMisc> events_misc;
	std::list<EventMouseButton> events_mouse_button;
	std::list<EventMouseMotion> events_mouse_motion;

	// Gather all events to execute them in a correct order.
	SDL_Event sevent;
	while(SDL_PollEvent(&sevent) != 0)
	{
		switch(sevent.type)
		{
			case SDL_QUIT:
				events_misc.push_back(EventMisc(QUIT));
				break;

			case SDL_KEYDOWN:
				events_key.push_back(
						EventKey(sevent.key.keysym.sym, true));
				break;

			case SDL_KEYUP:
				events_key.push_back(
						EventKey(sevent.key.keysym.sym, false));
				break;

			case SDL_MOUSEMOTION:
				events_mouse_motion.push_back(
						EventMouseMotion(sevent.motion.x, sevent.motion.y,
							sevent.motion.xrel, sevent.motion.yrel));
				break;

			case SDL_MOUSEBUTTONDOWN:
				events_mouse_button.push_back(
						EventMouseButton(sevent.button.button, true,
							sevent.button.x, sevent.button.y));
				break;

			case SDL_MOUSEBUTTONUP:
				events_mouse_button.push_back(
						EventMouseButton(sevent.button.button, false,
							sevent.button.x, sevent.button.y));
				break;

			default:
				// Unknown event, silently ignore.
				break;
		}
	}

	// If a state returns false, it wants to say it didn't handle an event
	// correctly, in which case the handling is passed onto an event in a
	// deeper level. If no-one returns true to tell they handled the event,
	// it's silently forgotten.
	BOOST_FOREACH(EventMisc &vv, events_misc)
	{
		BOOST_FOREACH(value_type &ww, _state_list)
		{
			if(ww->handle(vv, *this))
			{
				break;
			}
		}
	}
	BOOST_FOREACH(EventMouseMotion &vv, events_mouse_motion)
	{
		BOOST_FOREACH(value_type &ww, _state_list)
		{
			if(ww->handle(vv, *this))
			{
				break;
			}
		}
	}
	BOOST_FOREACH(EventKey &vv, events_key)
	{
		BOOST_FOREACH(value_type &ww, _state_list)
		{
			if(ww->handle(vv, *this))
			{
				break;
			}
		}
	}
	BOOST_FOREACH(EventMouseButton &vv, events_mouse_button)
	{
		BOOST_FOREACH(value_type &ww, _state_list)
		{
			if(ww->handle(vv, *this))
			{
				break;
			}
		}
	}
}

/** \brief Kill the topmost state.
*/
void UiStack::killTopState()
{
	if(!_state_list.empty())
	{
		_state_list.front()->die();
	}
}

void UiStack::pushState(value_type &state)
{
	_state_list.push_front(state);
	state->reinstate(*this);
}

void UiStack::pushStateAfter(value_type &state)
{
	if(_state_list.empty())
	{
		this->pushState(state);
		return;
	}
	std::list<value_type>::iterator iter = _state_list.begin();
	++iter;
	_state_list.insert(iter, state);
	// No reinstate since not on top.
}

bool UiStack::reinstateFirst()
{
	if(_state_list.empty())
	{
		return false;
	}
	(_state_list.front())->reinstate(*this);
	return true;
}

void UiStack::run() 
{
	while(!_state_list.empty())
	{
		int state = _fps_counter.check(true);
		// Timing failure.
		if(state <= 0)
		{
			continue;
		}

		UiState *topst = _state_list.front().get();

		thr::wait_privileged(boost::bind(&UiStack::handleEvents, this));
		// After handling the state may have decided it has died.
		if(!topst->isAlive())
		{
			_state_list.pop_front();
			this->reinstateFirst();
			continue;
		}

		thr::wait(boost::bind(&UiState::update, topst, boost::ref(_screen),
						boost::ref(*this), (1 < state)));
		// State may have decided to die also during update.
		if(!topst->isAlive())
		{
			_state_list.pop_front();
			this->reinstateFirst();
		}

		// Always check GL errors, they might be happening via asynchronous stuff
		// even if render was not invoked.
		thr::wait_privileged(gfx::check_opengl_errors);
	}

	// Destructing UI stack should take threading with it.
	thr::thr_quit();
}

void UiStack::switchTopStates()
{
	if(_state_list.size() >= 2)
	{
		std::list<value_type>::iterator aa = _state_list.begin(), bb = aa;
		++bb;
		value_type tmp = *bb;
		*bb = *aa;
		*aa = tmp;
		_state_list.front()->reinstate(*this);
	}
}

