#ifndef UI_TEXT_WORD_HPP
#define UI_TEXT_WORD_HPP

#include "ui/generic.hpp"

#include <sstream>

namespace ui
{
	/** \brief Represents one word within a text.
	 */
	class TextWord
	{
		private:
			/** Width of this word. */
			float _width;

			/** Word content. */
			std::wstringstream _word;

		public:
			/** \brief Constructor. */
			TextWord();

			/** \brief Constructor.
			 *
			 * @param cc Character to add.
			 * @param cwid Character width.
			 */
			TextWord(wchar_t cc, float cwid);

		public:
			/** \brief Add a character to this word.
			 *
			 * @param cc Character to add.
			 * @param cwid Character width.
			 * @return The new word length.
			 */
			float add(wchar_t cc, float cwid);

		public:
			/** \brief Clear all content of this word.
			 */
			inline void clear()
			{
				_width = 0.0f;
				_word.str(std::wstring());
			}

			/** \brief Accessor.
			 *
			 * @return Content.
			 */
			inline std::wstring getContent() const
			{
				return _word.str();
			}

			/** \brief Tell if this word is empty.
			 * 
			 * @return True if yes, false if not.
			 */
			inline bool isEmpty() const
			{
				return (_word.str().length() <= 0);
			}

			/** \brief Accessor.
			 *
			 * @return Length of the word in characters.
			 */
			inline unsigned getLength() const
			{
				return static_cast<unsigned>(this->getContent().length());
			}

			/** \brief Accessor.
			 *
			 * @return Width.
			 */
			inline float getWidth() const
			{
				return _width;
			}

		public:
			/** \brief Assignment operator.
			 *
			 * @param op Source.
			 */
			inline const TextWord& operator=(const TextWord &op)
			{
				_width = op._width;
				_word.str(op._word.str());
				return *this;
			}

		public:
			/** \brief Output this to a stream.
			 *
			 * @param ss Target stream.
			 * @return Stream after input.
			 */
			std::ostream& put(std::ostream &ss) const;

			/** \brief Output a word into a stream.
			 *
			 * @param lhs Left-hand-side operand.
			 * @param rhs Right-hand-side operand.
			 * @return Modified stream.
			 */
			friend inline std::ostream& operator<<(std::ostream &lhs,
					const TextWord &rhs)
			{
				return rhs.put(lhs);
			}
	};
}
#endif
