#ifndef UI_TEXT_RECT_HPP
#define UI_TEXT_RECT_HPP

#include "gfx/color.hpp"
#include "gfx/font.hpp"
#include "math/rect.hpp"
//#include "math/vec.hpp"
#include "ui/text_row.hpp"

namespace ui
{
	/** \brief Rectangle of text.
	 *
	 * The higher-class abstraction of text row and word.
	 */
	class TextRect
	{
		protected:
			/** Restricting dimensions. */
			math::vec2f _limits;

			/** Actual text content dimensions (width and height). */
			math::vec2f _dimensions;

			/** >Text contained within, arranged as per-line-basis. */
			std::vector<TextRow> _content;

		public:
			/** \brief Constructor.
			 *
			 * @param plimits Area restriction.
			 */
			TextRect(const math::vec2f &plimits);

			/** \brief Destructor. */
			~TextRect() { }

		public:
			/** \brief Clears all content in this.
			 */
			void clear();

			/** \brief Renders text in this textarea.
			 *
			 * Fill color must have been set beforehand.
			 *
			 * @param carea Content area.
			 * @param font Font to use.
			 * @param font_size Font size to use.
			 * @param justify Justification schema.
			 */
			void renderText(const math::rect2f &carea, const gfx::Font &font,
					float font_size, gfx::TextJustify = gfx::LEFT) const;

			/** \brief Sets the contents of this.
			 *
			 * If the text rectangle can't hold a single character, this will return the
			 * string it was passed.
			 *
			 * @param str Input string to use (as much as possible).
			 * @param fnt Font to use.
			 * @param fs Font size to use.
			 * @return Rest of the string (what was not used), may be empty.
			 */
			std::wstring setContent(const std::wstring &str, const gfx::Font &fnt,
					float fs);

		private:
			/** \brief Append a row.
			 *
			 * Modifies the dimensions accordingly.
			 *
			 * @param op Textrow to add.
			 * @param fs Font size.
			 * @return True if row added (there was room), false if not.
			 */
			bool add(const TextRow &op, float fs);

		public:
			/** \brief Accessor.
			 *
			 * @return Contents of this.
			 */
			inline const std::vector<TextRow>& getContent() const
			{
				return _content;
			}

			/** \brief Accessor.
			 *
			 * @return Dimensions vector.
			 */
			inline const math::vec2f getDimensions() const
			{
				return _dimensions;
			}

			/** \brief Accessor.
			 * 
			 * @return Boundary rectangle.
			 */
			inline const math::vec2f& getLimits() const
			{
				return _limits;
			}

			/** \brief Set the boundary area of this.
			 *
			 * Will implicitly clear the content.
			 *
			 * @param plimits New limits.
			 */
			inline void setLimits(const math::vec2f &plimits)
			{
				this->clear();
				_limits = plimits;
			}

			/** \brief Wrapper for limit set.
			 *
			 * @param pw Width.
			 * @param ph Height.
			 */
			inline void setLimits(float pw, float ph)
			{
				this->setLimits(math::vec2f(pw, ph));
			}
	};
}

#endif
