#ifndef UI_CONSOLE_HPP
#define UI_CONSOLE_HPP

#include "gfx/surface_screen.hpp"
#include "ui/input_line.hpp"
#include "ui/text_area.hpp"

#include <boost/thread.hpp>

namespace ui
{
	/** \brief Text console class.
	 *
	 * A (not necessarily interactive) console for displaying text and scrolling
	 * it and displaying an optional progressbar.
	 *
	 * Does not fully overlap with TextArea functionality, but shares enough of
	 * it warrant inheritance.
	 */
	class Console
	{
		private:
			/** Font. */
			const gfx::Font &_font;

			/** Font size. */
			float _font_size;

			/** View area. */
			math::rect2f _area;

			/** Area usable for text. */
			math::rect2f _area_text;

			/** Area usable for progressbar. */
			math::rect2f _area_bar;

			/** Input line. */
			InputLine m_input;

			/** Progress bar position. */
			int _progress;

			/** Progress bar maximum. */
			int _progress_max;

			/** List of previous lines. */
			std::list<TextRect> _lines;

			/** Background texture. */
			gfx::Texture2D _background;

			/** Mutex to prevent simultaneous accesses to the log. */
			boost::mutex _mut;

		protected:
			/** Text color. */
			gfx::Color m_color_background;

			/** Text color. */
			gfx::Color m_color_border;

			/** Text color. */
			gfx::Color m_color_input_bottom;

			/** Text color. */
			gfx::Color m_color_input_top;

			/** Text color. */
			gfx::Color m_color_progress_empty;

			/** Text color. */
			gfx::Color m_color_progress_fill;

			/** Text color. */
			gfx::Color m_color_text_bottom;

			/** Text color. */
			gfx::Color m_color_text_top;

		public:
			/** \brief Constructor.
			 *
			 * @param fnt Font to use.
			 * @param fs Font size for this text box.
			 * @param prestrict Text area restriction (in terms of screen size).
			 * @param pfname Backround image filename.
			 */
			Console(const gfx::Font &fnt, float fs, const math::rect2f &prestrict,
					const char *pfname);

			/** \brief Constructor.
			 *
			 * @param fnt Font to use.
			 * @param fs Font size for this text box.
			 * @param screen Take restriction area from this screen object.
			 * @param pfname Backround image filename.
			 */
			Console(const gfx::Font &fnt, float fs, const gfx::SurfaceScreen &screen,
					const char *pfname);

			/** \brief Destructor. */
			virtual ~Console() { }

		private:
			/** \brief Set default colors.
			 */
			void setDefaultColors();

		public:
			/** \brief Append a row.
			 *
			 * The row may actually be multiple rows.
			 *
			 * @param op String (the row) to append.
			 */
			void addRow(const std::wstring &op);

		protected:
			/** \brief Update the console area.
			 *
			 * This updates the subsidiary areas derived from font size and the area in
			 * use.
			 */
			void updateArea();

		public:
			/** \brief Execute and clear the input line.
			 */
			virtual void execute();

			/** \brief Render this console.
			 *
			 * @param screen Screen to render to.
			 */
			virtual void render(gfx::SurfaceScreen &screen);

			/** \brief Render the progressbar.
			 *
			 * @param screen Screen to render to.
			 */
			virtual void renderProgress(const gfx::SurfaceScreen &screen) const;

			/** \brief Render the text content in the console.
			 *
			 * By default, this method should not need to be overwritten. Just
			 * remember to set fill colors before calling.
			 *
			 * @param screen Screen to render to.
			 */
			virtual void renderText(const gfx::SurfaceScreen &screen);

		public:
			/** \brief Wrapper for addRow.
			 *
			 * @param op Row to add.
			 */
			void addRow(const std::string &op)
			{
				this->addRow(wstr_utf8(op));
			}

			/** \brief Accessor.
			 *
			 * @return Input line.
			 */
			inline InputLine& getInput()
			{
				return m_input;
			}

			/** \brief Accessor.
			 *
			 * @return Input line.
			 */
			inline const InputLine& getInput() const
			{
				return m_input;
			}

			/** \brief Get the current progress.
			 *
			 * @return Progress as integer.
			 */
			inline int getProgress()
			{
				return _progress;
			}

			/** \brief Set the current progress.
			 *
			 * @param op New progress.
			 */
			inline void setProgress(int op)
			{
				_progress = math::max(0, math::min(_progress_max, op));
			}

			/** \brief Set the maximum progress.
			 *
			 * Will also reset the progress.
			 *
			 * @param op New maximum progress.
			 */
			inline void setProgressMax(int op)
			{
				_progress = 0;
				_progress_max = math::abs(op);
			}

			/** \brief Decrement progress by one.
			 */
			inline void decProgress()
			{
				this->setProgress(_progress - 1);
			}

			/** \brief Increment progress by one.
			 */
			inline void incProgress()
			{
				this->setProgress(_progress + 1);
			}
	};
}

#endif
