#ifndef THR_DISPATCH_HPP
#define THR_DISPATCH_HPP

#include "thr/generic.hpp"

namespace thr
{
	/** \brief Add a job.
	 *
	 * Anyone may execute this job.
	 *
	 * This function will never block, the job will always be added to the queue
	 * for execution. The actual time of execution for jobs
	 *
	 * @param pfunctor Functor to store.
	 */
	extern void dispatch_ext(const Task &pfunctor);

	/** \brief Add a privileged job.
	 *
	 * Add a primary, privileged job for execution. Only the privileged main
	 * executor may execute this job. A typical job of this kind would be for
	 * example any OpenGL call.
	 *
	 * Privileged jobs are executed in a first-in-first-out -manner. The usual
	 * method would be to dispatch privileged jobs until no jobs remain, then
	 * call wait_privileged(const Task&) with a screen flip task, which would
	 * return when all privileged tasks have been executed.
	 *
	 * Dispatching a privileged job from inside the main thread (i.e. another
	 * privileged job) will cause the dispatching to block and execute privileged
	 * jobs until the queue is empty.
	 *
	 * @param pfunctor Functor to store.
	 */
	extern void dispatch_privileged_ext(const Task &pfunctor);

	/** \brief Start threading and dispatching.
	 *
	 * This function should be called from the main thread, it will block until
	 * thr_quit is called. The main thread calling will stay here executing
	 * primary tasks.
	 *
	 * @param ntasks Additional tasks to create. 0 to autodetect.
	 */
	extern void thr_main(unsigned ntasks = 1);

	/** \brief Stop threading and dispatching.
	 *
	 * Orders all threads to quit, including the main thread, which will join
	 * all other threads before returning (thus being the last thread to stop
	 * running in a dispatcher).
	 */
	extern void thr_quit();

	/** \brief Wait until all outstanding jobs are done.
	 *
	 * If no jobs are in execution, this function returns immediately.
	 *
	 * If called from within a worker thread, also execute the jobs until the job
	 * queue is empty.
	 *
	 * If called from a primary thread, execute all jobs with priority on the
	 * privileged jobs until no jobs of any kind remain.
	 */
	extern void wait();

	/** \brief Add a privileged job and wait for it to complete.
	 *
	 * As dispatch_privileged(const Task&), but always block until the
	 * privileged job queue is empty, not just if called from within the main
	 * thread itself.
	 *
	 * @param pfunctor Functor to store.
	 */
	extern void wait_privileged_ext(const Task &pfunctor);

	/** \brief Wrapper for dispatch_ext.
	 *
	 * @param op Any binding.
	 */
	template <typename Type> inline void dispatch(Type op)
	{
		dispatch_ext(Task(op));
	}
	/** \cond */
	template<> inline void dispatch(Task op)
	{
		dispatch_ext(op);
	}
	template<> inline void dispatch(Task &op)
	{
		dispatch_ext(op);
	}
	template<> inline void dispatch(const Task &op)
	{
		dispatch_ext(op);
	}
	/** \endcond */

	/** \brief Wrapper for dispatch_privileged_ext.
	 *
	 * @param op Any binding.
	 */
	template <typename Type> inline void dispatch_privileged(Type op)
	{
		dispatch_privileged_ext(Task(op));
	}
	/** \cond */
	template<> inline void dispatch_privileged(Task op)
	{
		dispatch_privileged_ext(op);
	}
	template<> inline void dispatch_privileged(Task &op)
	{
		dispatch_privileged_ext(op);
	}
	template<> inline void dispatch_privileged(const Task &op)
	{
		dispatch_privileged_ext(op);
	}
	/** \endcond */

	/** \brief Wait until all outstanding jobs are done.
	 *
	 * As wait(), but add one job before continuing.
	 *
	 * @param op Any binding.
	 */
	template <typename Type> inline void wait(Type op)
	{
		dispatch_ext(Task(op));
		wait();
	}
	/** \cond */
	template<> inline void wait(Task op)
	{
		dispatch_ext(op);
		wait();
	}
	template<> inline void wait(Task &op)
	{
		dispatch_ext(op);
		wait();
	}
	template<> inline void wait(const Task &op)
	{
		dispatch_ext(op);
		wait();
	}
	/** \endcond */

	/** \brief Wrapper for wait_privileged_ext.
	 *
	 * As wait(), but add one job before continuing.
	 *
	 * @param op Any binding.
	 */
	template <typename Type> inline void wait_privileged(Type op)
	{
		wait_privileged_ext(Task(op));
	}
	/** \cond */
	template<> inline void wait_privileged(Task op)
	{
		wait_privileged_ext(op);
	}
	template<> inline void wait_privileged(Task &op)
	{
		wait_privileged_ext(op);
	}
	template<> inline void wait_privileged(const Task &op)
	{
		wait_privileged_ext(op);
	}
	/** \endcond */
}

#endif
