#include "gfx/font.hpp"
#include "gfx/surface_screen.hpp"
#include "thr/dispatch.hpp"
#include "ui/ui_stack.hpp"
#include "ui/ui_state.hpp"

#include "boost/program_options.hpp"

namespace po = boost::program_options;

/** Font location. */
#ifdef DATADIR
static std::string datadir(DATADIR);
#else
static std::string datadir(".");
#endif

/** Console output content. */
static const char *usage = ""
"Usage: timerdisp <options>\n"
"Timer display for monitor input lag comparisons.\n"
"\n";

/** \brief Timer display. */
class TimerDisp : public ui::UiState
{
	private:
		/** Font used. */
		const gfx::Font &_fnt;

	public:
		/** \brief Constructor.
		 *
		 * @param pscreen Screen.
		 */
		TimerDisp(const gfx::Font &pfnt) :
			_fnt(pfnt)
		{
			gfx::Color fill(1.0f, 1.0f, 1.0f, 1.0f);
			draw_fill(0, fill);
			draw_fill(1, fill);
			draw_fill(2, fill);
			draw_fill(3, fill);
		}

		virtual ~TimerDisp()
		{
			// Do nothing.
		}

	public:
		/** \brief Render everything.
		 *
		 * @param screen Screen to use.
		 * @param st UI stack.
		 */
		void render(gfx::SurfaceScreen &screen, ui::UiStack &st)
		{
			static const float FONTSIZE = 0.33f;

			int ww = screen.getWidth(),
					hh = screen.getHeight();
			float fww = static_cast<float>(ww),
						fhh = static_cast<float>(hh),
						aspect = fww / fhh;

			screen.clear(true, true);
			screen.select2D();

			unsigned cnt = static_cast<unsigned>(st.getFrameCount() % static_cast<uint64_t>(1000)),
							 hundreds = cnt / 100;
			cnt -= hundreds * 100;
			unsigned tens = cnt / 10;
			cnt -= tens * 10;

			gfx::bind_shader_2d_font();
			gfx::draw_glyph(aspect * 0.5f - FONTSIZE * 1.5f, 0.5f - FONTSIZE * 0.5f,
					FONTSIZE, _fnt.getGlyph(static_cast<wchar_t>(hundreds + '0')));

			gfx::draw_glyph(aspect * 0.5f - FONTSIZE * 0.5f, 0.5f - FONTSIZE * 0.5f,
					FONTSIZE, _fnt.getGlyph(static_cast<wchar_t>(tens + '0')));

			gfx::draw_glyph(aspect * 0.5f + FONTSIZE * 0.5f, 0.5f - FONTSIZE * 0.5f,
					FONTSIZE, _fnt.getGlyph(static_cast<wchar_t>(cnt + '0')));

			screen.update();
		}


	public:
		/** \cond */
		virtual bool handle(ui::EventKey &ev, ui::UiStack &st)
		{
			if(ev.isPress())
			{
				switch(ev.getCode())
				{
					case SDLK_ESCAPE:
					case SDLK_q:
						_alive = false;
						break;

					default:
						break;
				}
			}
			return true;
		}

		virtual bool handle(ui::EventMisc &ev, ui::UiStack &st)
		{
			switch(ev.getType())
			{
				case ui::QUIT:
					_alive = false;
					break;

				default:
					break;
			}
			return true;
		}

		virtual bool handle(ui::EventMouseButton &ev, ui::UiStack &st)
		{
			return true;
		}

		virtual bool handle(ui::EventMouseMotion &ev, ui::UiStack &st)
		{
			return true;
		}

		virtual void update(gfx::SurfaceScreen &screen, ui::UiStack &st, bool prender)
		{
			if(prender)
			{
				thr::wait_privileged(boost::bind(&TimerDisp::render, *this,
							boost::ref(screen), boost::ref(st)));
			}
		}
		/** \cond */
};

int main(int argc, char **argv)
{
	try
	{
		unsigned w = 800,
						 h = 600,
						 b = 32;
		bool fullscreen = false;
		std::string fntfile;

		// Options in own block for destruction.
		if(argc > 0)
		{
			po::options_description desc("Options");
			desc.add_options()
				("font,n", po::value<std::string>(), "Font to use (other than default).")
				("fullscreen,f", "Full-screen mode instead of window.")
				("help,h", "Print help text.")
				("resolution,r", po::value<std::string>(), "Resolution to use.");

			po::variables_map vmap;
			po::store(po::parse_command_line(argc, argv, desc), vmap);
			po::notify(vmap);

			if(vmap.count("font"))
			{
				fntfile = vmap["outfile"].as<std::string>();
			}
			if(vmap.count("fullscreen"))
			{
				fullscreen = true;
			}
			if(vmap.count("help"))
			{
				std::cout << usage << desc << std::endl;
				return 0;
			}
			if(vmap.count("resolution"))
			{
				boost::tie(w, h, b) = gfx::SurfaceScreen::parseResolution(
						vmap["resolution"].as<std::string>());
			}
		}

		gfx::SurfaceScreen scr(w, h, b, fullscreen);

		// Must do after screen reservation due to OpenGL texture stuff.
		gfx::Font *fnt = NULL;
		{
			// Custom font file.
			if(fntfile.length() > 0)
			{
				try
				{
					fnt = new gfx::Font(fntfile);
				}
				catch(std::runtime_error err)
				{
					std::cerr << err.what() << std::endl;
				}
			}
			// Default font file.
			if(!fnt)
			{
				try
				{
					fnt = new gfx::Font(std::string("fnt/default.xml"));
				}
				catch(std::runtime_error err)
				{
					std::cerr << err.what() << std::endl;
				}
			}
		}

		if(!fnt)
		{
			std::cerr << "Could not find a suitable font.\n";
			return EXIT_FAILURE;
		}

		ui::UiStack stack(scr, 1000);
		stack.pushState(new TimerDisp(*fnt));
		boost::thread stack_thread(boost::bind(&ui::UiStack::run, &stack));
		thr::thr_main(1);
		stack_thread.join();
		delete fnt;
	}
	catch(const std::exception & e)
	{
		std::cerr << argv[0] << ": " << e.what() << std::endl;
		return -1;
	}

	return EXIT_SUCCESS;
}

