#include "snd/sample.hpp"

#include "data/generic.hpp"
#include "thr/generic.hpp"
#include "ui/generic.hpp"

#include <boost/filesystem.hpp>
#include <boost/property_tree/xml_parser.hpp>

using namespace snd;
namespace fs = boost::filesystem;
namespace pt = boost::property_tree;

/** Time delay in microseconds after which to allow samples again. */
static const uint64_t SAMPLE_BLOCK_TIME = 200000;

SampleStore snd::sample_store;

Sample::Sample() :
	m_buffer(0),
	m_last_play_time(0),
	m_gain(1.0f) { }

Sample::Sample(const char *pfname, float gain) :
	m_buffer(0),
	m_gain(gain)
{
	this->load(std::string(pfname));
}

Sample::Sample(const std::string &pfname, float gain) :
	m_buffer(0),
	m_last_play_time(0),
	m_gain(gain)
{
	this->load(pfname);
}

Sample::Sample(const uint8_t *data, unsigned size, unsigned freq,
		float gain) :
	m_buffer(0),
	m_last_play_time(0),
	m_gain(gain)
{
	this->generateBuffer();
	alBufferData(m_buffer, AL_FORMAT_STEREO16, data, size, freq);
}

Sample::~Sample()
{
	this->unreserve();
}

bool Sample::canPlay()
{
	uint64_t ts = thr::usec_get_timestamp();
	if(ts - m_last_play_time < SAMPLE_BLOCK_TIME)
	{
		return false;
	}
	m_last_play_time = ts;
	return true;
}

void Sample::generateBuffer()
{
	alGenBuffers(1, &m_buffer);
	if(0 == m_buffer)
	{
		std::stringstream sstr;
		sstr << "could not create audio buffer";
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}
}

void Sample::load(const std::string &pfname)
{
	this->unreserve();

	ui::log(std::string("loading sample '") + pfname + std::string("'"));

	this->generateBuffer();

	SDL_AudioSpec spec;
	uint8_t *buf;
	uint32_t len;

	if(&spec != SDL_LoadWAV(pfname.c_str(), &spec, &buf, &len))
	{
		std::stringstream sstr;
		sstr << "could not load wav file: '" << pfname << "': " <<
			SDL_GetError();
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}
	ALenum format;
	switch(spec.format)
	{
		case AUDIO_U8:
			format = (spec.channels > 1) ? AL_FORMAT_STEREO8 : AL_FORMAT_MONO8;
			break;

		case AUDIO_S8:
			format = (spec.channels > 1) ? AL_FORMAT_STEREO8 : AL_FORMAT_MONO8;
			break;

		case AUDIO_U16SYS:
			format = (spec.channels > 1) ? AL_FORMAT_STEREO16 : AL_FORMAT_MONO16;
			break;
	
		case AUDIO_S16SYS:
			format = (spec.channels > 1) ? AL_FORMAT_STEREO16 : AL_FORMAT_MONO16;
			break;

		default:
			{
				std::stringstream sstr;
				sstr << "unknown audio format: " << spec.format;
				BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
			}
			break;
	}

#if defined(DEBUG)
	std::cout << "Sample: " << size << " bytes, " << freq << " Hz, ";
	switch(format)
	{
		case AL_FORMAT_MONO8:
			std::cout << "8-bit mono";
			break;

		case AL_FORMAT_STEREO8:
			std::cout << "8-bit stereo";
			break;

		case AL_FORMAT_MONO16:
			std::cout << "16-bit mono";
			break;

		case AL_FORMAT_STEREO16:
			std::cout << "16-bit stereo";
			break;
	}
	std::cout << std::endl;
#endif

	alBufferData(m_buffer, format, buf, len, spec.freq);
	SDL_FreeWAV(buf);
}

void Sample::unreserve()
{
	if(0 < m_buffer)
	{
		alDeleteBuffers(1, &m_buffer);
		m_buffer = 0;
	}
}

SampleStore::SampleStore() :
	data::Store<Sample>(".wav", ".msnd") { }

void SampleStore::loadMeta(std::vector<SampleSptr> &dst,
		const std::string &pfname, const data::StoreCreator<Sample> &creator)
{
	ui::log(std::string("loading sample listing '") + pfname + std::string("'"));

	boost::property_tree::ptree xtree;
	boost::property_tree::read_xml(pfname, xtree);
	boost::property_tree::ptree::value_type root = xtree.front();
	if(root.first.compare("sample-list"))
	{
		std::stringstream err;
		err << "unknown root type: " << root.first;
		BOOST_THROW_EXCEPTION(std::invalid_argument(err.str()));
	}

	BOOST_FOREACH(boost::property_tree::ptree::value_type &vv, root.second)
	{
		const std::string &type = vv.first;
		const boost::property_tree::ptree &subtree = vv.second;

		if(!type.compare("sample"))
		{
			std::string single_fname = subtree.get<std::string>("file");
			fs::path subpath(fs::path(pfname).parent_path() /
					(single_fname + m_single));
			std::string subfilename = subpath.file_string();
			Sample *item = creator.create(subfilename);
			// Attributes are optional. 
			try {
				item->setGain(item->getGain() * subtree.get<float>("gain"));
			} catch(...) { }
			this->put(canonize(subfilename), item);
		}
	}
}

//snd::StoreCreatorSingle<snd::Sample, snd::Sample, float> pb;

