#ifndef MATH_VEC_HPP
#define MATH_VEC_HPP

#include "math/generic.hpp"

namespace math
{
	/** \brief Generic vector type. */
	template<unsigned S, typename T> struct vec
	{
		/** STL lookalike. */
		typedef T value_type;

		/** STL lookalike. */
		typedef T* iterator;

		/** Internal information. */
		static const unsigned static_size = S;

		/** Actual data. */
		T _array[S];

		/** \brief STL lookalike.
		 *
		 * @return Size of this.
		 */
		static inline unsigned size()
		{
			return S;
		}

		/** \brief STL lookalike.
		 *
		 * @return Begin iterator.
		 */
		inline iterator begin()
		{
			return _array;
		}

		/** \brief STL lookalike.
		 *
		 * @return Begin iterator.
		 */
		inline iterator begin() const
		{
			return _array;
		}

		/** \brief STL lookalike.
		 *
		 * @return End iterator.
		 */
		inline iterator end()
		{
			return _array + S;
		}

		/** \brief STL lookalike.
		 *
		 * @return End iterator.
		 */
		inline iterator end() const
		{
			return _array + S;
		}

		/** \brief Empty constructor. */
		inline vec() { }

		/** \brief Fill constructor.
		 *
		 * @param v Source.
		 */
		inline vec(const T &v)
		{
			std::fill(_array, _array + S, v);
		}

		/** \brief Copy constructor.
		 *
		 * @param v Source.
		 */
		inline vec(const T *v)
		{
			std::copy(v, v + S, _array);
		}

		/** \brief Copy constructor.
		 *
		 * @param other Source.
		 */
		inline vec(const vec<S,T> &other)
		{
			std::copy(other._array, other._array + S, _array);
		}

		/** \brief Copy from vector of another type.
		 *
		 * @param other Source.
		 */
		template<unsigned S2> inline vec(const vec<S2,T> &other)
		{
			if(S <= S2)
			{
				for(unsigned i = 0; (i != S); ++i)
				{
					std::copy(other._array, other._array + S, _array);
				}
			}
			else
			{
				std::copy(other._array, other._array + S2, _array);
				std::fill(_array + S2, _array + S, static_cast<T>(0));
			}
		}

		/** \brief Copy from vector of another type.
		 *
		 * FIXME: isn't this already explicit - the keyword does not work.
		 *
		 * @param other Source.
		 */
		template<typename T2> inline vec(const vec<S, T2> &other)
		{
			for(unsigned ii = 0; (ii < S); ++ii)
			{
				_array[ii] = static_cast<T>(other._array[ii]);
			}
		}

		/** \brief Access operator.
		 *
		 * @param i Index to fetch.
		 * @return i:th value.
		 */
		inline value_type& operator[](unsigned i) 
		{
			return _array[i];
		}

		/** \brief Access operator.
		 *
		 * @param i Index to fetch.
		 * @return i:th value.
		 */
		inline const value_type& operator[](unsigned i) const
		{
			return _array[i];
		}

		/** \brief Access operator.
		 *
		 * Same as operator[], but throws an error if out of range.
		 * 
		 * @param i Index to fetch.
		 * @return i:th value.
		 */
		inline value_type & at(unsigned i)
		{
			if(i >= S)
			{
        BOOST_THROW_EXCEPTION(std::range_error("index out of range"));
			}
			return _array[i];
		}

		/** \brief Access operator.
		 *
		 * Same as operator[], but throws an error if out of range.
		 * 
		 * @param i Index to fetch.
		 * @return i:th value.
		 */
		inline const value_type& at(unsigned i) const
		{
			if(i >= S)
			{
        BOOST_THROW_EXCEPTION(std::range_error("index out of range"));
			}
			return _array[i];
		}

		/** \brief Component-wise multiplication assign
		 *
		 * @param other Other operand.
		 * @return This after operation.
		 */
		inline vec<S,T>& operator*=(const vec<S,T> &other)
		{
			for(unsigned ii = 0; (ii != S); ++ii)
			{
				_array[ii] *= other._array[ii];
			}
			return *this;
		}

		/** \brief Scalar multiplication.
		 *
		 * @param other Other operand.
		 * @return This after operation.
		 */
		inline vec<S,T>& operator*=(const T &scalar)
		{
			for(unsigned ii = 0; (ii != S); ++ii)
			{
				_array[ii] *= scalar;
			}
			return *this;
		}

		/** \brief Component-wise division assign.
		 *
		 * @param other Other operand.
		 * @return This after operation.
		 */
		inline vec<S,T>& operator/=(const vec<S,T> &other)
		{
			for(unsigned i = 0; (i != S); ++i)
			{
				_array[i] /= other._array[i];
			}
			return *this;
		}

		/** \brief Scalar division assign.
		 *
		 * @param other Other operand.
		 * @return This after operation.
		 */
		inline vec<S,T>& operator/=(const T &scalar)
		{
			for(unsigned i = 0; (i != S); ++i)
			{
				_array[i] /= scalar;
			}
			return *this;
		}

		/** \brief Component-wise addition assign.
		 *
		 * @param other Other operand.
		 * @return This after operation.
		 */
		inline vec<S,T>& operator+=(const vec<S,T> &other)
		{
			for(unsigned i = 0; (i != S); ++i)
			{
				_array[i] += other._array[i];
			}
			return *this;
		}

		/** \brief Scalar addition assign.
		 *
		 * @param other Other operand.
		 * @return This after operation.
		 */
		inline vec<S,T>& operator+=(const T &scalar)
		{
			for(unsigned i = 0; (i != S); ++i)
			{
				_array[i] += scalar;
			}
			return *this;
		}

		/** \brief Component-wise substraction assign.
		 *
		 * @param other Other operand.
		 * @return This after operation.
		 */
		inline vec<S,T>& operator-=(const vec<S,T> &other)
		{
			for(unsigned i = 0; (i != S); ++i)
			{
				_array[i] -= other._array[i];
			}
			return *this;
		}

		/** \brief Scalar substraction assign.
		 *
		 * @param other Other operand.
		 * @return This after operation.
		 */
		inline vec<S,T>& operator-=(const T &scalar)
		{
			for(unsigned i = 0; (i != S); ++i)
			{
				_array[i] -= scalar;
			}
			return *this;
		}

		/** \brief Negate.
		 *
		 * @return Result.
		 */
		inline vec<S,T> operator-() const
		{
			vec<S,T> ret;
			for(unsigned i = 0; (i != S); ++i)
			{
				ret._array[i] = -_array[i];
			}
			return ret;
		}

		/** \brief Component-wise addition.
		 *
		 * @param other Other operand.
		 * @return Result.
		 */
		inline vec<S,T> operator+(const vec<S,T> &other) const
		{
			vec<S,T> ret;
			for(unsigned i = 0; (i != S); ++i)
			{
				ret._array[i] = _array[i] + other._array[i];
			}
			return ret;    
		}

		/** \brief Scalar addition.
		 *
		 * @param other Other operand.
		 * @return Result.
		 */
		inline vec<S,T> operator+(const T &scalar) const
		{
			vec<S,T> ret;
			for(unsigned i = 0; (i != S); ++i)
			{
				ret._array[i] = _array[i] + scalar;
			}
			return ret;
		}

		/** \brief Component-wise substraction.
		 *
		 * @param other Other operand.
		 * @return Result.
		 */
		inline vec<S,T> operator-(const vec<S,T> &other) const
		{
			vec<S,T> ret;
			for(unsigned i = 0; (i != S); ++i)
			{
				ret._array[i] = _array[i] - other._array[i];
			}
			return ret;
		}

		/** \brief Scalar substraction.
		 *
		 * @param other Other operand.
		 * @return Result.
		 */
		inline vec<S,T> operator-(const T &scalar) const
		{
			vec<S,T> ret;
			for(unsigned i = 0; (i != S); ++i)
			{
				ret._array[i] = _array[i] - scalar;
			}
			return ret;
		}

		/** \brief Component-wise multiplication.
		 *
		 * @param other Other operand.
		 * @return Result.
		 */
		inline vec<S,T> operator*(const vec<S,T> &other) const
		{
			vec<S,T> ret;
			for(unsigned i = 0; (i != S); ++i)
			{
				ret._array[i] = _array[i] * other._array[i];
			}
			return ret;
		}

		/** \brief Scalar multiplication.
		 *
		 * @param other Other operand.
		 * @return Result.
		 */
		inline vec<S,T> operator*(const T &scalar)
		{
			vec<S,T> ret;
			for(unsigned i = 0; (i != S); ++i)
			{
				ret._array[i] = _array[i] * scalar;
			}
			return ret;
		}

		/** \brief Component-wise division.
		 *
		 * If other has zeroes will generate an error.
		 *
		 * @param other Other operand.
		 * @return Result.
		 */
		inline vec<S,T> operator/(const vec<S,T> &other) const
		{
			vec<S,T> ret;
			for(unsigned i = 0; (i != S); ++i)
			{
				ret._array[i] = _array[i] / other._array[i];
			}
			return ret;
		}

		/** \brief Scalar division.
		 *
		 * @param other Other operand.
		 * @return Result.
		 */
		inline vec<S,T> operator/(const T &scalar) const
		{
			vec<S,T> ret;
			for(unsigned i = 0; (i != S); ++i)
			{
				ret._array[i] = _array[i] / scalar;
			}
			return ret;
		}

		/** \brief Equality comparison.
		 *
		 * @param other Other operand.
		 * @return True if equal, false if not.
		 */
		inline bool operator==(const vec<S,T> &other) const
		{
			for(unsigned i = 0; (i != S); ++i)
			{
				if(_array[i] != other._array[i])
				{
					return false;
				}
			}
			return true;
		}

		/** \brief Inequality comparison.
		 *
		 * @param other Other operand.
		 * @return True if equal, false if not.
		 */
		inline bool operator!=(const vec<S,T> & other) const
		{
			for(unsigned i = 0; (i != S); ++i)
			{
				if(_array[i] == other._array[i])
				{
					return false;
				}
			}
			return true;
		}

		/** \brief Output a vector into a stream.
		 *
		 * @param lhs Stream to output to.
		 * @param rhs Vector to print.
		 */
		friend std::ostream& operator<<(std::ostream &lhs, const vec<S, T> &rhs)
		{
			lhs << "(";
			for(unsigned ii = 0; (ii < S - 1); ++ii)
			{
				lhs << rhs._array[ii] << " ; ";
			}
			return lhs << rhs._array[S - 1] << ")";
		}
	};

	/** \brief Scalar added to a vector.
	 *
	 * @param scalar Scalar.
	 * @param rhs Vector.
	 * @return Result.
	 */
	template<unsigned int S, typename T> inline vec<S,T> operator+(const T &scalar, const vec<S,T> &rhs)
	{
		return rhs + scalar;
	}

	/** \brief Scalar from which a vector is substracted from.
	 *
	 * @param scalar Scalar.
	 * @param rhs Vector.
	 * @return Result.
	 */
	template<unsigned int S, typename T> inline vec<S,T> operator-(const T &scalar, const vec<S,T> &rhs)
	{
		vec<S,T> ret;
		for(unsigned i = 0; (i != S); ++i)
		{
			ret._array[i] = scalar - rhs[i];
		}
		return ret;
	}

	/** \brief Scalar multiplied by vector.
	 *
	 * @param scalar Scalar.
	 * @param rhs Vector.
	 * @return Result.
	 */
	template<unsigned int S, typename T> inline vec<S,T> operator*(const T &scalar, const vec<S,T> &rhs)
	{
		return rhs * scalar;
	}

	/** \brief Two-component vector with symbolic accessors. */
	template<typename T> struct vec2 : vec<2,T>
	{
		private:
			/** Parent type for internal use. */
			typedef vec<2,T> parent;

		public:
			/** \brief Empty constructor.
			*/
			inline vec2() : parent() { }

			/** \brief Initializing constructor.
			 *
			 * @param px First component.
			 * @param py Second component.
			 */
			inline vec2(const T &px, const T &py)
			{
				parent::_array[0] = px;
				parent::_array[1] = py;
			}

			/** \brief Copy constructor.
			 *
			 * @param other Copied object.
			 */
			inline vec2(const parent &other) : parent(other) { }

		public:
			/** \brief Set contents.
			 *
			 * @param px First component.
			 * @param py Second component.
			 */
			inline void set(const T &px, const T &py)
			{
				parent::_array[0] = px;
				parent::_array[1] = py;
			}

		public:
			/** \brief Accessor.
			 *
			 * @return Vector component.
			 */
			inline T& x()
			{
				return parent::_array[0];
			}

			/** \brief Accessor.
			 *
			 * @return Vector component.
			 */
			inline const T& x() const
			{
				return parent::_array[0];
			}

			/** \brief Accessor.
			 *
			 * @return Vector component.
			 */
			inline T& y()
			{
				return parent::_array[1];
			}

			/** \brief Accessor.
			 *
			 * @return Vector component.
			 */
			inline const T& y() const
			{
				return parent::_array[1];
			}
	};

	/** \brief Three-component vector with symbolic accessors. */
	template<typename T> struct vec3 : vec<3,T>
	{
		private:
			/** Parent type for internal use. */
			typedef vec<3,T> parent;

		public:
			/** \brief Empty constructor.
			*/
			inline vec3() : parent() { }

			/** \brief Initializing constructor.
			 *
			 * @param px First component.
			 * @param py Second component.
			 * @param pz Third component.
			 */
			inline vec3(const T &px, const T &py, const T &pz)
			{
				parent::_array[0] = px;
				parent::_array[1] = py;
				parent::_array[2] = pz;
			}

			/** \brief Copy constructor.
			 *
			 * @param other Copied object.
			 */
			inline vec3(const parent &other) : parent(other) { }

		public:
			/** \brief Set contents.
			 *
			 * @param px First component.
			 * @param py Second component.
			 * @param pz Third component.
			 */
			inline void set(const T &px, const T &py, const T &pz)
			{
				parent::_array[0] = px;
				parent::_array[1] = py;
				parent::_array[2] = pz;
			}

		public:
			/** \brief Accessor.
			 *
			 * @return Vector component.
			 */
			inline T& x()
			{
				return parent::_array[0];
			}

			/** \brief Accessor.
			 *
			 * @return Vector component.
			 */
			inline const T& x() const
			{
				return parent::_array[0];
			}

			/** \brief Accessor.
			 *
			 * @return Vector component.
			 */
			inline T& y()
			{
				return parent::_array[1];
			}

			/** \brief Accessor.
			 *
			 * @return Vector component.
			 */
			inline const T& y() const
			{
				return parent::_array[1];
			}

			/** \brief Accessor.
			 *
			 * @return Vector component.
			 */
			inline T& z()
			{
				return parent::_array[2];
			}

			/** \brief Accessor.
			 *
			 * @return Vector component.
			 */
			inline const T& z() const
			{
				return parent::_array[2];
			}
	};

	/** Four-component vector with symbolic accessors. */
	template<typename T> struct vec4 : vec<4,T>
	{
		private:
			/** Parent type for internal use. */
			typedef vec<4,T> parent;

		public:
			/** \brief Empty constructor. */
			inline vec4() : parent() { }

			/** \brief Initializing constructor.
			 *
			 * @param px First component.
			 * @param py Second component.
			 * @param pz Third component.
			 * @param pw Fourth component.
			 */
			inline vec4(const T &px, const T &py, const T &pz, const T &pw)
			{
				parent::_array[0] = px;
				parent::_array[1] = py;
				parent::_array[2] = pz;
				parent::_array[3] = pw;
			}

			/** \brief Copy constructor.
			 *
			 * @param other Copied object.
			 */
			inline vec4(const parent &other) : parent(other) { }

		public:
			/** \brief Set contents.
			 *
			 * @param px First component.
			 * @param py Second component.
			 * @param pz Third component.
			 * @param pw Fourth component.
			 */
			inline void set(const T &px, const T &py, const T &pz, const T &pw)
			{
				parent::_array[0] = px;
				parent::_array[1] = py;
				parent::_array[2] = pz;
				parent::_array[3] = pw;
			}

		public:
			/** \brief Accessor.
			 *
			 * @return Vector component.
			 */
			inline T& x()
			{
				return parent::_array[0];
			}

			/** \brief Accessor.
			 *
			 * @return Vector component.
			 */
			inline const T& x() const
			{
				return parent::_array[0];
			}

			/** \brief Accessor.
			 *
			 * @return Vector component.
			 */
			inline T& y()
			{
				return parent::_array[1];
			}

			/** \brief Accessor.
			 *
			 * @return Vector component.
			 */
			inline const T& y() const
			{
				return parent::_array[1];
			}

			/** \brief Accessor.
			 *
			 * @return Vector component.
			 */
			inline T& z()
			{
				return parent::_array[2];
			}

			/** \brief Accessor.
			 *
			 * @return Vector component.
			 */
			inline const T& z() const
			{
				return parent::_array[2];
			}

			/** \brief Accessor.
			 *
			 * @return Vector component.
			 */
			inline T& w()
			{
				return parent::_array[3];
			}

			/** \brief Accessor.
			 *
			 * @return Vector component.
			 */
			inline const T& w() const
			{
				return parent::_array[3];
			}
	};

	/** Convenience typedef. */
	typedef vec2<double> vec2d;

	/** Convenience typedef. */
	typedef vec3<double> vec3d;

	/** Convenience typedef. */
	typedef vec2<float> vec2f;

	/** Convenience typedef. */
	typedef vec3<float> vec3f;

	/** Convenience typedef. */
	typedef vec4<float> vec4f;

	/** Convenience typedef. */
	typedef vec2<int> vec2i;

	/** Convenience typedef. */
	typedef vec3<int> vec3i;

	/** Convenience typedef. */
	typedef vec4<int> vec4i;

	/** Convenience typedef. */
	typedef vec4<unsigned> vec4u;

	/** \brief Dot product.
	 *
	 * @param lhs Left-hand-side operand.
	 * @param rhs Right-hand-side operand.
	 * @return Result.
	 */
	template<typename T> inline typename T::value_type dot(const T &lhs, const T &rhs)
	{
		typename T::value_type ret = static_cast<typename T::value_type>(0);
		for(unsigned i = 0; (i != T::static_size); ++i)
		{
			ret += lhs[i] * rhs[i];
		}
		return ret;
	}

	/** \brief Square of of the length of a vector.
	 *
	 * @param v Vector.
	 * @return Result.
	 */
	template<typename T> inline typename T::value_type length2(const T &v)
	{
		return dot(v, v);
	}

	/** \brief Length of a vector.
	 *
	 * @param v Vector.
	 * @return Result.
	 */
	template<typename T> inline typename T::value_type length(const T &v)
	{
		return sqrt(length2(v));
	}

	/** \brief Normalize to unit vector.
	 *
	 * @param v Vector.
	 * @return Result.
	 */
	template<typename T> inline T normalize(const T &v)
	{
		typename T::value_type l = length(v);
		return v / l;
	}

	/** \brief Component-wise min.
	 *
	 * @param lhs Left-hand-side operand.
	 * @param rhs Right-hand-side operand.
	 * @return Result.
	 */
	template<typename T> inline T vec_min(const T &lhs, const T &rhs)
	{
		T ret;
		for(unsigned i = 0; (i != T::static_size); ++i)
		{
			ret[i] = std::min(lhs[i], rhs[i]);
		}
		return ret;
	}
	/** \cond */
	template<> inline math::vec2f min(const vec2f &lhs, const vec2f &rhs)
	{
		return vec_min(lhs, rhs);
	}
	template<> inline math::vec3f min(const vec3f &lhs, const vec3f &rhs)
	{
		return vec_min(lhs, rhs);
	}
	template<> inline math::vec4f min(const vec4f &lhs, const vec4f &rhs)
	{
		return vec_min(lhs, rhs);
	}
	template<> inline math::vec2i min(const vec2i &lhs, const vec2i &rhs)
	{
		return vec_min(lhs, rhs);
	}
	template<> inline math::vec3i min(const vec3i &lhs, const vec3i &rhs)
	{
		return vec_min(lhs, rhs);
	}
	template<> inline math::vec4i min(const vec4i &lhs, const vec4i &rhs)
	{
		return vec_min(lhs, rhs);
	}
	template<> inline math::vec4u min(const vec4u &lhs, const vec4u &rhs)
	{
		return vec_min(lhs, rhs);
	}
	/** \endcond */

	/** \brief Component-wise max.
	 *
	 * @param lhs Left-hand-side operand.
	 * @param rhs Right-hand-side operand.
	 * @return Result.
	 */
	template<typename T> inline T vec_max(const T &lhs, const T &rhs)
	{
		T ret;
		for(unsigned i = 0; (i != T::static_size); ++i)
		{
			ret[i] = std::max(lhs[i], rhs[i]);
		}
		return ret;
	}
	/** \cond */
	template<> inline math::vec2f max(const vec2f &lhs, const vec2f &rhs)
	{
		return vec_max(lhs, rhs);
	}
	template<> inline math::vec3f max(const vec3f &lhs, const vec3f &rhs)
	{
		return vec_max(lhs, rhs);
	}
	template<> inline math::vec4f max(const vec4f &lhs, const vec4f &rhs)
	{
		return vec_max(lhs, rhs);
	}
	template<> inline math::vec2i max(const vec2i &lhs, const vec2i &rhs)
	{
		return vec_max(lhs, rhs);
	}
	template<> inline math::vec3i max(const vec3i &lhs, const vec3i &rhs)
	{
		return vec_max(lhs, rhs);
	}
	template<> inline math::vec4i max(const vec4i &lhs, const vec4i &rhs)
	{
		return vec_max(lhs, rhs);
	}
	template<> inline math::vec4u max(const vec4u &lhs, const vec4u &rhs)
	{
		return vec_max(lhs, rhs);
	}
	/** \endcond */

	/** \brief Get the angle of a vector in assumed plane.
	 *
	 * Positive x, zero y is 0, positive y, zero x is M_PI / 2.
	 *
	 * @param op Vector to calculate angle on.
	 * @return Angle in the range [0, 2PI[.
	 */
	template<typename T> inline T angle(const vec2<T> &op)
	{
		if(static_cast<T>(0) < op.x())
		{
			T ret = atan(op.y() / op.x());
			if(ret - static_cast<T>(0))
			{
				ret += static_cast<T>(M_PI) * static_cast<T>(2);
			}
			return ret;
		}
		if(static_cast<T>(0) > op.x())
		{
			return static_cast<T>(M_PI) + atan(op.y() / op.x());
		}
		return (static_cast<T>(0) < op.y()) ?
			static_cast<T>(M_PI_2) :
			static_cast<T>(M_PI_2 + M_PI_4);
	}

	/** \brief Cross product of 3-dimensional vectors.
	 *
	 * @param lhs Left-hand-side operand.
	 * @param rhs Right-hand-side operand.
	 * @return Result.
	 */
	template<typename T> inline vec<3, T> cross(const vec<3, T> &lhs, const vec<3, T> &rhs)
	{
		vec<3,T> ret;
		ret[0] = (lhs[1] * rhs[2]) - (lhs[2] * rhs[1]);
		ret[1] = (lhs[2] * rhs[0]) - (lhs[0] * rhs[2]);
		ret[2] = (lhs[0] * rhs[1]) - (lhs[1] * rhs[0]);
		return ret;
	}

	/** \brief Distance squared between two points.
	 *
	 * @param lhs Left-hand-side operand.
	 * @param rhs Right-hand-side operand.
	 * @return Result.
	 */
	template<typename T> inline typename T::value_type dist2_point_point(const T &lhs, const T &rhs)
	{
		return length2(lhs - rhs);
	}

	/** \brief Distance Between two points.
	 *
	 * @param lhs Left-hand-side operand.
	 * @param rhs Right-hand-side operand.
	 * @return Result.
	 */
	template<typename T> inline typename T::value_type dist_point_point(const T &lhs, const T &rhs)
	{
		return math::sqrt(dist2_point_point(lhs, rhs));
	}

	/** \brief Squared distance between a line and a point.
	 *
	 * Algorithm from Wolfram MathWorld:
	 * http://mathworld.wolfram.com/Point-LineDistance3-Dimensional.html
	 *
	 * @param l1 First line point 1.
	 * @param l2 First line point 2.
	 * @param pp Point.
	 * @return True if collide, false if not.
	 */
	template <typename T> inline T dist2_line_point(const math::vec3<T> &l1,
			const math::vec3<T> &l2, const math::vec3<T> &pp)
	{
		math::vec3<T> l2l1 = l2 - l1;
		math::vec3<T> l1pp = l1 - pp;
		T len2 = math::length2(l2l1);

		// Reduction to point-point.
		if(len2 <= static_cast<T>(FLT_EPSILON))
		{
			return math::length2(l1pp);
		}

		float tt = -math::dot(l1pp, l2l1) / len2;

		// Closer to l1.
		if(tt < static_cast<T>(0))
		{
			return math::length2(l1pp);
		}
		// Closer to l2.
		if(tt > static_cast<T>(1))
		{
			return math::length2(l2 - pp);
		}
		// General case.
		return math::length2(math::cross(l2l1, l1pp)) / len2;
	}

	/** \brief Distance between a line and a and point.
	 *
	 * @param l1 First line point 1.
	 * @param l2 First line point 2.
	 * @param pp Point.
	 * @return True if collide, false if not.
	 */
	template <typename T> inline T dist_line_point(const math::vec3<T> &l1,
			const math::vec3<T> &l2, const math::vec3<T> &pp)
	{
		return math::sqrt(dist2_line_point(l1, l2, pp));
	}

	/** \brief Calculate reflection vector.
	 *
	 * @param v Ray direction (coming towards origo).
	 * @param n Normal (starting from origo).
	 * @return Result.
	 */
	template<typename T> inline T reflect(const T &v, const T &n)
	{
		return v - 2 * dot(n, v) * n;
	}

	/** \brief Ray intersection.
	 *
	 * Returns the first intersection (not second).
	 *
	 * Algorithm from Scott Owen / SIGGRAPH:
	 * http://www.siggraph.org/education/materials/HyperGraph/raytrace/rtinter1.htm
	 *
	 * @param dst Intersect position.
	 * @param pos Position.
	 * @param dir Direction.
	 * @param bpos Ball position.
	 * @param brad Ball radius
	 * @return True if intersected, false if not.
	 */
	template<typename T> bool intersect_ray_ball(vec3<T> &intersect,
			const vec3<T> &pos, const vec3<T> &dir, const vec3<T> &bpos,
			const T &brad)
	{
		vec3<T> p_b = pos - bpos;
		T aa = length2(dir);
		T bb = dot(dir, p_b) * static_cast<T>(2);
		T cc = length2(p_b) - brad * brad;

		T disc = bb * bb - static_cast<T>(4) * aa * cc;

		// No roots -> no solutions.
		if(static_cast<T>(0) > disc)
		{
			return false;
		}

		// One root -> one solution.
		if(static_cast<T>(0) == disc)
		{
			T tt = -bb * static_cast<T>(0.5);
			if(tt >= static_cast<T>(0))
			{
				intersect = pos + dir * tt;
				return true;
			}
			return false;
		}

		// Two roots -> two solutions.
		disc = sqrt(disc);
		T tt1 = (-bb - disc) * static_cast<T>(0.5);
		T tt2 = (-bb + disc) * static_cast<T>(0.5);
		if(tt1 >= static_cast<T>(0))
		{
			if(tt2 >= static_cast<T>(0))
			{
				intersect = pos + dir * min(tt1, tt2);
				return true;
			}
			intersect = pos + dir * tt1;
			return true;
		}
		if(tt2 >= static_cast<T>(0))
		{
			intersect = pos + dir * tt2;
			return true;
		}
		return false;

		/*b = 
		
		c = pos.x()*pos.x() + bpos.x()*bpos.x() + pos.y()*pos.y() + bpos.y()*bpos.y() + pos.z()*pos.z() + bpos.z()*bpos.z() - 2*(pos.x()*bpos.x() + pos.y()*bpos.y() + pos.z()*bpos.z());


		a = (x2 - x1)2 + (y2 - y1)2 + (z2 - z1)2

		b = 2[ (x2 - x1) (x1 - x3) + (y2 - y1) (y1 - y3) + (z2 - z1) (z1 - z3) ]

		c = x32 + y32 + z32 + x12 + y12 + z12 - 2[x3 x1 + y3 y1 + z3 z1] - r2 */
	}
}

#endif
