#ifndef MATH_RANDOM_HPP
#define MATH_RANDOM_HPP

#include "defaults.hpp"

#include <boost/random/mersenne_twister.hpp>
#include <boost/random/uniform_int.hpp>
#include <boost/random/uniform_real.hpp>
#include <boost/random/variate_generator.hpp>

namespace math
{
	/** Convenience typedef. */
	typedef boost::mt19937 mrgen;

	/** The global random number generator. */
	extern mrgen global_mrgen;

	/** \brief Generate a random number.
	 *
	 * Uniform distribution, inclusive.
	 *
	 * @param rmin Minimum value.
	 * @param rmax Maximum value.
	 */
	inline int mrand(int rmin, int rmax)
	{
		boost::uniform_int<> distribution(rmin, rmax);
		boost::variate_generator<mrgen&, boost::uniform_int<> >
			randomize(global_mrgen, distribution);
		return randomize();
	}

	/** \brief Generate a random number.
	 *
	 * Uniform distribution, min inclusive, max exclusive.
	 *
	 * The exclusiveness of max is not numerically significant.
	 *
	 * @param rmin Minimum value.
	 * @param rmax Maximum value.
	 */
	inline float mrand(float rmin, float rmax)
	{
		boost::uniform_real<> distribution(static_cast<double>(rmin),
				static_cast<double>(rmax));
		boost::variate_generator<mrgen&, boost::uniform_real<> >
			randomize(global_mrgen, distribution);
		return static_cast<float>(randomize());
	}

	/** \brief Generate a random number.
	 *
	 * Uniform distribution, min inclusive, max exclusive.
	 *
	 * The exclusiveness of max is not numerically significant.
	 *
	 * @param rmin Minimum value.
	 * @param rmax Maximum value.
	 */
	inline double mrand(double rmin, double rmax)
	{
		boost::uniform_real<> distribution(rmin, rmax);
		boost::variate_generator<mrgen&, boost::uniform_real<> >
			randomize(global_mrgen, distribution);
		return randomize();
	}
}

#endif
