#ifndef GFX_UNIFORM_HPP
#define GFX_UNIFORM_HPP

#include "gfx/attribute.hpp"
#include "math/mat.hpp"

namespace gfx
{
	/** \brief Uniform shader variable.
	 *
	 * Uniforms are not type-checked due to the fract that OpenGL allows them to
	 * be specified as types different from what they were specified in the
	 * shader.
	 */
	struct Uniform : public Attribute
	{
		public:
			/** \brief Empty constructor. */
			inline Uniform() { }

			/** \brief Constructor.
			 *
			 * @param src Previous incarnation.
			 * @param pid New id.
			 */
			inline Uniform(const Uniform &src, GLuint pid) :
				Attribute(src, pid) { }

			/** \brief Constructor.
			 *
			 * @param pname Name
			 * @param ptype Type.
			 */
			inline Uniform(const std::string &pname, const std::string &ptype) :
				Attribute(pname, ptype) { }

		public:
			/** \param Supply uniform data.
			 *
			 * @param p1 First parameter.
			 */
			inline void update(int p1) const
			{
				glUniform1i(_id, p1);
			}

			/** \param Supply uniform data.
			 *
			 * @param p1 First parameter.
			 */
			inline void update(float p1) const
			{
				glUniform1f(_id, p1);
			}

			/** \param Supply uniform data.
			 *
			 * @param p1 First parameter.
			 * @param p2 Second parameter.
			 */
			inline void update(int p1, int p2) const
			{
				glUniform2i(_id, p1, p2);
			}

			/** \param Supply uniform data.
			 *
			 * @param p1 First parameter.
			 * @param p2 Second parameter.
			 */
			inline void update(float p1, float p2) const
			{
				glUniform2f(_id, p1, p2);
			}

			/** \param Supply uniform data.
			 *
			 * @param p1 First parameter.
			 * @param p2 Second parameter.
			 * @param p3 Third parameter.
			 */
			inline void update(int p1, int p2, int p3) const
			{
				glUniform3i(_id, p1, p2, p3);
			}

			/** \param Supply uniform data.
			 *
			 * @param p1 First parameter.
			 * @param p2 Second parameter.
			 * @param p3 Third parameter.
			 */
			inline void update(float p1, float p2, float p3) const
			{
				glUniform3f(_id, p1, p2, p3);
			}

			/** \param Supply uniform data.
			 *
			 * @param p1 First parameter.
			 * @param p2 Second parameter.
			 * @param p3 Third parameter.
			 * @param p4 Fourth parameter.
			 */
			inline void update(int p1, int p2, int p3, int p4) const
			{
				glUniform4i(_id, p1, p2, p3, p4);
			}

			/** \param Supply uniform data.
			 *
			 * @param p1 First parameter.
			 * @param p2 Second parameter.
			 * @param p3 Third parameter.
			 * @param p4 Fourth parameter.
			 */
			inline void update(float p1, float p2, float p3, float p4) const
			{
				glUniform4f(_id, p1, p2, p3, p4);
			}

			/** \param Supply uniform data.
			 *
			 * @param p1 First two parameters.
			 * @param p2 Latter two parameters.
			 */
			inline void update(const math::vec2f &p1, const math::vec2f &p2) const
			{
				glUniform4f(_id, p1.x(), p1.y(), p2.x(), p2.y());
			}

			/** \param Update one vector.
			 *
			 * @param op Vector to update.
			 */
			inline void update(const math::vec2f &op) const
			{
				glUniform2fv(_id, 1, op._array);
			}

			/** \param Update one vector.
			 *
			 * @param op Vector to update.
			 */
			inline void update(const math::vec2i &op) const
			{
				glUniform2iv(_id, 1, reinterpret_cast<const GLint*>(op._array));
			}

			/** \param Update one vector.
			 *
			 * @param op Vector to update.
			 */
			inline void update(const math::vec3f &op) const
			{
				glUniform3fv(_id, 1, op._array);
			}

			/** \param Update one vector.
			 *
			 * @param op Vector to update.
			 */
			inline void update(const math::vec3i &op) const
			{
				glUniform3iv(_id, 1, reinterpret_cast<const GLint*>(op._array));
			}

			/** \param Update one vector.
			 *
			 * @param op Vector to update.
			 */
			inline void update(const math::vec4f &op) const
			{
				glUniform4fv(_id, 1, op._array);
			}

			/** \param Update one vector.
			 *
			 * @param op Vector to update.
			 */
			inline void update(const math::vec4i &op) const
			{
				glUniform4iv(_id, 1, reinterpret_cast<const GLint*>(op._array));
			}

			/** \brief Update vector array.
			 *
			 * @param count Number of vectors to update.
			 * @param array Array of vectors.
			 */
			inline void update(unsigned count, const math::vec2f *array) const
			{
				glUniform2fv(_id, count, array[0]._array);
			}

			/** \brief Update vector array.
			 *
			 * @param count Number of vectors to update.
			 * @param array Array of vectors.
			 */
			inline void update(unsigned count, const math::vec2i *array) const
			{
				glUniform2iv(_id, count, reinterpret_cast<const GLint*>(array[0]._array));
			}

			/** \brief Update vector array.
			 *
			 * @param count Number of vectors to update.
			 * @param array Array of vectors.
			 */
			inline void update(unsigned count, const math::vec3f *array) const
			{
				glUniform3fv(_id, count, array[0]._array);
			}

			/** \brief Update vector array.
			 *
			 * @param count Number of vectors to update.
			 * @param array Array of vectors.
			 */
			inline void update(unsigned count, const math::vec3i *array) const
			{
				glUniform3iv(_id, count, reinterpret_cast<const GLint*>(array[0]._array));
			}

			/** \brief Update vector array.
			 *
			 * @param count Number of vectors to update.
			 * @param array Array of vectors.
			 */
			inline void update(unsigned count, const math::vec4f *array) const
			{
				glUniform4fv(_id, count, array[0]._array);
			}

			/** \brief Update vector array.
			 *
			 * @param count Number of vectors to update.
			 * @param array Array of vectors.
			 */
			inline void update(unsigned count, const math::vec4i *array) const
			{
				glUniform4iv(_id, count, reinterpret_cast<const GLint*>(array[0]._array));
			}

			/** \brief Update one matrix.
			 *
			 * @param op A matrix.
			 * @param transpose Set to true for transpose.
			 */
			inline void update(const math::mat2f &op, bool transpose = false) const
			{
				glUniformMatrix2fv(_id, 1, transpose, op._array);
			}

			/** \brief Update one matrix.
			 *
			 * @param op A matrix.
			 * @param transpose Set to true for transpose.
			 */
			inline void update(const math::mat3f &op, bool transpose = false) const
			{
				glUniformMatrix3fv(_id, 1, transpose, op._array);
			}

			/** \brief Update one matrix.
			 *
			 * @param op A matrix.
			 * @param transpose Set to true for transpose.
			 */
			inline void update(const math::mat4f &op, bool transpose = false) const
			{
				glUniformMatrix4fv(_id, 1, transpose, op._array);
			}

			/** \brief Update matrix array.
			 *
			 * @param count Number of matrices to update.
			 * @param array Array of matrices.
			 * @param transpose Set to true for transpose.
			 */
			inline void update(unsigned count, const math::mat2f *array, bool transpose = false) const
			{
				glUniformMatrix2fv(_id, count, transpose, array[0]._array);
			}

			/** \brief Update matrix array.
			 *
			 * @param count Number of matrices to update.
			 * @param array Array of matrices.
			 * @param transpose Set to true for transpose.
			 */
			inline void update(unsigned count, const math::mat3f *array, bool transpose = false) const
			{
				glUniformMatrix3fv(_id, count, transpose, array[0]._array);
			}

			/** \brief Update matrix array.
			 *
			 * @param count Number of matrices to update.
			 * @param array Array of matrices.
			 * @param transpose Set to true for transpose.
			 */
			inline void update(unsigned count, const math::mat4f *array, bool transpose = false) const
			{
				glUniformMatrix4fv(_id, count, transpose, array[0]._array);
			}

		public:
			/** \brief Output a surface into a stream.
			 *
			 * @param lhs Left-hand-side operand.
			 * @param rhs Right-hand-side operand.
			 * @return Modified stream.
			 */
			friend inline std::ostream& operator<<(std::ostream &lhs,
					const Uniform &rhs)
			{
				return lhs << "Uniform " << rhs.getType().c_str() << " \"" << rhs.getName().c_str() << "\": " << rhs.id();
			}
	};
}

#endif
