#ifndef GFX_TEXTURE_HPP
#define GFX_TEXTURE_HPP

#include "gfx/uniform.hpp"

namespace gfx
{
	/** \brief Base texture class.
	 */
	template <GLenum T> class Texture
	{
		protected:
			/** Texture identifier. */
			GLuint _id;

		public:
			/** \brief Accessor.
			 *
			 * @return Id of this.
			 */
			inline unsigned id() const
			{
				return _id;
			}

		public:
			/** \brief Bind this texture into use. */
			inline void bind() const
			{
				glBindTexture(T, _id);
			}

			/** \brief Bind this texture to a specific texture unit.
			 *
			 * @param op Texture unit index.
			 */
			inline void bind(unsigned op) const
			{
				glActiveTexture(GL_TEXTURE0 + op);
				glBindTexture(T, _id);
			}

			/** \brief Assign a texture unit and bind this texture to it.
			 *
			 * @param uform Uniform to assign.
			 * @param op Texture unit to use.
			 */
			inline void bind(const Uniform &uform, unsigned op) const
			{
				uform.update(static_cast<int>(op));
				this->bind(op);
			}

			/** Unreserve all resources allocated by this. */
			void unreserve()
			{
				if(_id)
				{
					glDeleteTextures(1, &_id);
					_id = 0;
				}
			}

		public:
			/** \brief Empty constructor. */
			Texture() : _id(0) { }

			/** Destructor. */
			~Texture()
			{
				this->unreserve();
			}

		public:
			/** \brief Transform a pixel bit depth into pixel format.
			 *
			 * There are only few allowed pixel formats, so this is okay.
			 *
			 * @param bpp Bit depth.
			 * @return Pixel format.
			 */
			static GLenum bpp_to_pformat(unsigned bpp)
			{
				switch(bpp)
				{
					case 8:
						return GL_LUMINANCE;

					case 16:
						return GL_LUMINANCE_ALPHA;

					case 24:
						return GL_RGB;

					case 32:
					default:
						return GL_RGBA;
				}
			}
	};
}

#endif
