#ifndef GFX_SURFACE_HPP
#define GFX_SURFACE_HPP

#include "gfx/array.hpp"
#include "gfx/color.hpp"
#include "gfx/light_directional.hpp"
#include "gfx/shader.hpp"
#include "gfx/surface_base.hpp"
#include "math/rect.hpp"

namespace gfx
{
	class Fbo;
	class Font;
	class Glyph;
	class Texture2D;

	/** \brief Drawing surface.
	 *
	 * This top-class surface contains both the basic elements of surfaces and
	 * the general drawing primitives.
	 *
	 * The static variables assigned for the surface are in the class namespace
	 * simply and only so that they should not pollute the gfx:: namespace, as
	 * they are related to drawing and surfaces anyway. The primitive operations
	 * are still on the global gfx namespace for code brevity.
	 *
	 * The class itself is used as an abstraction of any drawable surface,
	 * i.e. screen or texture.
	 */
	class Surface : public SurfaceBase
	{
		public:
			/** Default font boldness factor. */
			static const float DEFAULT_FONT_BOLDNESS;

			/** Default font antialiaising factor. */
			static const float DEFAULT_FONT_ANTIALIAISING;

		protected:
			/** Point sprite buffer length. */
			static const int BILLBOARD_ARRAY_SIZE = 16384;

			/** Space in the primitive drawing buffers. */
			static const int PRIMITIVE_ARRAY_SIZE = 10;

			/** Color buffer for primitive operations. */
			static ArrayA4f array_color;

			/** Vertex buffer for primitive operations. */
			static ArrayA4f array_vertex;

			/** Billboard rendering array. */
			static ArrayBillboard array_billboard;

			/** 2D shader. */
			static Shader *shader_2d;

			/** Font shader. */
			static Shader *shader_2d_font;

			/** Font shader. */
			static Shader *shader_2d_texture;

			/** Shader uniform for fast lookup. */
			static const Uniform *shader_3d_light_ambient;

			/** Shader uniform for fast lookup. */
			static const Uniform *shader_3d_light_diffuse;

			/** Shader uniform for fast lookup. */
			static const Uniform *shader_3d_light_dir;

			/** Shader uniform for fast lookup. */
			static const Uniform *shader_3d_modelview;

			/** Shader uniform for fast lookup. */
			static const Uniform *shader_3d_transform;

			/** Addition to be applied when transforming 2D values. */
			static math::vec2f shader_2d_offset;

			/** Multiplier to be applied when transforming 2D values. */
			static math::vec2f shader_2d_scale;

			/** Projection matrix for 3D shaders. */
			static math::mat4f shader_3d_projection;

			/** Projection matrix for 3D shaders. */
			static math::mat4f shader_3d_stack;

			/** Multiplier to be applied when inputting integer coordinates for
			 * primitives. */
			static float shader_2d_mul;

		public:
			/** \brief Empty constructor. */
			Surface() { }

			/** \brief Constructor.
			 *
			 * @param pw Current width.
			 * @param ph Current height.
			 * @param pb Current bit depth.
			 */
			Surface(unsigned pw, unsigned ph, unsigned pb);

			/** \brief Destructor. */
			virtual ~Surface() { }

		public:
			/** \brief Get the 2D area.
			 *
			 * Returns the floating-point area of this surface.
			 *
			 * @return Rectangle representing the area.
			 */
			math::rect2f getArea() const;

			/** \brief Select this surface for 2D drawing.
			 * 
			 * @param px Left border of clip rectangle.
			 * @param py Bottom border of clip rectangle.
			 * @param pw Width of the clip rectangle.
			 * @param ph Height of the clip rectangle.
			 */
			void select2D(unsigned px, unsigned py, unsigned pw, unsigned ph);

			/** \brief Select this surface for 3D drawing.
			 *
			 * The pixel aspect ratio is intended for displaying correct images on
			 * monitors such as 1280x1024 with 4:3 aspect ratio, wherein the ratio
			 * would be (4/3)/(5/4) = 1.0667
			 *
			 * The view angle is calculated in such a fashion, it is what would be
			 * in effect if the screen was exactly square. non-square aspect ratio
			 * will cause increased field of view to that direction and decreased to
			 * another directio. This is opposed to most systems wherein the angle
			 * is represented along the Y axis.
			 *
			 * @param px Left border of clip rectangle.
			 * @param py Bottom border of clip rectangle.
			 * @param pw Width of the clip rectangle.
			 * @param ph Height of the clip rectangle.
			 * @param pview View angle in radians assuming the screen was square.
			 * @param paspect Pixel aspect ratio (defaults to 1).
			 */
			void select3D(unsigned px, unsigned py, unsigned pw, unsigned ph,
					float pview, float paspect = 1.0f, float pnear = 1.0f,
					float pfar = 1024.0f);

			/** \brief Convert a floating-point area into a pixel area.
			 *
			 * @param parea Input area.
			 * @return Same area in pixel coordinates.
			 */
			math::rect2i toPixelArea(const math::rect2f &parea) const;

			/** \brief Update billboard parameters.
			 *
			 * Take into account the screen estate.
			 *
			 * @param sh Shader to update.
			 * @param divisor_min Divisor for minimum billboard size.
			 * @param divisor_min Divisor for maximum billboard size.
			 */
			void updateBillboardParams(const Shader &sh, float divisor_min,
					float divisor_max);

		public:
			/** \brief Clear the screen.
			 *
			 * @param pc Color buffer clear flag.
			 * @param pd Depth buffer clear flag.
			 */
			virtual void clear(bool pc = true, bool pd = true) = 0;

			/** \brief Set the clip boundary for this surface.
			 * 
			 * @param px Left border of clip rectangle.
			 * @param py Bottom border of clip rectangle.
			 * @param pw Width of the clip rectangle.
			 * @param ph Height of the clip rectangle.
			 */
			virtual void setBoundary(int px, int py, int pw, int ph) = 0;

			/** \brief Update the contents of the screen. */
			virtual void update() = 0;

		public:
			/** \brief Convert a 3D eye coordinate into 2D coordinate.
			 *
			 * @param epos Normalized eye coordinate position.
			 * @return 2D pixel position.
			 */
			inline math::vec2i convertTo2D(const math::vec4f &rpos) const
			{
				math::vec2f rr(rpos.x() / rpos.z(), rpos.y() / rpos.z());
				return math::vec2i(math::lround((rr.x() * 0.5f + 0.5f) * static_cast<float>(m_w)),
						math::lround((rr.y() * 0.5f + 0.5f) * static_cast<float>(m_h)));
			}

			/** \brief Project a real-world coordinate.
			 *
			 * @param rpos Real world coordinate position.
			 * @return 2D pixel position.
			 */
			inline math::vec2i project(const math::vec3f &rpos) const
			{
				math::vec4f tpos(rpos.x(), rpos.y(), rpos.z(), 1.0f);
				std::cout << "Projected position: " << shader_3d_stack * tpos << std::endl;
				return this->convertTo2D(shader_3d_stack * tpos);
			}

			/** \brief Select this surface for 2D drawing.
			 *
			 * A wrapper for 2D selection, uses surface size as the view area.
			 */
			inline void select2D()
			{
				this->select2D(0, 0, m_w, m_h);
			}

			/** \brief Select this surface for 3D drawing.
			 *
			 * A wrapper for 3D selection, uses surface size as the view area.
			 *
			 * @param pview View angle along the shorter axis.
			 * @param paspect Pixel aspect ratio (defaults to 1).
			 */
			inline void select3D(float pview, float paspect = 1.0f, float pnear = 1.0f,
					float pfar = 1024.0f)
			{
				this->select3D(0, 0, m_w, m_h, pview, paspect, pnear, pfar);
			}

		public:
			/** \brief Accessor.
			 *
			 * @return 2d multiplier.
			 */
			static inline float get_mul_2d()
			{
				return shader_2d_mul;
			}

			/** \brief Get the projection matrix.
			 *
			 * @return Projection matrix.
			 */
			static const math::mat4f get_projection()
			{
				return shader_3d_projection;
			}

			/** \brief Accessor.
			 *
			 * @return The matrix stack.
			 */
			static const math::mat4f& get_stack()
			{
				return shader_3d_stack;
			}

		public:
			/** \brief Output this to a stream.
			 *
			 * @param ss Target stream.
			 * @return Stream after input.
			 */
			virtual std::ostream& put(std::ostream &ss) const;

			/** \brief Output a surface into a stream.
			 *
			 * @param lhs Left-hand-side operand.
			 * @param rhs Right-hand-side operand.
			 * @return Modified stream.
			 */
			friend inline std::ostream& operator<<(std::ostream &lhs,
					const Surface &rhs)
			{
				return rhs.put(lhs);
			}

			/** \cond */
			friend void billboard_fill(uint32_t, const math::vec2f&, uint32_t, const math::vec3f&, float);
			friend void billboard_fill(uint32_t, const math::vec3f&, float);
			friend void billboard_flush(GLenum);
			friend void billboard_reset(const Texture2D&);
			friend void bind_shader_2d();
			friend void bind_shader_2d_font(float = Surface::DEFAULT_FONT_BOLDNESS, float = Surface::DEFAULT_FONT_ANTIALIAISING);
			friend void bind_shader_2d_texture();
			friend void bind_shader_3d(const Shader&);
			friend void bind_shader_3d_billboard(const Shader&);
			friend void bind_shader_3d_line(const Shader&);
			friend inline void draw_fill(unsigned, const Color&);
			friend inline void draw_fill(unsigned, int, int);
			friend inline void draw_fill(unsigned, float, float);
			friend inline void draw_fill(unsigned, int, int, float, float);
			friend inline void draw_fill(unsigned, float, float, float, float);
			friend inline void draw_fill(float, float, float, const Color&, const math::vec3f&);
			friend inline void load_light(const LightDirectional&, const math::mat4f&);
			friend inline void load_modelview(const math::mat4f&);
			friend inline void load_transform(const math::mat4f&);
			friend inline void set_transform(const math::mat4f&);
			/** \endcond */
	};

	/** \brief Fill the next element in a billboard buffer.
	 *
	 * @param tm Texture morph.
	 * @param tt Texture coords.
	 * @param col Color.
	 * @param pos Position.
	 * @param ts Texture size.
	 */
	extern void billboard_fill(uint32_t tmx, const math::vec2f &tt,
			uint32_t col, const math::vec3f &pos, float ts);

	/** \brief Fill the next element in a billboard buffer.
	 * 
	 * @param col Color.
	 * @param pos Position.
	 * @param ts Texture size.
	 */
	extern void billboard_fill(uint32_t col, const math::vec3f &pos,
			float ts);

	/** \brief Flush all drawn billboards to the graphics card.
	 *
	 * Note that this will not reset the fill counters, just draw everything.
	 *
	 * @param type Type of elements to draw, usually GL_POINTS.
	 */
	extern void billboard_flush(GLenum type = GL_POINTS);

	/** \brief Reset billboard drawing.
	 *
	 * Will set the billboard fill counters to the beginning and define a
	 * texture to use for the next billboard operations.
	 *
	 * @param tex Texture to use.
	 */
	extern void billboard_reset(const Texture2D &tex);

	/** \brief Bind the default 2D shader.
	 *
	 * Shader bindings must be done after selecting a surface for drawing.
	 */
	extern void bind_shader_2d();

	/** \brief Bind the default 2D font shader.
	 *
	 * Shader bindings must be done after selecting a surface for drawing.
	 *
	 * Also set non-default text parameters.
	 * 
	 * @param fnt_bd Font boldness.
	 * @param fnt_aa Font antialiaising factor.
	 */
	extern void bind_shader_2d_font(float fnt_bd, float fnt_aa);

	/** \brief Bind the default 2D texture shader.
	 *
	 * Shader bindings must be done after selecting a surface for drawing.
	 */
	extern void bind_shader_2d_texture();

	/** \brief Bind the given 3D shader.
	 *
	 * Shader bindings must be done after selecting a surface for drawing.
	 *
	 * The default 3D shader uniforms will be assigned.
	 *
	 * @param sh Shader to bind.
	 */
	extern void bind_shader_3d(const Shader &sh);

	/** \brief Bind the given 3D billboard shader.
	 *
	 * Shader bindings must be done after selecting a surface for drawing.
	 *
	 * This function will assign all data required for billboard draw
	 * operations.
	 *
	 * @param sh Shader to bind.
	 */
	extern void bind_shader_3d_billboard(const Shader &sh);

	/** \brief Bind the given 3D line-drawing shader.
	 *
	 * Line-drawing shader works a lot like the billboard shader, but is used
	 * to drawing lines in 3D as opposed to textures.
	 *
	 * @param sh Shader to bind.
	 */
	extern void bind_shader_3d_line(const Shader &sh);

	/** \brief Draw one pixel.
	 *
	 * @param px X coordinate.
	 * @param py Y coordinate.
	 * @param pc Color.
	 */
	extern void draw_pixel(int px, int py, const Color &pc);

	/** \brief Draw one pixel.
	 *
	 * @param px X coordinate.
	 * @param py Y coordinate.
	 * @param pc Color.
	 */
	extern void draw_pixel(float px, float py, const Color &pc);

	/** \brief Draw one pixel.
	 *
	 * Uses the colors specified previously.
	 *
	 * @param px X coordinate.
	 * @param py Y coordinate.
	 */
	extern void draw_pixel(int x, int y);

	/** \brief Draw one pixel.
	 *
	 * Uses the colors specified previously.
	 *
	 * @param px X coordinate.
	 * @param py Y coordinate.
	 */
	extern void draw_pixel(float x, float y);

	/** \brief Draw a line.
	 *
	 * @param x1 First X coordinate.
	 * @param y1 First Y coordinate.
	 * @param x2 Second X coordinate.
	 * @param y2 Second Y coordinate.
	 * @param pc1 First coordinate color.
	 * @param pc2 Second coordinate color.
	 */
	extern void draw_line(int x1, int y1, int x2, int y2, const Color &pc1,
			const Color &pc2);

	/** \brief Draw a line.
	 *
	 * @param x1 First X coordinate.
	 * @param y1 First Y coordinate.
	 * @param x2 Second X coordinate.
	 * @param y2 Second Y coordinate.
	 * @param pc1 First coordinate color.
	 * @param pc2 Second coordinate color.
	 */
	extern void draw_line(float x1, float y1, float x2, float y2, const Color &pc1,
			const Color &pc2);

	/** \brief Draw a line.
	 *
	 * @param x1 First X coordinate.
	 * @param y1 First Y coordinate.
	 * @param x2 Second X coordinate.
	 * @param y2 Second Y coordinate.
	 * @param pc Color to draw the line with.
	 */
	extern void draw_line(int x1, int y1, int x2, int y2, const Color &pc);

	/** \brief Draw a line.
	 *
	 * @param x1 First X coordinate.
	 * @param y1 First Y coordinate.
	 * @param x2 Second X coordinate.
	 * @param y2 Second Y coordinate.
	 * @param pc Color to draw the line with.
	 */
	extern void draw_line(float x1, float y1, float x2, float y2, const Color &pc);

	/** \brief Draw a line.
	 *
	 * Uses the colors specified previously.
	 *
	 * @param x1 First X coordinate.
	 * @param y1 First Y coordinate.
	 * @param x2 Second X coordinate.
	 * @param y2 Second Y coordinate.
	 */
	extern void draw_line(int x1, int y1, int x2, int y2);

	/** \brief Draw a line.
	 *
	 * Uses the colors specified previously.
	 *
	 * @param x1 First X coordinate.
	 * @param y1 First Y coordinate.
	 * @param x2 Second X coordinate.
	 * @param y2 Second Y coordinate.
	 */
	extern void draw_line(float x1, float y1, float x2, float y2);

	/** \brief Draw a line loop.
	 *
	 * @param x1 First X coordinate.
	 * @param y1 First Y coordinate.
	 * @param x2 Second X coordinate.
	 * @param y2 Second Y coordinate.
	 * @param x3 Third X coordinate.
	 * @param y3 Third Y coordinate.
	 * @param x4 Fourth X coordinate.
	 * @param y4 Fourth Y coordinate.
	 * @param pc Color to use.
	 */
	extern void draw_line_loop(int x1, int y1, int x2, int y2, int x3, int y3,
			int x4, int y4, const Color &pc);

	/** \brief Draw a line loop.
	 *
	 * @param x1 First X coordinate.
	 * @param y1 First Y coordinate.
	 * @param x2 Second X coordinate.
	 * @param y2 Second Y coordinate.
	 * @param x3 Third X coordinate.
	 * @param y3 Third Y coordinate.
	 * @param x4 Fourth X coordinate.
	 * @param y4 Fourth Y coordinate.
	 * @param pc Color to use.
	 */
	extern void draw_line_loop(float x1, float y1, float x2, float y2, float x3,
			float y3, float x4, float y4, const Color &pc);

	/** \brief Draw a line loop.
	 *
	 * Use colors specified previously.
	 *
	 * @param x1 First X coordinate.
	 * @param y1 First Y coordinate.
	 * @param x2 Second X coordinate.
	 * @param y2 Second Y coordinate.
	 * @param x3 Third X coordinate.
	 * @param y3 Third Y coordinate.
	 * @param x4 Fourth X coordinate.
	 * @param y4 Fourth Y coordinate.
	 * @param pc Color to use.
	 */
	extern void draw_line_loop(int x1, int y1, int x2, int y2, int x3, int y3,
			int x4, int y4);

	/** \brief Draw a line loop.
	 *
	 * Use colors specified previously.
	 *
	 * @param x1 First X coordinate.
	 * @param y1 First Y coordinate.
	 * @param x2 Second X coordinate.
	 * @param y2 Second Y coordinate.
	 * @param x3 Third X coordinate.
	 * @param y3 Third Y coordinate.
	 * @param x4 Fourth X coordinate.
	 * @param y4 Fourth Y coordinate.
	 * @param pc Color to use.
	 */
	extern void draw_line_loop(float x1, float y1, float x2, float y2, float x3,
			float y3, float x4, float y4);

	/** \brief Draw a rectangle.
	 *
	 * @param px Left x coordinate.
	 * @param py Bottom y coordinate.
	 * @param pw Width.
	 * @param ph Height.
	 * @param pc Color.
	 */
	extern void draw_rect(int px, int py, int pw, int ph, const Color &pc);

	/** \brief Draw a rectangle.
	 *
	 * @param px Left x coordinate.
	 * @param py Bottom y coordinate.
	 * @param pw Width.
	 * @param ph Height.
	 * @param pc Color.
	 */
	extern void draw_rect(float px, float py, float pw, float ph, const Color &pc);

	/** \brief Draw a rectangle.
	 *
	 * Uses the colors specified previously.
	 *
	 * @param px Left x coordinate.
	 * @param py Bottom y coordinate.
	 * @param pw Width.
	 * @param ph Height.
	 */
	extern void draw_rect(int px, int py, int pw, int ph);

	/** \brief Draw a rectangle.
	 *
	 * Uses the colors specified previously.
	 *
	 * @param px Left x coordinate.
	 * @param py Bottom y coordinate.
	 * @param pw Width.
	 * @param ph Height.
	 */
	extern void draw_rect(float px, float py, float pw, float ph);

	/** \brief Draw a contour rectangle.
	 *
	 * @param px Left x coordinate.
	 * @param py Bottom y coordinate.
	 * @param pw Width.
	 * @param ph Height.
	 * @param pc Color.
	 */
	extern void draw_rect_contour(int px, int py, int pw, int ph, const Color &pc);

	/** \brief Draw a contour rectangle.
	 *
	 * @param px Left x coordinate.
	 * @param py Bottom y coordinate.
	 * @param pw Width.
	 * @param ph Height.
	 * @param pc Color.
	 */
	extern void draw_rect_contour(float px, float py, float pw, float ph,
			const Color &pc);

	/** \brief Draw a contour rectangle.
	 *
	 * Uses the colors specified previously.
	 *
	 * @param px Left x coordinate.
	 * @param py Bottom y coordinate.
	 * @param pw Width.
	 * @param ph Height.
	 */
	extern void draw_rect_contour(int px, int py, int pw, int ph);

	/** \brief Draw a contour rectangle.
	 *
	 * Uses the colors specified previously.
	 *
	 * @param px Left x coordinate.
	 * @param py Bottom y coordinate.
	 * @param pw Width.
	 * @param ph Height.
	 */
	extern void draw_rect_contour(float px, float py, float pw, float ph);

	/** \brief Draw a contour rectangle.
	 *
	 * This version has a width inward.
	 *
	 * @param px Left x coordinate.
	 * @param py Bottom y coordinate.
	 * @param pw Width.
	 * @param ph Height.
	 * @param pb Border width.
	 * @param pc Color.
	 */
	extern void draw_rect_contour(int px, int py, int pw, int ph, int pb,
			const Color &pc);

	/** \brief Draw a contour rectangle.
	 *
	 * This version has a width inward.
	 *
	 * Uses the colors specified previously.
	 *
	 * @param px Left x coordinate.
	 * @param py Bottom y coordinate.
	 * @param pw Width.
	 * @param ph Height.
	 * @param pb Border width.
	 */
	extern void draw_rect_contour(int px, int py, int pw, int ph, int pb);

	/** \brief Draw a contour rectangle.
	 *
	 * This version has a width inward.
	 *
	 * @param px Left x coordinate.
	 * @param py Bottom y coordinate.
	 * @param pw Width.
	 * @param ph Height.
	 * @param pb Border width.
	 * @param pc Color.
	 */
	extern void draw_rect_contour(float px, float py, float pw, float ph, float pb,
			const Color &pc);

	/** \brief Draw a contour rectangle.
	 *
	 * This version has a width inward.
	 *
	 * Uses the colors specified previously.
	 *
	 * @param px Left x coordinate.
	 * @param py Bottom y coordinate.
	 * @param pw Width.
	 * @param ph Height.
	 * @param pb Border width.
	 */
	extern void draw_rect_contour(float px, float py, float pw, float ph, float pb);

	/** \brief Draw a textured rectangle.
	 * 
	 * @param px Left x coordinate.
	 * @param py Bottom y coordinate.
	 * @param pw Width.
	 * @param ph Height.
	 * @param pc Color.
	 * @param pt Texture.
	 */
	extern void draw_rect_textured(int px, int py, int pw, int ph, const Color &pc,
			const Texture2D &t);

	/** \brief Draw a textured rectangle.
	 * 
	 * @param px Left x coordinate.
	 * @param py Bottom y coordinate.
	 * @param pw Width.
	 * @param ph Height.
	 * @param pc Color.
	 * @param pt Texture.
	 */
	extern void draw_rect_textured(float px, float py, float pw, float ph,
			const Color &pc, const Texture2D &t);

	/** \brief Draw a textured rectangle.
	 * 
	 * @param px Left x coordinate.
	 * @param py Bottom y coordinate.
	 * @param pw Width.
	 * @param ph Height.
	 * @param pc Color.
	 * @param pt Texture.
	 * @param srepeat S repeat.
	 * @param trepeat T repeat.
	 */
	extern void draw_rect_textured(float px, float py, float pw, float ph,
			const Color &pc, const Texture2D &pt, float srepeat, float trepeat);

	/** \brief Draw a textured rectangle.
	 * 
	 * Uses the colors specified previously.
	 *
	 * @param px Left x coordinate.
	 * @param py Bottom y coordinate.
	 * @param pw Width.
	 * @param ph Height.
	 * @param pt Texture.
	 */
	extern void draw_rect_textured(int px, int py, int pw, int ph,
			const Texture2D &t);

	/** \brief Draw a textured rectangle.
	 * 
	 * Uses the colors specified previously.
	 *
	 * @param px Left x coordinate.
	 * @param py Bottom y coordinate.
	 * @param pw Width.
	 * @param ph Height.
	 * @param pt Texture.
	 */
	extern void draw_rect_textured(float px, float py, float pw, float ph,
			const Texture2D &t);

	/** \brief Fill an area but keep texture aspect ratio.
	 *
	 * Clips texture as needed.
	 *
	 * @param area Area to fill.
	 * @param col Color to use.
	 * @param tex Texture to use.
	 */
	extern void draw_rect_textured_fill(const math::rect2f &area,
			const Color &col, const Texture2D &tex);

	/** \brief Fill a texture into an area, keeping the aspect ratio.
	 *
	 * @param area Area to fill.
	 * @param col Color to use.
	 * @param tex Texture to use.
	 */
	extern void draw_rect_textured_fit(const math::rect2f &area,
			const Color &col, const Texture2D &tex);

	/** \brief Draw an FBO onto the screen one-to-one.
	 *
	 * @param pscreen Screen to draw into.
	 * @param fbo Framebuffer object to draw.
	 */
	extern void draw_fbo(const Surface &pscreen, const Fbo &fbo);

	/** \brief Draw a glyph.
	 *
	 * Color must have been already specified.
	 *
	 * @param px Left x coordinate.
	 * @param py Bottom y coordinate.
	 * @param fs Font size.
	 * @param gly Glyph to draw.
	 * @return X distance to advance.
	 */
	extern float draw_glyph(float px, float py, float fs, const Glyph &gly);

	/** \brief Draw one line of text.
	 * 
	 * The text is drawn from the lower left coordinate of the first glyph.
	 * 
	 * Color must have been already specified.
	 * 
	 * @param px Left x coordinate of the first character in the first row.
	 * @param py Bottom y coordinate of the first character of the first row.
	 * @param fs Font size.
	 * @param text Text to render.
	 * @param fnt Font to use.
	 * @param idx Starting index (default: zero).
	 * @return The index of the start of the next row or zero if at end.
	 */
	extern unsigned draw_text_line(float px, float py, float fs,
			const std::wstring &text, const Font &fnt, unsigned idx = 0);

	/** \brief Draw some text.
	 *
	 * Color must have been already specified.
	 *
	 * @param px X coordinate of text center.
	 * @param py Y coordinate of text center.
	 * @param fs Font size.
	 * @param text Text to render.
	 * @param fnt Font to use.
	 * @param pc Color.
	 * @param justify Justification mode.
	 */
	extern void draw_text(float x, float y, float fs, const std::wstring &text,
			const Font &fnt, TextJustify justify = LEFT);

	/** \brief Draw some text.
	 * 
	 * @param px X coordinate of text center.
	 * @param py Y coordinate of text center.
	 * @param fs Font size.
	 * @param text Text to render.
	 * @param fnt Font to use.
	 * @param pc Color.
	 * @param justify Justification mode.
	 */
	extern void draw_text(float x, float y, float fs, const std::wstring &text,
			const Font &fnt, const Color &pc, TextJustify justify = LEFT);

	/** \brief Wrapper for draw_text.
	 *
	 * @param px X coordinate of text center.
	 * @param py Y coordinate of text center.
	 * @param fs Font size.
	 * @param text Text to render.
	 * @param fnt Font to use.
	 * @param justify Text justify.
	 */
	inline void draw_text(int px, int py, int fs, 	const std::wstring &text,
			const Font &fnt, TextJustify justify = LEFT)
	{
		float mul2d = Surface::get_mul_2d();
		draw_text(static_cast<float>(px) * mul2d, static_cast<float>(py) * mul2d,
				static_cast<float>(fs) * mul2d, text, fnt, justify);
	}

	/** \brief Wrapper for draw_text.
	 *
	 * @param px X coordinate of text center.
	 * @param py Y coordinate of text center.
	 * @param fs Font size.
	 * @param text Text to render.
	 * @param pc Color.
	 * @param justify Text justify.
	 */
	inline void draw_text(int px, int py, int fs,
			const std::wstring &text, const Font &fnt, const Color &pc,
			TextJustify justify = LEFT)
	{
		float mul2d = Surface::get_mul_2d();
		draw_text(static_cast<float>(px) * mul2d, static_cast<float>(py) * mul2d,
				static_cast<float>(fs) * mul2d, text, fnt, pc, justify);
	}

	/** \brief Draw elements quickly.
	 *
	 * @param type What kind of elements are we drawing?
	 * @param cnt How many elements to feed.
	 */
	inline void draw_arrays(unsigned type, unsigned cnt)
	{
		glDrawArrays(type, 0, cnt);
	}

	/** \brief Fill nth element of the color buffer.
	 *
	 * @param idx Color index.
	 * @param pc Color to fill with.
	 */
	inline void draw_fill(unsigned idx, const Color &pc)
	{
		idx *= 4;
		Surface::array_color[idx + 0] = pc.r();
		Surface::array_color[idx + 1] = pc.g();
		Surface::array_color[idx + 2] = pc.b();
		Surface::array_color[idx + 3] = pc.a();
	}

	/** \brief Fill nth element of the vertex buffer.
	 *
	 * @param idx Vertex index.
	 * @param px X coordinate.
	 * @param py Y coordinate.
	 */
	inline void draw_fill(unsigned idx, int px, int py)
	{
		idx *= 4;
		Surface::array_vertex[idx + 0] = static_cast<float>(px) * Surface::shader_2d_mul;
		Surface::array_vertex[idx + 1] = static_cast<float>(py) * Surface::shader_2d_mul;
	}

	/** \brief Fill nth element of the vertex buffer.
	 *
	 * @param idx Vertex index.
	 * @param px X coordinate.
	 * @param py Y coordinate.
	 */
	inline void draw_fill(unsigned idx, float px, float py)
	{
		idx *= 4;
		Surface::array_vertex[idx + 0] = px;
		Surface::array_vertex[idx + 1] = py;
	}

	/** \brief Fill nth element of the vertex buffer.
	 *
	 * @param idx Vertex index.
	 * @param pp Coordinates.
	 */
	inline void draw_fill(unsigned idx, const math::vec2f &pp)
	{
		draw_fill(idx, pp.x(), pp.y());
	}

	/** \brief Fill vertex buffer and the texture coordinates.
	 *
	 * @param idx Vertex index.
	 * @param px X coordinate.
	 * @param py Y coordinate.
	 * @param ps S coordinate.
	 * @param pt T coordinate.
	 */
	inline void draw_fill(unsigned idx, int px, int py, float ps, float pt)
	{
		idx *= 4;
		Surface::array_vertex[idx + 0] = static_cast<float>(px) * Surface::shader_2d_mul;
		Surface::array_vertex[idx + 1] = static_cast<float>(py) * Surface::shader_2d_mul;
		Surface::array_vertex[idx + 2] = ps;
		Surface::array_vertex[idx + 3] = pt;
	}

	/** \brief Fill vertex buffer and the texture coordinates.
	 *
	 * @param idx Vertex index.
	 * @param px X coordinate.
	 * @param py Y coordinate.
	 * @param ps S coordinate.
	 * @param pt T coordinate.
	 */
	inline void draw_fill(unsigned idx, float px, float py, float ps, float pt)
	{
		idx *= 4;
		Surface::array_vertex[idx + 0] = px;
		Surface::array_vertex[idx + 1] = py;
		Surface::array_vertex[idx + 2] = ps;
		Surface::array_vertex[idx + 3] = pt;
	}

	/** \brief Load a new directional light.
	 *
	 * @param op New directional light.
	 * @param wm World matrix (camera orientation).
	 */
	inline void load_light(const LightDirectional &op, const math::mat4f &wm)
	{
		Surface::shader_3d_light_ambient->update(op.getAmbient());
		Surface::shader_3d_light_diffuse->update(op.getDiffuse());
		Surface::shader_3d_light_dir->update(math::normalize(-(wm * op.getDir())));
	}

	/** \brief Set the matrix stack.
	 *
	 * Sets the stack matrix but does not load it into the graphics card.
	 *
	 * @param op New modelview matrix.
	 */
	inline void set_transform(const math::mat4f &op)
	{
		Surface::shader_3d_stack = Surface::shader_3d_projection * op;
	}

	/** \brief Load a new transformation matrix.
	 *
	 * @param op New modelview matrix.
	 */
	inline void load_transform(const math::mat4f &op)
	{
		set_transform(op);
		Surface::shader_3d_transform->update(Surface::shader_3d_stack);
	}

	/** \brief Load a new modelview matrix.
	 *
	 * @param op New modelview matrix.
	 */
	inline void load_modelview(const math::mat4f &op)
	{
		Surface::shader_3d_modelview->update(op);
		load_transform(op);
	}
}

#endif
