#include "gfx/surface.hpp"

#include "gfx/fbo.hpp"
#include "gfx/font.hpp"

#include <sstream>

using namespace gfx;

const float Surface::DEFAULT_FONT_BOLDNESS = 0.40f;
const float Surface::DEFAULT_FONT_ANTIALIAISING = 10.0f;
ArrayA4f Surface::array_color(PRIMITIVE_ARRAY_SIZE);
ArrayA4f Surface::array_vertex(PRIMITIVE_ARRAY_SIZE);
ArrayBillboard Surface::array_billboard(BILLBOARD_ARRAY_SIZE);
Shader *Surface::shader_2d = NULL;
Shader *Surface::shader_2d_font = NULL;
Shader *Surface::shader_2d_texture = NULL;
const Uniform *Surface::shader_3d_light_ambient = NULL;
const Uniform *Surface::shader_3d_light_diffuse = NULL;
const Uniform *Surface::shader_3d_light_dir = NULL;
const Uniform *Surface::shader_3d_modelview = NULL;
const Uniform *Surface::shader_3d_transform = NULL;
math::vec2f Surface::shader_2d_offset;
math::vec2f Surface::shader_2d_scale;
math::mat4f Surface::shader_3d_projection;
math::mat4f Surface::shader_3d_stack;
float Surface::shader_2d_mul;

/** \brief 2d shader disables.
 *
 * Disable 2D shader elements that need to be taken care of once per shader.
 */
static void bind_shader_2d_cleanup()
{
	glBindBuffer(GL_ARRAY_BUFFER, 0);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);

	// Maximum of 6 vertex attribs in use. All vertex attribs per shader start
	// from 0 and move upward, cleaning this way is safe.
	glDisableVertexAttribArray(2);
	glDisableVertexAttribArray(3);
	glDisableVertexAttribArray(4);
	glDisableVertexAttribArray(5);

	glDisable(GL_POINT_SPRITE);
}

Surface::Surface(unsigned pw, unsigned ph, unsigned pb) :
	SurfaceBase(pw, ph, pb) { }

math::rect2f Surface::getArea() const
{
	float ww = static_cast<float>(m_w),
				hh = static_cast<float>(m_h);

	if(m_w > m_h)
	{
		return math::rect2f(0.0f, 0.0f, ww / hh, 1.0f);
	}
	return math::rect2f(0.0f, 0.0f, 1.0f, hh / ww);
}

void Surface::select2D(unsigned px, unsigned py, unsigned pw, unsigned ph)
{
	float h_1 = static_cast<float>(m_h - 1),
				w_1 = static_cast<float>(m_w - 1),
				divw = 1.0f / static_cast<float>(m_w),
				divh = 1.0f / static_cast<float>(m_h),
				divwh = w_1 / h_1,
				divhw = h_1 / w_1;

	if(m_w > m_h)
	{
		shader_2d_mul = 1.0f / h_1;
		shader_2d_offset = math::vec2f(-1.0f + divw,
				-1.0f + divh);
		shader_2d_scale = math::vec2f((2.0f - 2.0f * divw) * divhw,
				2.0f - 2.0f * divh);
	}
	else
	{
		shader_2d_mul = 1.0f / w_1;
		shader_2d_offset = math::vec2f(-1.0f + divw,
				-1.0f + divh);
		shader_2d_scale = math::vec2f(2.0f - 2.0f * divw,
				(2.0f - 2.0f * divh) * divwh);
	}

	glDepthMask(GL_FALSE);
	glDisable(GL_CULL_FACE);
	glDisable(GL_DEPTH_TEST);

	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	this->setBoundary(px, py, pw, ph);
}

void Surface::select3D(unsigned px, unsigned py, unsigned pw, unsigned ph,
		float pview, float paspect, float pnear, float pfar)
{
	float ratiox = static_cast<float>(pw) * paspect,
				ratioy = static_cast<float>(ph);

	if(ratiox > ratioy)
	{
		ratiox = static_cast<float>(ratiox) / static_cast<float>(ratioy);
		ratioy = 1.0f;
	}
	else
	{
		ratioy = static_cast<float>(ratioy) / static_cast<float>(ratiox);
		ratiox = 1.0f;
	}

	pview = atanf((ratioy / ratiox) * tanf(pview * 0.5f)) * 2.0f;
	//std::cout << "ratios: " << ratiox << " / " << ratioy << " -> " << pview << std::endl;
	shader_3d_projection.loadPerspective(pview, ratiox / ratioy, pnear, pfar);

	glDepthMask(GL_TRUE);
	glEnable(GL_CULL_FACE);
	glEnable(GL_DEPTH_TEST);

	this->setBoundary(px, py, pw, ph);
}

math::rect2i Surface::toPixelArea(const math::rect2f &parea) const
{
	float ww = static_cast<float>(m_w),
				hh = static_cast<float>(m_h);

	if(ww > hh)
	{
		return math::rect2i(math::lround(parea.x1() * ww / (ww / hh)),
				math::lround(parea.y1() * hh),
				math::lround(parea.w() * ww / (ww / hh)),
				math::lround(parea.h() * hh));
	}
	return math::rect2i(math::lround(parea.x1() * ww),
			math::lround(parea.y1() * hh / (hh / ww)),
			math::lround(parea.w() * ww),
			math::lround(parea.h() * hh / (hh / ww)));
}

void Surface::updateBillboardParams(const Shader &sh, float divisor_min,
		float divisor_max)
{
	float smaller_axis = math::min(static_cast<float>(m_w),
			static_cast<float>(m_h));
	sh.getUniform("billboard_params")->update(
			smaller_axis / divisor_min, smaller_axis / divisor_max, smaller_axis);
}

void gfx::billboard_fill(uint32_t tm, const math::vec2f &tt, uint32_t col,
		const math::vec3f &pos, float ts)
{
	Surface::array_billboard.add(tm, tt, col, pos, ts);
}

void gfx::billboard_fill(uint32_t col, const math::vec3f &pos, float ts)
{
	Surface::array_billboard.add(col, pos, ts);
}

void gfx::billboard_flush(GLenum type)
{
	draw_arrays(type, Surface::array_billboard.getCount());
}

void gfx::billboard_reset(const Texture2D &tex)
{
	tex.bind();
	Surface::array_billboard.reset();
}

void gfx::bind_shader_2d()
{
	Shader *sh = Surface::shader_2d;
	sh->bind();
	bind_shader_2d_cleanup();

	Surface::array_color.feed(sh->getAttrColor());
	Surface::array_vertex.feed(sh->getAttrVertex());

	sh->getUniformTransform().update(Surface::shader_2d_offset,
			Surface::shader_2d_scale);
}

void gfx::bind_shader_2d_font(float fnt_bd, float fnt_aa)
{
	Shader *sh = Surface::shader_2d_font;
	sh->bind();
	bind_shader_2d_cleanup();

	Surface::array_color.feed(Surface::shader_2d_font->getAttrColor());
	Surface::array_vertex.feed(Surface::shader_2d_font->getAttrVertex());

	sh->getUniformTransform().update(Surface::shader_2d_offset,
			Surface::shader_2d_scale);
	sh->getUniformTparams().update(fnt_bd, fnt_aa);
	sh->getUniformTex().update(0);

	glActiveTexture(GL_TEXTURE0);
}

void gfx::bind_shader_2d_texture()
{
	Shader *sh = Surface::shader_2d_texture;
	sh->bind();
	bind_shader_2d_cleanup();

	Surface::array_color.feed(sh->getAttrColor());
	Surface::array_vertex.feed(sh->getAttrVertex());

	sh->getUniformTransform().update(Surface::shader_2d_offset,
			Surface::shader_2d_scale);
	sh->getUniformTex().update(0);

	glActiveTexture(GL_TEXTURE0);
}

void gfx::bind_shader_3d(const Shader &sh)
{
	sh.bind();

	Surface::shader_3d_light_ambient = &(sh.getUniformLightAmbient());
	Surface::shader_3d_light_diffuse = &(sh.getUniformLightDiffuse());
	Surface::shader_3d_light_dir = &(sh.getUniformLightDir());
	Surface::shader_3d_modelview = &(sh.getUniformModelview());
	Surface::shader_3d_transform = &(sh.getUniformTransform());

	glDisable(GL_POINT_SPRITE);
}

void gfx::bind_shader_3d_billboard(const Shader &sh)
{
	sh.bind();

	glBindBuffer(GL_ARRAY_BUFFER, 0);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
	glDisableVertexAttribArray(4);
	glDisableVertexAttribArray(5);

	Surface::array_billboard.feed(*(sh.getAttribute("morph")),
			sh.getAttrTexcoord(), sh.getAttrColor(), sh.getAttrVertex());
	Surface::shader_3d_modelview = &(sh.getUniformModelview());
	Surface::shader_3d_transform = &(sh.getUniformTransform());
	
	sh.getUniformProjection().update(Surface::shader_3d_projection);
	sh.getUniformTex().update(0);
	glActiveTexture(GL_TEXTURE0);
	
	glEnable(GL_POINT_SPRITE);
	glEnable(GL_VERTEX_PROGRAM_POINT_SIZE);
	glPointParameterf(GL_POINT_SPRITE_COORD_ORIGIN, GL_UPPER_LEFT);
	glTexEnvi(GL_POINT_SPRITE, GL_COORD_REPLACE, GL_TRUE);
}

void gfx::bind_shader_3d_line(const Shader &sh)
{
	sh.bind();

	glBindBuffer(GL_ARRAY_BUFFER, 0);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
	glDisableVertexAttribArray(4);
	glDisableVertexAttribArray(5);

	Surface::array_billboard.feed(*(sh.getAttribute("morph")),
			sh.getAttrTexcoord(), sh.getAttrColor(), sh.getAttrVertex());
	Surface::shader_3d_transform = &(sh.getUniformTransform());
	
	sh.getUniformTex().update(0);
	glActiveTexture(GL_TEXTURE0);
	
	glDisable(GL_POINT_SPRITE);
}

void gfx::draw_pixel(int px, int py, const Color &pc)
{
	draw_fill(0, px, py);
	draw_fill(0, pc);
	draw_arrays(GL_POINTS, 1);
}

void gfx::draw_pixel(float px, float py, const Color &pc)
{
	draw_fill(0, px, py);
	draw_fill(0, pc);
	draw_arrays(GL_POINTS, 1);
}

void gfx::draw_pixel(int px, int py)
{
	draw_fill(0, px, py);
	draw_arrays(GL_POINTS, 1);
}

void gfx::draw_pixel(float px, float py)
{
	draw_fill(0, px, py);
	draw_arrays(GL_POINTS, 1);
}

void gfx::draw_line(int x1, int y1, int x2, int y2, const Color &pc1,
		const Color &pc2)
{
	draw_fill(0, x1, y1);
	draw_fill(1, x2, y2);
	draw_fill(0, pc1);
	draw_fill(1, pc2);
	draw_arrays(GL_LINES, 2);
}

void gfx::draw_line(float x1, float y1, float x2, float y2, const Color &pc1,
		const Color &pc2)
{
	draw_fill(0, x1, y1);
	draw_fill(1, x2, y2);
	draw_fill(0, pc1);
	draw_fill(1, pc2);
	draw_arrays(GL_LINES, 2);
}

void gfx::draw_line(int x1, int y1, int x2, int y2, const Color &pc)
{
	draw_fill(0, x1, y1);
	draw_fill(1, x2, y2);
	draw_fill(0, pc);
	draw_fill(1, pc);
	draw_arrays(GL_LINES, 2);
}

void gfx::draw_line(float x1, float y1, float x2, float y2, const Color &pc)
{
	draw_fill(0, x1, y1);
	draw_fill(1, x2, y2);
	draw_fill(0, pc);
	draw_fill(1, pc);
	draw_arrays(GL_LINES, 2);
}

void gfx::draw_line(int x1, int y1, int x2, int y2)
{
	draw_fill(0, x1, y1);
	draw_fill(1, x2, y2);
	draw_arrays(GL_LINES, 2);
}

void gfx::draw_line(float x1, float y1, float x2, float y2)
{
	draw_fill(0, x1, y1);
	draw_fill(1, x2, y2);
	draw_arrays(GL_LINES, 2);
}

void gfx::draw_line_loop(int x1, int y1, int x2, int y2, int x3, int y3,
		int x4, int y4, const Color &pc)
{
	draw_fill(0, x1, y1);
	draw_fill(1, x2, y2);
	draw_fill(2, x3, y3);
	draw_fill(3, x4, y4);
	draw_fill(0, pc);
	draw_fill(1, pc);
	draw_fill(2, pc);
	draw_fill(3, pc);
	draw_arrays(GL_LINE_LOOP, 4);
}

void gfx::draw_line_loop(float x1, float y1, float x2, float y2, float x3,
		float y3, float x4, float y4, const Color &pc)
{
	draw_fill(0, x1, y1);
	draw_fill(1, x2, y2);
	draw_fill(2, x3, y3);
	draw_fill(3, x4, y4);
	draw_fill(0, pc);
	draw_fill(1, pc);
	draw_fill(2, pc);
	draw_fill(3, pc);
	draw_arrays(GL_LINE_LOOP, 4);
}

void gfx::draw_line_loop(int x1, int y1, int x2, int y2, int x3, int y3,
		int x4, int y4)
{
	draw_fill(0, x1, y1);
	draw_fill(1, x2, y2);
	draw_fill(2, x3, y3);
	draw_fill(3, x4, y4);
	draw_arrays(GL_LINE_LOOP, 4);
}

void gfx::draw_line_loop(float x1, float y1, float x2, float y2, float x3,
		float y3, float x4, float y4)
{
	draw_fill(0, x1, y1);
	draw_fill(1, x2, y2);
	draw_fill(2, x3, y3);
	draw_fill(3, x4, y4);
	draw_arrays(GL_LINE_LOOP, 4);
}

void gfx::draw_rect(int px, int py, int pw, int ph, const Color &pc)
{
	int x2 = px + pw - 1,
			y2 = py + ph - 1;
	draw_fill(0, px, py);
	draw_fill(1, x2, py);
	draw_fill(2, x2, y2);
	draw_fill(3, px, y2);
	draw_fill(0, pc);
	draw_fill(1, pc);
	draw_fill(2, pc);
	draw_fill(3, pc);
	draw_arrays(GL_QUADS, 4);
}

void gfx::draw_rect(float px, float py, float pw, float ph, const Color &pc)
{
	float x2 = px + pw,
				y2 = py + ph;
	draw_fill(0, px, py);
	draw_fill(1, x2, py);
	draw_fill(2, x2, y2);
	draw_fill(3, px, y2);
	draw_fill(0, pc);
	draw_fill(1, pc);
	draw_fill(2, pc);
	draw_fill(3, pc);
	draw_arrays(GL_QUADS, 4);
}

void gfx::draw_rect(int px, int py, int pw, int ph)
{
	int x2 = px + pw - 1,
			y2 = py + ph - 1;
	draw_fill(0, px, py);
	draw_fill(1, x2, py);
	draw_fill(2, x2, y2);
	draw_fill(3, px, y2);
	draw_arrays(GL_QUADS, 4);
}

void gfx::draw_rect(float px, float py, float pw, float ph)
{
	float x2 = px + pw,
				y2 = py + ph;
	draw_fill(0, px, py);
	draw_fill(1, x2, py);
	draw_fill(2, x2, y2);
	draw_fill(3, px, y2);
	draw_arrays(GL_QUADS, 4);
}

void gfx::draw_rect_contour(int px, int py, int pw, int ph,
		const Color &pc)
{
	int x2 = px + pw - 1,
			y2 = py + ph - 1;
	draw_line_loop(px, py, x2, py, x2, y2, px, y2, pc);
}

void gfx::draw_rect_contour(float px, float py, float pw, float ph,
		const Color &pc)
{
	float x2 = px + pw,
				y2 = py + ph;
	draw_line_loop(px, py, x2, py, x2, y2, px, y2, pc);
}

void gfx::draw_rect_contour(int px, int py, int pw, int ph)
{
	int x2 = px + pw - 1,
			y2 = py + ph - 1;
	draw_line_loop(px, py, x2, py, x2, y2, px, y2);
}

void gfx::draw_rect_contour(float px, float py, float pw, float ph)
{
	float x2 = px + pw,
				y2 = py + ph;
	draw_line_loop(px, py, x2, py, x2, y2, px, y2);
}

void gfx::draw_rect_contour(int px, int py, int pw, int ph, int pb,
		const Color &pc)
{
	draw_fill(0, pc);
	draw_fill(1, pc);
	draw_fill(2, pc);
	draw_fill(3, pc);
	draw_fill(4, pc);
	draw_fill(5, pc);
	draw_fill(6, pc);
	draw_fill(7, pc);
	draw_fill(8, pc);
	draw_fill(9, pc);
	draw_rect_contour(px, py, pw, ph, pb);
}

void gfx::draw_rect_contour(int px, int py, int pw, int ph, int pb)
{
	int x2 = px + pw - 1,
			y2 = py + ph - 1;

	draw_fill(0, px, py);
	draw_fill(1, px + pb, py + pb);
	draw_fill(2, x2, py);
	draw_fill(3, x2 - pb, py + pb);
	draw_fill(4, x2, y2);
	draw_fill(5, x2 - pb, y2 - pb);
	draw_fill(6, px, y2);
	draw_fill(7, px + pb, y2 -pb);
	draw_fill(8, px, py);
	draw_fill(9, px + pb, py + pb);
	draw_arrays(GL_TRIANGLE_STRIP, 10);
}

void gfx::draw_rect_contour(float px, float py, float pw, float ph, float pb,
		const Color &pc)
{
	draw_fill(0, pc);
	draw_fill(1, pc);
	draw_fill(2, pc);
	draw_fill(3, pc);
	draw_fill(4, pc);
	draw_fill(5, pc);
	draw_fill(6, pc);
	draw_fill(7, pc);
	draw_fill(8, pc);
	draw_fill(9, pc);
	draw_rect_contour(px, py, pw, ph, pb);
}

void gfx::draw_rect_contour(float px, float py, float pw, float ph, float pb)
{
	float x2 = px + pw,
				y2 = py + ph;

	draw_fill(0, px, py);
	draw_fill(1, px + pb, py + pb);
	draw_fill(2, x2, py);
	draw_fill(3, x2 - pb, py + pb);
	draw_fill(4, x2, y2);
	draw_fill(5, x2 - pb, y2 - pb);
	draw_fill(6, px, y2);
	draw_fill(7, px + pb, y2 -pb);
	draw_fill(8, px, py);
	draw_fill(9, px + pb, py + pb);
	draw_arrays(GL_TRIANGLE_STRIP, 10);
}

void gfx::draw_rect_textured(int px, int py, int pw, int ph,
		const Color &pc, const Texture2D &pt)
{
	int x2 = px + pw - 1,
			y2 = py + ph - 1;
	pt.bind();
	draw_fill(0, px, py, 0.0f, 1.0f);
	draw_fill(1, x2, py, 1.0f, 1.0f);
	draw_fill(2, x2, y2, 1.0f, 0.0f);
	draw_fill(3, px, y2, 0.0f, 0.0f);
	draw_fill(0, pc);
	draw_fill(1, pc);
	draw_fill(2, pc);
	draw_fill(3, pc);
	draw_arrays(GL_QUADS, 4);
}

void gfx::draw_rect_textured(float px, float py, float pw, float ph,
		const Color &pc, const Texture2D &pt)
{
	float x2 = px + pw,
				y2 = py + ph;
	pt.bind();
	draw_fill(0, px, py, 0.0f, 1.0f);
	draw_fill(1, x2, py, 1.0f, 1.0f);
	draw_fill(2, x2, y2, 1.0f, 0.0f);
	draw_fill(3, px, y2, 0.0f, 0.0f);
	draw_fill(0, pc);
	draw_fill(1, pc);
	draw_fill(2, pc);
	draw_fill(3, pc);
	draw_arrays(GL_QUADS, 4);
}

void gfx::draw_rect_textured(float px, float py, float pw, float ph,
		const Color &pc, const Texture2D &pt, float srepeat, float trepeat)
{
	float x2 = px + pw,
				y2 = py + ph;
	pt.bind();
	draw_fill(0, px, py, 0.0f, trepeat);
	draw_fill(1, x2, py, srepeat, trepeat);
	draw_fill(2, x2, y2, srepeat, 0.0f);
	draw_fill(3, px, y2, 0.0f, 0.0f);
	draw_fill(0, pc);
	draw_fill(1, pc);
	draw_fill(2, pc);
	draw_fill(3, pc);
	draw_arrays(GL_QUADS, 4);
}

void gfx::draw_rect_textured(int px, int py, int pw, int ph,
		const Texture2D &pt)
{
	int x2 = px + pw - 1,
			y2 = py + ph - 1;
	pt.bind();
	draw_fill(0, px, py, 0.0f, 1.0f);
	draw_fill(1, x2, py, 1.0f, 1.0f);
	draw_fill(2, x2, y2, 1.0f, 0.0f);
	draw_fill(3, px, y2, 0.0f, 0.0f);
	draw_arrays(GL_QUADS, 4);
}

void gfx::draw_rect_textured(float px, float py, float pw, float ph,
		const Texture2D &pt)
{
	float x2 = px + pw,
				y2 = py + ph;
	pt.bind();
	draw_fill(0, px, py, 0.0f, 1.0f);
	draw_fill(1, x2, py, 1.0f, 1.0f);
	draw_fill(2, x2, y2, 1.0f, 0.0f);
	draw_fill(3, px, y2, 0.0f, 0.0f);
	draw_arrays(GL_QUADS, 4);
}

void gfx::draw_fbo(const Surface &pscreen, const Fbo &fbo)
{
	int x2 = pscreen.getWidth() - 1,
			y2 = pscreen.getHeight() - 1;
	// TODO: implement correctly
	glEnable(GL_TEXTURE_RECTANGLE_ARB);
	fbo.bindTexture(0);
	draw_fill(0, 0, 0, 0.0f, 512.0f);
	draw_fill(1, x2, 0, 512.0f, 512.0f);
	draw_fill(2, x2, y2, 512.0f, 0.0f);
	draw_fill(3, 0, y2, 0.0f, 0.0f);
	draw_arrays(GL_QUADS, 4);
	glDisable(GL_TEXTURE_RECTANGLE_ARB);
}

void gfx::draw_rect_textured_fill(const math::rect2f &area,
		const Color &col, const Texture2D &tex)
{
	float ratiot = static_cast<float>(tex.getWidth()) / static_cast<float>(tex.getHeight());
	float ratioa = area.w() / area.h();
	if(ratiot > ratioa)
	{
		gfx::draw_rect_textured(area.x1(), area.y1(),
				ratiot / ratioa * area.w(), area.h(),
				col, tex);
	}
	else
	{
		gfx::draw_rect_textured(area.x1(), area.y1(),
				area.w(), ratioa / ratiot * area.h(),
				col, tex);
	}
}

void gfx::draw_rect_textured_fit(const math::rect2f &area,
		const Color &col, const Texture2D &tex)
{
	float ratiot = static_cast<float>(tex.getWidth()) / static_cast<float>(tex.getHeight());
	float ratioa = area.w() / area.h();
	if(ratiot > ratioa)
	{
		gfx::draw_rect_textured(area.x1(), area.y1(),
				area.w(), ratioa / ratiot * area.h(),
				col, tex);
	}
	else
	{
		gfx::draw_rect_textured(area.x1(), area.y1(),
				ratiot / ratioa * area.w(), area.h(),
				col, tex);
	}
}

float gfx::draw_glyph(float px, float py, float fs, const Glyph &gly)
{
	px += gly.getLeft() * fs;
	py -= fs * (gly.getHeight() - gly.getTop());
	float rendx1 = px - ((1.0f - gly.getWidth()) * 0.5f * fs),
				rendy1 = py - ((1.0f - gly.getHeight()) * 0.5f * fs),
				rendx2 = rendx1 + fs,
				rendy2 = rendy1 + fs;

	//Console.WriteLine("Glyph drawn: [" + gly.TexS1 + ";" + gly.TexT1 + "] -> [" + gly.TexS2 + ";" + gly.TexT2 + "]");

	const math::vec2f &st1 = gly.getST1();
	const math::vec2f &st2 = gly.getST2();

	gly.bind();
	draw_fill(0, rendx1, rendy1, st1.x(), st1.y());
	draw_fill(1, rendx2, rendy1, st2.x(), st1.y());
	draw_fill(2, rendx2, rendy2, st2.x(), st2.y());
	draw_fill(3, rendx1, rendy2, st1.x(), st2.y());
	draw_arrays(GL_QUADS, 4);

	//Console.WriteLine("Pos: " + x + ";" + y + " Advance: " + gly.AdvanceX + ";" + gly.AdvanceY + " fs: " + fs);

	return gly.getAdvance().x() * fs;
}

unsigned gfx::draw_text_line(float px, float py, float fs,
		const std::wstring &text, const Font &fnt, unsigned idx)
{
	for(;; ++idx)
	{
		if(idx >= text.length())
		{
			return 0;
		}
		wchar_t cc = text[idx];
		if(cc == '\n')
		{
			return idx + 1;
		}

		px += draw_glyph(px, py, fs, fnt.getGlyph(cc));
	}
}

void gfx::draw_text(float px, float py, float fs, const std::wstring &text,
		const Font &fnt, TextJustify justify)
{
	switch(justify)
	{
		case LEFT:
			{
				unsigned idx = 0;
				do {
					idx = draw_text_line(px, py, fs, text, fnt, idx);
					py -= fs;
				} while(idx);
			}
			return;

		case RIGHT:
			{
				unsigned idx = 0;
				do {
					unsigned tmp;
					float fwid;
					boost::tie(tmp, fwid) = fnt.calcLineWidth(fs, text, idx);
					idx = draw_text_line(px - fwid, py, fs, text, fnt, idx);
					py -= fs;
				} while(idx);
			}
			return;

		case CENTER:
		default:
			{
				std::list<float> *rlen = fnt.calcTextLengths(fs, text);
				py += static_cast<float>(static_cast<int>(rlen->size()) - 2) * 0.5f * fs;
				unsigned idx = 0;
				BOOST_FOREACH(float fwid, *rlen)
				{
					idx = draw_text_line(px - fwid * 0.5f, py, fs, text, fnt, idx);
					py -= fs;
				}
				delete rlen;
			}
			return;
	}
}

void gfx::draw_text(float px, float py, float fs, const std::wstring &text,
		const Font &fnt, const Color &pc, TextJustify justify)
{
	draw_fill(0, pc);
	draw_fill(1, pc);
	draw_fill(2, pc);
	draw_fill(3, pc);
	draw_text(px, py, fs, text, fnt, justify);
}

std::ostream& Surface::put(std::ostream &ss) const
{
	return ss << m_w << "x" << m_h << "@" << m_b << "bpp";
}

