#ifndef GFX_SHADER_HPP
#define GFX_SHADER_HPP

#include "gfx/attribute.hpp"
#include "gfx/uniform.hpp"

#include <map>
#include <string>

namespace gfx
{
	/** Convenience typeded. */
	typedef std::map<std::string, Attribute> AttributeMap;

	/** Convenience typeded. */
	typedef std::map<std::string, Uniform> UniformMap;

	/** \brief Representation of a shader.
	 *
	 * Shaders contain both a vertex program and a fragment program. If you want
	 * to reuse programs, please use a bit of copypasta.
	 *
	 * Several well-known vertex attributes are stored in the shader should they
	 * exist, the rest are retrievable via the attribute map. Same for uniforms.
	 * Requesting a well-known attribute or uniform that is not specified within
	 * the shader results in unspecified behavior.
	 */
	class Shader
	{
		private:
			/** Currently active shader. */
			static const Shader *current_shader;

		private:
			/** Vertex program. */
			std::string _vshader;

			/** Fragment program. */
			std::string _fshader;

			/** Map of attributes. */
			AttributeMap _attributes;

			/** Map of uniforms. */
			UniformMap _uniforms;

			/** Well-known attribute. */
			Attribute _bone_ref;

			/** Well-known attribute. */
			Attribute _bone_weight;

			/** Well-known attribute. */
			Attribute _color;

			/** Well-known attribute. */
			Attribute _normal;

			/** Well-known attribute. */
			Attribute _texcoord;

			/** Well-known attribute. */
			Attribute _vertex;

			/** Well-known uniform. */
			Uniform _light_ambient;

			/** Well-known uniform. */
			Uniform _light_diffuse;

			/** Well-known uniform. */
			Uniform _light_dir;

			/** Well-known uniform. */
			Uniform _modelview;

			/** Well-known uniform. */
			Uniform _normalmap;

			/** Well-known uniform. */
			Uniform _projection;

			/** Well-known uniform. */
			Uniform _tex;

			/** Well-known uniform. */
			Uniform _tparams;

			/** Well-known uniform. */
			Uniform _transform;

			/** Vertex shader id. */
			unsigned _vsid;

			/** Fragment shader id. */
			unsigned _fsid;

			/** Shader program id. */
			unsigned _id;

		public:
			/** \brief Get internal attribute.
			 *
			 * Results may be undefined if attribute missing from shader.
			 *
			 * @return Vertex attribute reference.
			 */
			const Attribute& getAttrBoneRef() const
			{
				return _bone_ref;
			}

			/** \brief Get internal attribute.
			 *
			 * Results may be undefined if attribute missing from shader.
			 *
			 * @return Vertex attribute reference.
			 */
			const Attribute& getAttrBoneWeight() const
			{
				return _bone_weight;
			}

			/** \brief Get internal attribute.
			 *
			 * Results may be undefined if attribute missing from shader.
			 *
			 * @return Vertex attribute reference.
			 */
			const Attribute& getAttrColor() const
			{
				return _color;
			}

			/** \brief Get internal attribute.
			 *
			 * Results may be undefined if attribute missing from shader.
			 *
			 * @return Vertex attribute reference.
			 */
			const Attribute& getAttrNormal() const
			{
				return _normal;
			}

			/** \brief Get internal attribute.
			 *
			 * Results may be undefined if attribute missing from shader.
			 *
			 * @return Vertex attribute reference.
			 */
			const Attribute& getAttrTexcoord() const
			{
				return _texcoord;
			}

			/** \brief Get internal attribute.
			 *
			 * Results may be undefined if attribute missing from shader.
			 *
			 * @return Vertex attribute reference.
			 */
			const Attribute& getAttrVertex() const
			{
				return _vertex;
			}

			/** \brief Get internal attribute.
			 *
			 * Results may be undefined if uniform missing from shader.
			 *
			 * @return Uniform reference.
			 */
			const Uniform& getUniformLightAmbient() const
			{
				return _light_ambient;
			}

			/** \brief Get internal attribute.
			 *
			 * Results may be undefined if uniform missing from shader.
			 *
			 * @return Uniform reference.
			 */
			const Uniform& getUniformLightDiffuse() const
			{
				return _light_diffuse;
			}

			/** \brief Get internal attribute.
			 *
			 * Results may be undefined if uniform missing from shader.
			 *
			 * @return Uniform reference.
			 */
			const Uniform& getUniformLightDir() const
			{
				return _light_dir;
			}

			/** \brief Get internal attribute.
			 *
			 * Results may be undefined if uniform missing from shader.
			 *
			 * @return Uniform reference.
			 */
			const Uniform& getUniformModelview() const
			{
				return _modelview;
			}

			/** \brief Get internal attribute.
			 *
			 * Results may be undefined if uniform missing from shader.
			 *
			 * @return Uniform reference.
			 */
			const Uniform& getUniformNormalmap() const
			{
				return _normalmap;
			}

			/** \brief Get internal attribute.
			 *
			 * Results may be undefined if uniform missing from shader.
			 *
			 * @return Uniform reference.
			 */
			const Uniform& getUniformProjection() const
			{
				return _projection;
			}

			/** \brief Get internal attribute.
			 *
			 * Results may be undefined if uniform missing from shader.
			 *
			 * @return Uniform reference.
			 */
			const Uniform& getUniformTex() const
			{
				return _tex;
			}

			/** \brief Get internal attribute.
			 *
			 * Results may be undefined if uniform missing from shader.
			 *
			 * @return Uniform reference.
			 */
			const Uniform& getUniformTparams() const
			{
				return _tparams;
			}

			/** \brief Get internal attribute.
			 *
			 * Results may be undefined if uniform missing from shader.
			 *
			 * @return Uniform reference.
			 */
			const Uniform& getUniformTransform() const
			{
				return _transform;
			}

		public:
			/** \brief Constructor.
			 *
			 * @param pfname Filename to open.
			 */
			Shader(const std::string &pfname);

			/** \brief Destructor. */
			~Shader();

		public:
			/** \brief Bind this shader for use.
			 *
			 * @return True if bound, false if already bound.
			 */
			bool bind() const;

			/** \brief Compile this shader.
			 *
			 * Despite being public, no-one else but the shader should ever call'
			 * this method.
			 */
			void compile();

			/** \brief Get a named attribute.
			 *
			 * @param name Name of the attribute.
			 * @return Pointer to the named attribute or NULL.
			 */
			const Attribute* getAttribute(const std::string &name) const;

			/** \brief Get a named uniform.
			 *
			 * @param name Name of the uniform.
			 * @return Pointer to the named uniform or NULL.
			 */
			const Uniform* getUniform(const std::string &name) const;

			/** \brief Load shader from disk.
			 *
			 * Will also compile the shader. Due to this, shaders should not be
			 * reserved prior to initializing the OpenGL context.
			 *
			 * @param filename Filename to load from.
			 */
			void load(const std::string &filename);

		private:
			/** \brief Unreserve. */
			void unreserve();

			/** \brief Replace an attribute id within the shader.
			 *
			 * @param op Attribute to replace.
			 */
			void updateAttribute(Attribute &op);

			/** \brief Replace an uniform id within the shader.
			 *
			 * @param op Uniform to replace.
			 */
			void updateUniform(Uniform &op);

		public:
			/** \brief Get the currently active shader.
			 *
			 * @return Currently active shader.
			 */
			static inline const Shader* getCurrent()
			{
				return current_shader;
			}

		public:
			/** \brief Output this to a stream.
			 *
			 * @param ss Target stream.
			 * @return Stream after input.
			 */
			std::ostream& put(std::ostream &ss) const;

			/** \brief Output a surface into a stream.
			 *
			 * @param lhs Left-hand-side operand.
			 * @param rhs Right-hand-side operand.
			 * @return Modified stream.
			 */
			friend inline std::ostream& operator<<(std::ostream &lhs,
					const Shader &rhs)
			{
				return rhs.put(lhs);
			}
	};

	/** Convenience typedef. */
	typedef boost::shared_ptr<Shader> ShaderSptr;
}

#endif
