#include "gfx/shader.hpp"

#include "data/generic.hpp"
#include "thr/dispatch.hpp"
#include "ui/generic.hpp"

#include "boost/property_tree/ptree.hpp"
#include "boost/property_tree/xml_parser.hpp"

static const unsigned GL_LOG_INITIAL_SIZE = 4096;

using namespace gfx;
namespace pt = boost::property_tree;

Shader const *Shader::current_shader = NULL;

/** \brief Get a program log.
 *
 * @param op Program id.
 * @return Program info log as a string.
 */
std::string get_program_log(unsigned op)
{
	GLsizei logsize = GL_LOG_INITIAL_SIZE,
					reqsize;
	char *log = NULL;

	do {
		delete[] log;
		log = new char[logsize];
		glGetProgramInfoLog(op, logsize, &reqsize, log);
	} while(reqsize >= logsize);

	std::string ret(log);
	delete[] log;
	return ret;
}

/** \brief Get a shader log.
 *
 * @param op Shader id.
 * @return Shader info log as a string.
 */
std::string get_shader_log(unsigned op)
{
	GLsizei logsize = GL_LOG_INITIAL_SIZE,
					reqsize;
	char *log = NULL;

	do {
		delete[] log;
		log = new char[logsize];
		glGetShaderInfoLog(op, logsize, &reqsize, log);
	} while(reqsize >= logsize);

	std::string ret(log);
	delete[] log;
	return ret;
}

/** \brief Create a program out of two shaders.
 *
 * @param vprog Vertex shader.
 * @param fprog Fragmen shader.
 * @return Program that was successfully created or zero.
 */
unsigned create_program(unsigned vprog, unsigned fprog,
		const AttributeMap &attr)
{
	unsigned ret = glCreateProgram();

	// Ensure linear attribute fill.
	unsigned idx = 0;
	BOOST_FOREACH(const AttributeMap::value_type &vv, attr)
	{
		glBindAttribLocation(ret, idx, vv.second.getName().c_str());
		++idx;
	}

	glAttachShader(ret, vprog);
	glAttachShader(ret, fprog);
	glLinkProgram(ret);

	GLint compile_info;
	glGetProgramiv(ret, GL_LINK_STATUS, &compile_info);
	if(GL_TRUE != compile_info)
	{
		std::string log = get_program_log(ret);
		glDeleteProgram(ret);

		std::stringstream err;
		err << "shader link failed: " << log;
		BOOST_THROW_EXCEPTION(std::runtime_error(err.str().c_str()));
		return 0;
	}

	return ret;
}

/** \brief Create a shader of given type and data.
 *
 * @param type Shader type.
 * @param hdr Shader header.
 * @param code Shader code.
 * @return Shader that was successfully created or zero.
 */
unsigned create_shader(unsigned type, const std::string &hdr,
		const std::string &code)
{
	unsigned ret = glCreateShader(type);
	const char *data[] = { hdr.c_str(), code.c_str() };

	glShaderSource(ret, 2, data, NULL);
	glCompileShader(ret);

	GLint compile_info;
	glGetShaderiv(ret, GL_COMPILE_STATUS, &compile_info);
	if(GL_TRUE != compile_info)
	{
		std::string log = get_shader_log(ret);
		glDeleteShader(ret);

		std::stringstream err;
		err << "shader compile failed: \"" << hdr << '\n' << code << "\": " << log;
		BOOST_THROW_EXCEPTION(std::runtime_error(err.str().c_str()));
		return 0;
	}

	return ret;
}

Shader::Shader(const std::string &filename) :
	_vsid(0),
	_fsid(0),
	_id(0)
{
	this->load(filename);
}

Shader::~Shader()
{
	this->unreserve();
}

bool Shader::bind() const
{
	if(current_shader != this)
	{
		glUseProgram(_id);
		current_shader = this;
		return true;
	}
	return false;
}

void Shader::compile()
{
	// Compile now when done.
	std::stringstream hdr_attribute, hdr_uniform;
	BOOST_FOREACH(const AttributeMap::value_type &vv, _attributes)
	{
		hdr_attribute << "attribute " << vv.second.getType() << ' ' << vv.first <<
			";\n";
	}
	BOOST_FOREACH(const UniformMap::value_type &vv, _uniforms)
	{
		hdr_uniform << "uniform " << vv.second.getType() << ' ' << vv.first << ";\n";
	}

	_vsid = create_shader(GL_VERTEX_SHADER,
			hdr_attribute.str() + hdr_uniform.str(),
			_vshader);
	if(!_vsid)
	{
		unreserve();
		BOOST_THROW_EXCEPTION(std::runtime_error("shader id was 0"));
	}
	_fsid = create_shader(GL_FRAGMENT_SHADER,
			hdr_uniform.str(),
			_fshader);
	if(!_fsid)
	{
		unreserve();
		BOOST_THROW_EXCEPTION(std::runtime_error("shader id was 0"));
	}
	_id = create_program(_vsid, _fsid, _attributes);
	if(!_id)
	{
		unreserve();
		BOOST_THROW_EXCEPTION(std::runtime_error("program id was 0"));
	}

#if defined(ENABLE_DEBUG)
	std::cout << *this << std::endl;
#endif

	// Compiling succeeded, get all attribute and uniform locations.
	BOOST_FOREACH(AttributeMap::value_type &vv, _attributes)
	{
		this->updateAttribute(vv.second);
	}
	BOOST_FOREACH(UniformMap::value_type &vv, _uniforms)
	{
		this->updateUniform(vv.second);
	}
}

const Attribute* Shader::getAttribute(const std::string &name) const
{
	AttributeMap::const_iterator iter = _attributes.find(name);
	if(_attributes.end() == iter)
	{
		return NULL;
	}
	return &((*iter).second);
}

const Uniform* Shader::getUniform(const std::string &name) const
{
	UniformMap::const_iterator iter = _uniforms.find(name);
	if(_uniforms.end() == iter)
	{
		return NULL;
	}
	return &((*iter).second);
}

void Shader::load(const std::string &pfname)
{
	ui::log(std::string("loading shader: '") + pfname + std::string("'"));

	this->unreserve();

	_fshader.assign("");
	_vshader.assign("");
	_attributes.clear();
	_uniforms.clear();

	data::shristr pfile(data::open_read(pfname));
	pt::ptree xtree;
	pt::read_xml(*pfile, xtree);

	pt::ptree::value_type root = xtree.front();
	if(root.first.compare("shader"))
	{
		std::stringstream err;
		err << "unknown root type: " << root.first;
		BOOST_THROW_EXCEPTION(std::invalid_argument(err.str()));
	}

	BOOST_FOREACH(pt::ptree::value_type &vv, root.second)
	{
		const std::string &type = vv.first;
		const pt::ptree &subtree = vv.second;

		if(!type.compare("attribute"))
		{
			std::string pname = subtree.get<std::string>("name"),
				ptype = subtree.get<std::string>("type");
			_attributes[pname] = Attribute(pname, ptype);
		}
		else if(!type.compare("uniform"))
		{
			std::string pname = subtree.get<std::string>("name"),
				ptype = subtree.get<std::string>("type");
			_uniforms[pname] = Uniform(pname, ptype);
		}
		else if(!type.compare("vertexprogram"))
		{
			_vshader = subtree.get<std::string>("");
		}
		else if(!type.compare("fragmentprogram"))
		{
			_fshader = subtree.get<std::string>("");
		}
		else if(type.compare("<xmlattr>"))
		{
			std::stringstream err;
			err << "unknown element: " << type;
			BOOST_THROW_EXCEPTION(std::invalid_argument(err.str()));
		}
	}

	thr::wait_privileged(boost::bind(&Shader::compile, this));
}

void Shader::updateAttribute(Attribute &op)
{
	const std::string &pname = op.getName();
	int pid = glGetAttribLocation(_id, pname.c_str());
	if(0 > pid)
	{
		std::ostringstream err;
		err << "no attribute \"" << pname << "\" in shader " << std::endl << *this;
		BOOST_THROW_EXCEPTION(std::runtime_error(err.str()));
		return;
	}
	op.id() = pid;

#if defined(ENABLE_DEBUG)
	std::cout << op << std::endl;
#endif

	// Well-known names.
	if(!pname.compare("bone_ref"))
	{
		_bone_ref = op;
	}
	else if(!pname.compare("bone_weight"))
	{
		_bone_weight = op;
	}
	else if(!pname.compare("color"))
	{
		_color = op;
	}
	else if(!pname.compare("normal"))
	{
		_normal = op;
	}
	else if(!pname.compare("texcoord"))
	{
		_texcoord = op;
	}
	else if(!pname.compare("vertex"))
	{
		_vertex = op;
	}
}

void Shader::updateUniform(Uniform &op)
{
	const std::string &pname = op.getName();
	int pid = glGetUniformLocation(_id, pname.c_str());
	if(0 > pid)
	{
		std::ostringstream err;
		err << "no uniform \"" << pname << "\" in shader " << std::endl << *this;
		BOOST_THROW_EXCEPTION(std::runtime_error(err.str()));
		return;
	}
	op.id() = pid;

#if defined(ENABLE_DEBUG)
	std::cout << op << std::endl;
#endif

	if(!pname.compare("light_ambient"))
	{
		_light_ambient = op;
	}
	else if(!pname.compare("light_diffuse"))
	{
		_light_diffuse = op;
	}
	else if(!pname.compare("light_dir"))
	{
		_light_dir = op;
	}
	else if(!pname.compare("modelview"))
	{
		_modelview = op;
	}
	else if(!pname.compare("normalmap"))
	{
		_normalmap = op;
	}
	else if(!pname.compare("projection"))
	{
		_projection = op;
	}
	else if(!pname.compare("tex"))
	{
		_tex = op;
	}
	else if(!pname.compare("tparams"))
	{
		_tparams = op;
	}
	else if(!pname.compare("transform"))
	{
		_transform = op;
	}
}

void Shader::unreserve()
{
	if(_id)
	{
		// It's not possible that the shader is reserved while and programs are
		// not reserved or attached into it.
		glDetachShader(_id, _vsid);
		glDetachShader(_id, _fsid);
		glDeleteProgram(_id);
		_id = 0;
	}
	// However, either vertex shader of fragment program may exist even if the
	// shader does not.
	if(_vsid)
	{
		glDeleteShader(_vsid);
		_vsid = 0;
	}
	if(_fsid)
	{
		glDeleteShader(_fsid);
		_fsid = 0;
	}
	_id = _vsid = _fsid = 0;
}

std::ostream& Shader::put(std::ostream &ss) const
{
	return ss << "== Vertex ==\n" << _vshader << "\n== Fragment ==\n" <<
		_fshader << "\n== id: " << _id << " ==";
}

