#include "gfx/mesh.hpp"

#include "data/generic.hpp"
#include "thr/dispatch.hpp"
#include "ui/generic.hpp"

#include <boost/filesystem.hpp>
#include <boost/property_tree/xml_parser.hpp>

using namespace gfx;
namespace fs = boost::filesystem;
namespace pt = boost::property_tree;

MeshStore gfx::mesh_store;

Mesh::Mesh() :
	m_offset(0.0f, 0.0f, 0.0f) { }

Mesh::Mesh(const std::string &pfname) :
	m_offset(0.0f, 0.0f, 0.0f)
{
	this->load(pfname);
}

void Mesh::addTexture(const std::string &id, const void *tex)
{
	if(id.compare(0, 7, "texture"))
	{
		std::stringstream sstr;
		sstr << "unknown texture id: " << id;
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}
	m_textures.push_back(static_cast<const Texture2D*>(tex));
}

void Mesh::addTextureFile(const std::string &type, const std::string &pfname,
		bool clamp)
{
	Texture2DStore::container_type &con =
		texture_2d_store.load<Texture2D>(pfname, clamp);
	this->addTexture(type, con[0].get());
}

void Mesh::calcNormals()
{
	unsigned vsize = static_cast<unsigned>(m_vertex.size());
	m_normal.resize(0);
	m_normal.resize(vsize, math::vec3f(0.0f, 0.0f, 0.0f));
	std::vector<unsigned> refcnts;;
	refcnts.resize(vsize, 0);

	TriVec faces;
	m_lod.collect(faces);

	BOOST_FOREACH(const Triangle &tri, faces)
	{
		math::vec3f nor = this->calcNormal(tri);
		unsigned ia = tri.a(),
						 ib = tri.b(),
						 ic = tri.c();
		m_normal[ia] += nor;
		m_normal[ib] += nor;
		m_normal[ic] += nor;
		++(refcnts[ia]);
		++(refcnts[ib]);
		++(refcnts[ic]);
	}

	for(unsigned ii = 0; (ii < vsize); ++ii)
	{
		math::vec3f &nn = m_normal[ii];

		unsigned rcnt = refcnts[ii];
		if(rcnt > 0)
		{
			nn = math::normalize(nn / static_cast<float>(rcnt));
		}
		else
		{
			// Fallback normal points up.
			nn = math::vec3f(0.0f, 1.0f, 0.0f);
		}
	}

	data::stl_trim(m_color);
	data::stl_trim(m_texcoord);
	data::stl_trim(m_vertex);
}

math::rect3f Mesh::getBoundary() const
{
	if(m_vertex.size() <= 0)
	{
		std::stringstream sstr;
		sstr << "boundary requested for mesh that has no vertices";
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}
	math::rect3f ret(m_vertex[0]);
	for(unsigned ii = 1; (ii < m_vertex.size()); ++ii)
	{
		ret.expand(m_vertex[ii]);
	}
	return ret;
}

void Mesh::load(const std::string &pfname)
{
	math::vec3f obj_scale(1.0f, 1.0f, 1.0f);
	bool enable_center = false,
			 enable_into = false,
			 enable_scale = false;

	this->unreserve();

	ui::log(std::string("loading mesh '") + pfname + std::string("'"));

	pt::ptree xtree;
	pt::read_xml(pfname, xtree);
	pt::ptree::value_type root = xtree.front();
	if(root.first.compare("mesh"))
	{
		std::stringstream err;
		err << "unknown root type: " << root.first;
		BOOST_THROW_EXCEPTION(std::invalid_argument(err.str()));
	}

	BOOST_FOREACH(pt::ptree::value_type &vv, root.second)
	{
		const std::string &type = vv.first;
		const pt::ptree &subtree = vv.second;

		if(!type.compare("offset"))
		{
			math::vec3f off(subtree.get<float>("x"),
					subtree.get<float>("y"),
					subtree.get<float>("z"));
			m_offset = math::vec3f(off.y(), off.z(), -off.x());
		}
		else if(!type.compare("scale"))
		{
			enable_scale = true;
			enable_center = subtree.get<bool>("center", 0);
			enable_into = subtree.get<bool>("into", 0);
			obj_scale.x() = subtree.get<float>("x", 1.0f);
			obj_scale.y() = subtree.get<float>("y", 1.0f);
			obj_scale.z() = subtree.get<float>("z", 1.0f);
		}
		else if(!type.compare("vertex"))
		{
			this->readVertex(&subtree);
		}
		else if(!type.compare("face"))
		{
			// Must flip faces for correct orientation.
			m_lod.addFace(Triangle(subtree.get<unsigned>("a"),
						subtree.get<unsigned>("c"),
						subtree.get<unsigned>("b")));
		}
		else if((type.compare(0, 7, "texture") == 0) ||
				(type.compare(0, 9, "normalmap") == 0))
		{
			fs::path subpath(fs::path(pfname).parent_path().parent_path() /
					subtree.get<std::string>(""));
			this->addTextureFile(type, subpath.file_string());
		}
		else if(type.compare(0, 6, "volume") == 0)
		{
			std::stringstream sstr;
			sstr << "can not load volume: " << type;
			BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
		}
	}

	if(enable_scale)
	{
		if(enable_into || enable_center)
		{
			math::rect3f area = this->getBoundary();
			//std::cout << "model area: " << area << std::endl;

			// Center before scaling for easiness.
			if(enable_center)
			{
				math::vec3f center = area.center();
				this->translate(-center);
				area = math::rect3f(area.getMin() - center, area.getMax() - center);
			}

			// Scale into after potential centering.
			if(enable_into)
			{
				math::vec3f size = area.getMax() - area.getMin();
				float common_scale = math::min(
						obj_scale.x() / size.x(),
						obj_scale.y() / size.y(),
						obj_scale.z() / size.z());
				obj_scale = math::vec3f(common_scale, common_scale, common_scale);
			}
		}
		this->scale(obj_scale);
	}

	this->compile();
}

void Mesh::readVertex(const void *subtree)
{
	const pt::ptree *st = reinterpret_cast<const pt::ptree*>(subtree);

	try {
		m_bone_ref.push_back(math::vec4u(st->get<unsigned>("bone_ref.a"),
					st->get<unsigned>("bone_ref.b"),
					st->get<unsigned>("bone_ref.c"),
					st->get<unsigned>("bone_ref.d")));
	} catch(...) { }
	try {
		m_bone_weight.push_back(math::vec4f(st->get<float>("bone_weight.a"),
					st->get<float>("bone_weight.b"),
					st->get<float>("bone_weight.c"),
					st->get<float>("bone_weight.d")));
	} catch(...) { }
	try {
		m_color.push_back(Color(st->get<float>("color.r"),
					st->get<float>("color.g"),
					st->get<float>("color.b"),
					st->get<float>("color.a")));
	} catch(...) { }
	try {
		m_normal.push_back(math::vec3f(st->get<float>("normal.x"),
					st->get<float>("normal.y"),
					st->get<float>("normal.z")));
	} catch(...) { }
	try {
		m_texcoord.push_back(math::vec2f(st->get<float>("texcoord.s"),
					st->get<float>("texcoord.t")));
	} catch(...) { }
	// Vertex MUST exist, others are voluntary.
	math::vec3f ver(st->get<float>("x"),
				st->get<float>("y"),
				st->get<float>("z"));
	// Must 'rotate' to correct orientation.
	m_vertex.push_back(math::vec3f(ver.y(), ver.z(), -ver.x()));
}

void Mesh::scale(const math::vec3f &svec)
{
	BOOST_FOREACH(math::vec3f &vv, m_vertex)
	{
		vv *= svec;
	}
	// Scale is also modified.
	m_offset *= svec;
}

void Mesh::taskElem()
{
	m_elem.update(m_lod);
}

void Mesh::taskTextureFile(const char *type, const std::string &pfname)
{
	this->addTexture(std::string(type), new Texture2D(pfname));
}

void Mesh::taskTextureFileClamped(const char *type, const std::string &pfname)
{
	this->addTexture(std::string(type), new Texture2D(pfname, true));
}

void Mesh::taskTextureGray8(const char *type, const ImageGray8 *img)
{
	this->addTexture(std::string(type), new Texture2D(*img, true));
}

void Mesh::taskTextureGray16(const char *type, const ImageGray16 *img)
{
	this->addTexture(std::string(type), new Texture2D(*img, true));
}

void Mesh::taskTextureRGB(const char *type, const ImageRGB *img)
{
	this->addTexture(std::string(type), new Texture2D(*img, true));
}

void Mesh::taskTextureRGBA(const char *type, const ImageRGBA *img)
{
	this->addTexture(std::string(type), new Texture2D(*img, true));
}

void Mesh::translate(const math::vec3f &tvec)
{
	BOOST_FOREACH(math::vec3f &vv, m_vertex)
	{
		vv += tvec;
	}
}

void Mesh::unreserve()
{
	m_bone_ref.clear();
	m_bone_weight.clear();
	m_color.clear();
	m_normal.clear();
	m_texcoord.clear();
	m_vertex.clear();
	m_lod.unreserve();
	m_textures.clear();
}

MeshStore::MeshStore() :
	data::Store<Mesh>(".mesh", ".mmesh") { }

void MeshStore::loadMeta(std::vector<MeshSptr> &dst,
		const std::string &pfname, const data::StoreCreator<Mesh> &creator)
{
	math::vec3f obj_scale(1.0f, 1.0f, 1.0f);
	bool enable_center = false,
			 enable_into = false,
			 enable_scale = false;

	ui::log(std::string("loading meta-mesh '") + pfname + std::string("'"));

	boost::property_tree::ptree xtree;
	boost::property_tree::read_xml(pfname, xtree);
	boost::property_tree::ptree::value_type root = xtree.front();
	if(root.first.compare("meta-mesh"))
	{
		std::stringstream err;
		err << "unknown root type: " << root.first;
		BOOST_THROW_EXCEPTION(std::invalid_argument(err.str()));
	}

	BOOST_FOREACH(boost::property_tree::ptree::value_type &vv, root.second)
	{
		const std::string &type = vv.first;
		const boost::property_tree::ptree &subtree = vv.second;

		if(!type.compare("scale"))
		{
			enable_scale = true;
			enable_center = subtree.get<bool>("center", 0);
			enable_into = subtree.get<bool>("into", 0);
			obj_scale.x() = subtree.get<float>("x", 1.0f);
			obj_scale.y() = subtree.get<float>("y", 1.0f);
			obj_scale.z() = subtree.get<float>("z", 1.0f);
		}
		else if(!type.compare("mesh"))
		{
			fs::path subpath(fs::path(pfname).parent_path() /
					subtree.get<std::string>(""));
			dst.push_back(MeshSptr(creator.create(subpath.file_string())));
		}
	}

	if(dst.empty())
	{
		// Store will throw the exception for this.
		return;
	}

	if(enable_scale)
	{
		if(enable_into || enable_center)
		{
			math::rect3f area = dst[0]->getBoundary();
			for(unsigned ii = 1; (ii < dst.size()); ++ii)
			{
				area.expand(dst[ii]->getBoundary());
			}

			// Center before scaling for easiness.
			if(enable_center)
			{
				math::vec3f center = area.center();
				BOOST_FOREACH(MeshSptr &vv, dst)
				{
					vv->translate(-center);
				}
				area = math::rect3f(area.getMin() - center, area.getMax() - center);
			}

			// Scale into after potential centering.
			if(enable_into)
			{
				math::vec3f size = area.getMax() - area.getMin();
				float common_scale = math::min(
						obj_scale.x() / size.x(),
						obj_scale.y() / size.y(),
						obj_scale.z() / size.z());
				obj_scale = math::vec3f(common_scale, common_scale, common_scale);
			}
		}
		BOOST_FOREACH(MeshSptr &vv, dst)
		{
			vv->scale(obj_scale);
		}
		// Meshes have been already compiled on load. if we don't scale, we don't
		// need to recompile.
		BOOST_FOREACH(MeshSptr &vv, dst)
		{
			vv->compile();
		}
	}

	// Try to load submeshes now that object scale is known.
	for(unsigned ii = 0; (true); ++ii)
	{
		try
		{
			std::stringstream sstr;
			sstr << loaderize(pfname) << '-' << ii << m_single;
			Mesh *msh = creator.create(sstr.str());
			if(enable_scale)
			{
				msh->scale(obj_scale);
				msh->compile();
			}
			this->put(canonize(sstr.str()), msh);
		}
		catch(...)
		{
			break;
		}
	}
}

