#ifndef GFX_COLOR_HPP
#define GFX_COLOR_HPP

#include "gfx/generic.hpp"
#include "math/generic.hpp"

namespace gfx
{
	/** \brief Representation of one color.
	 *
	 * The colors are saved as single-precision floating point. While they can
	 * be modified as integers, this happens in a floating-point level and will
	 * sacrifice precision.
	 *
	 * The operations done to the color values are not bounded. The user is
	 * responsible for keeping the values legal.
	 */
	struct Color
	{
		/** Internal array. */
		float _array[4];

		/** \brief Accessor.
		 *
		 * @return Component.
		 */
		inline float& r()
		{
			return _array[0];
		}

		/** \brief Accessor.
		 *
		 * @return Component.
		 */
		inline const float& r() const
		{
			return _array[0];
		}

		/** \brief Accessor.
		 *
		 * @return Component.
		 */
		inline float& g()
		{
			return _array[1];
		}

		/** \brief Accessor.
		 *
		 * @return Component.
		 */
		inline const float& g() const
		{
			return _array[1];
		}

		/** \brief Accessor.
		 *
		 * @return Component.
		 */
		inline float& b()
		{
			return _array[2];
		}

		/** \brief Accessor.
		 *
		 * @return Component.
		 */
		inline const float& b() const
		{
			return _array[2];
		}

		/** \brief Accessor.
		 *
		 * @return Component.
		 */
		inline float& a()
		{
			return _array[3];
		}

		/** \brief Accessor.
		 *
		 * @return Component.
		 */
		inline const float& a() const
		{
			return _array[3];
		}

		/** \brief Integer accessor.
		 *
		 * @return Component.
		 */
		inline int ri() const
		{
			return math::lround(_array[0] * 255.0f);
		}

		/** \brief Integer accessor.
		 *
		 * @return Component.
		 */
		inline int gi() const
		{
			return math::lround(_array[1] * 255.0f);
		}

		/** \brief Integer accessor.
		 *
		 * @return Component.
		 */
		inline int bi() const
		{
			return math::lround(_array[2] * 255.0f);
		}

		/** \brief Integer accessor.
		 *
		 * @return Component.
		 */
		inline int ai() const
		{
			return math::lround(_array[3] * 255.0f);
		}

		/** \brief Unsigned byte representation.
		 *
		 * Takes some time, so for most purposes it's better to precalculate this.
		 *
		 * @return Color representation fitting into one doubleword.
		 */
		inline uint32_t ub4()
		{
			uint32_t ret;
			uint8_t *ret8 = reinterpret_cast<uint8_t*>(&ret);
			ret8[0] = static_cast<uint8_t>(this->ri());
			ret8[1] = static_cast<uint8_t>(this->gi());
			ret8[2] = static_cast<uint8_t>(this->bi());
			ret8[3] = static_cast<uint8_t>(this->ai());
			return ret;
		}

		/** \brief Integer set.
		 *
		 * @param op New component.
		 */
		inline void setR(int op)
		{
			_array[0] = static_cast<float>(op) / 255.0f;
		}

		/** \brief Integer set.
		 *
		 * @param op New component.
		 */
		inline void setG(int op)
		{
			_array[1] = static_cast<float>(op) / 255.0f;
		}

		/** \brief Integer set.
		 *
		 * @param op New component.
		 */
		inline void setB(int op)
		{
			_array[2] = static_cast<float>(op) / 255.0f;
		}

		/** \brief Integer set.
		 *
		 * @param op New component.
		 */
		inline void setA(int op)
		{
			_array[3] = static_cast<float>(op) / 255.0f;
		}

		/** \brief Set all values.
		 *
		 * Alpha can be omitted in which case it is 255.
		 *
		 * @param pr Red component.
		 * @param pg Green component.
		 * @param pb Blue component.
		 * @param pa Alpha component.
		 */
		inline void set(int pr, int pg, int pb, int pa = 255)
		{
			this->setR(pr);
			this->setG(pg);
			this->setB(pb);
			this->setA(pa);
		}

		/** \brief Set all values.
		 *
		 * Alpha can be omitted in which case it is 1.
		 *
		 * @param pr Red component.
		 * @param pg Green component.
		 * @param pb Blue component.
		 * @param pa Alpha component.
		 */
		inline void set(float pr, float pg, float pb, float pa = 1.0f)
		{
			_array[0] = pr;
			_array[1] = pg;
			_array[2] = pb;
			_array[3] = pa;
		}

		/** \brief Empty constructor.
		 *
		 * Only so that STL vector resize() would work.
		 */
		inline Color() { }

		/** \brief Constructor for HTML notation.
		 *
		 * @param noration.
		 */
		Color(const char *notation);

		/** \brief Constructor.
		 *
		 * Alpha can be omitted in which case it is 255.
		 *
		 * @param pr Red component.
		 * @param pg Green component.
		 * @param pb Blue component.
		 * @param pa Alpha component.
		 */
		inline Color(int pr, int pg, int pb, int pa = 255)
		{
			this->setR(pr);
			this->setG(pg);
			this->setB(pb);
			this->setA(pa);
		}

		/** \brief Constructor.
		 *
		 * Alpha can be omitted in which case it is 1.
		 *
		 * @param pr Red component.
		 * @param pg Green component.
		 * @param pb Blue component.
		 * @param pa Alpha component.
		 */
		inline Color(float pr, float pg, float pb, float pa = 1.0f)
		{
			_array[0] = pr;
			_array[1] = pg;
			_array[2] = pb;
			_array[3] = pa;
		}

		/** Create from CMYK values.
		 *
		 * Note: This is NOT a true CMYK conversion algorithm.
		 *
		 * @param pc Cyan component.
		 * @param pm Magenta component.
		 * @param py Yellow component.
		 * @param pk Black component.
		 * @param pa Alpha component.
		 * @return New color.
		 */
		static Color fromCMYK(float pc, float pm, float py, float pk, float pa);

		/** \brief Addition operator.
		 *
		 * @param rhs Right-hand-side operand.
		 * @return New color.
		 */
		inline Color operator+(const Color &rhs) const
		{
			return Color(_array[0] + rhs.r(),
					_array[1] + rhs.g(),
					_array[2] + rhs.b(),
					_array[3] + rhs.a());
		}

		/** \brief Multiplication operator.
		 *
		 * @param rhs Right-hand-side operand.
		 * @return New color.
		 */
		inline Color operator*(float rhs) const
		{
			return Color(_array[0] * rhs,
					_array[1] * rhs,
					_array[2] * rhs,
					_array[3] * rhs);
		}
	};
}

#endif
