#ifndef GFX_ATTRIBUTE_HPP
#define GFX_ATTRIBUTE_HPP

#include "gfx/generic.hpp"

namespace gfx
{
	/** \brief Per-vertex attribute.
	 *
	 * Note that attributes do not know their names. They exist only as
	 * referenced by shaders.
	 *
	 * Only the id of the attribute can be modified after it's creation, but
	 * should not be touched unless the accompanying shader is recompiled.
	 */
	struct Attribute
	{
		protected:
			/** Name. */
			std::string _name;

			/** Type. */
			std::string _type;

			/** Attribute id. */
			GLuint _id;

		public:
			/** \brief Accessor.
			 *
			 * @return Type of this.
			 */
			inline const std::string& getType() const
			{
				return _type;
			}

			/** \brief Accessor.
			 *
			 * @return Name of this.
			 */
			inline const std::string& getName() const
			{
				return _name;
			}

			/** \brief Accessor.
			 *
			 * @return Id of this.
			 */
			inline GLuint& id()
			{
				return _id;
			}

			/** \brief Accessor.
			 *
			 * @return Id of this.
			 */
			inline const GLuint& id() const
			{
				return _id;
			}

			/** Assignment.
			 *
			 * @param rhs Right-hand-size operand.
			 */
			inline const Attribute& operator=(const Attribute &rhs)
			{
				_name = rhs._name;
				_type = rhs._type;
				_id = rhs._id;
				return *this;
			}

		public:
			/** \brief Empty constructor. */
			inline Attribute() : _id(0) { }

			/** \brief Constructor.
			 *
			 * @param src Previous incarnation.
			 * @param pid New id.
			 */
			inline Attribute(const Attribute &src, GLuint pid) :
				_name(src._name),
				_type(src._type),
				_id(pid) { }

			/** \brief Constructor.
			 *
			 * @param pname Name
			 * @param ptype Type.
			 */
			inline Attribute(const std::string &pname, const std::string &ptype) :
				_name(pname),
				_type(ptype),
				_id(0) { }

		public:
			/** \brief Disables this vertex attribute array.
			 */
			inline void disable() const
			{
				glDisableVertexAttribArray(_id);
			}

			/** \brief Enables this vertex attribute array.
			 */
			inline void enable() const
			{
				glEnableVertexAttribArray(_id);
			}

		public:
			/** \brief Output a surface into a stream.
			 *
			 * @param lhs Left-hand-side operand.
			 * @param rhs Right-hand-side operand.
			 * @return Modified stream.
			 */
			friend inline std::ostream& operator<<(std::ostream &lhs,
					const Attribute &rhs)
			{
				return lhs << "Attribute " << rhs.getType().c_str() << " \"" << rhs.getName().c_str() << "\": " << rhs.id();
			}
	};
}

#endif
