#ifndef GFX_ARRAY_HPP
#define GFX_ARRAY_HPP

#include "gfx/attribute.hpp"
#include "gfx/color.hpp"
#include "math/vec.hpp"

namespace gfx
{
	/** \brief Array base class.
	 *
	 * Array is a class used to contain data for feeding it to the video card.
	 * The array itself does not handle the actual potential vertex buffer
	 * object (VBO), instead, Buffer class is used for that.
	 *
	 * @param T Elementary type.
	 * @param C Number of elements within.
	 */
	template <typename T, unsigned C> class Array
	{
		protected:
			/** Internal array. */
			T* _array;

			/** Size of the array (in elements). */
			unsigned _array_size;

		protected:
			/** \brief Empty constructor.
			 *
			 * Array created in this manner is not usable right away. It must be
			 * reserved first.
			 */
			Array() :
				_array(NULL),
				_array_size(0) { }

			/** \brief Constructor.
			 *
			 * @param count Number of elements to reserve.
			 * @param buftype Type of buffer, usually GL_ARRAY_BUFFER.
			 */
			Array(unsigned count) :
				_array(NULL), _array_size(0)
			{
				this->reserve(count);
			}

			/** \brief Destructor. */
			~Array()
			{
				unreserve();
			}

		public:
			/** \brief Reserve data for this array.
			 *
			 * If array data already exists, it will be deleted beforehand.
			 *
			 * The size of the array is dependant on the count of elementary units
			 * per one logical element for this array.
			 *
			 * @param count Number of elements to reserve.
			 * @return Size of the created array in bytes.
			 */
			unsigned reserve(unsigned count)
			{
				this->unreserve();

				_array_size = count;
				_array = new T[_array_size * C];
				return _array_size * C * static_cast<unsigned>(sizeof(T));
			}

			/** \brief Unreserve all data. */
			void unreserve()
			{
				delete[] _array;
				_array = NULL;
			}

		public:
			/** \brief Access operator for elementary type within.
			 *
			 * @param idx Index to access.
			 */
			inline T& operator[](unsigned idx)
			{
				return _array[idx];
			}

			/** \brief Access operator for elementary type within.
			 *
			 * @param idx Index to access.
			 */
			inline const T& operator[](unsigned idx) const
			{
				return _array[idx];
			}

		public:
			/** \brief Get pointer to the data within this.
			 *
			 * @return Data pointer.
			 */
			inline const T* getData() const
			{
				return _array;
			}

			/** \brief Get the size of this.
			 *
			 * Size is returned in full elements. To get size in elementary types,
			 * multiply this with get_elementary_count().
			 *
			 * @return Size in elements.
			 */
			inline unsigned getSize() const
			{
				return _array_size;
			}

			/** \brief Get the size of this.
			 *
			 * @return Size in bytes.
			 */
			inline unsigned getSizeBytes() const
			{
				return _array_size * sizeof(T) * C;
			}

		public:
			/** \brief Get the number of elementary types per element.
			 *
			 * @return Elementary type count as unsigned.
			 */
			static inline unsigned getElementaryCount()
			{
				return C;
			}
	};

	/** \brief Element buffer.
	 *
	 * Used for drawing instead of supplying the data.
	 */
	template <typename T> class ArrayE : public Array<T, 1>
	{
		public:
			/** \brief Empty constructor. */
			ArrayE() :
				Array<T, 1>() { }

			/** \brief Constructor.
			 *
			 * @param count Number of elements to reserve.
			 */
			ArrayE(unsigned count) :
				Array<int, 1>(count) { }

		public:
			/** \brief Draw according to the elements in this.
			 *
			 * @param type What kind of elements are we drawing?
			 * @param cnt How many elements to feed.
			 */
			inline void draw(unsigned type, unsigned cnt) const
			{
				glDrawElements(type, cnt, get_opengl_type<T>(),
						Array<T, 1>::_array);
			}

			/** \brief Draw according to the elements in this.
			 *
			 * Like draw(unsigned, unsigned), but feeds all elements.
			 *
			 * @param type What kind of elements are we drawing?
			 */
			inline void draw(unsigned type) const
			{
				this->draw(type, Array<T, 1>::_array_size);
			}
	};

	/** Convenience typedef. */
	typedef ArrayE<uint32_t> ArrayElem;

	/** \brief Vertex attribute buffer.
	 *
	 * This class is used to instantiate the distinct vertex attribute arrays.
	 */
	template <typename T, unsigned C> class ArrayA : public Array<T, C>
	{
		public:
			/** \brief Empty constructor. */
			ArrayA() :
				Array<T, C>() { }

			/** \brief Constructor.
			 *
			 * @param count Number of elements to reserve.
			 */
			ArrayA(unsigned count) :
				Array<T, C>(count) { }

		public:
			/** \brief Feed this buffer.
			 *
			 * @param idx Vertex attribute index to feed as.
			 */
			inline void feed(GLuint idx) const
			{
				glEnableVertexAttribArray(idx);
				glVertexAttribPointer(idx, C, get_opengl_type<T>(), GL_FALSE, 0,
						Array<T, C>::_array);
			}

			/** \brief Feed this buffer.
			 *
			 * Extract the index from the attribute abstraction.
			 *
			 * @param op Attribute.
			 */
			inline void feed(const Attribute &op)
			{
				this->feed(op.id());
			}
	};

	/** Convenience typedef. */
	typedef ArrayA<float, 2> ArrayA2f;

	/** Convenience typedef. */
	typedef ArrayA<float, 3> ArrayA3f;

	/** Convenience typedef. */
	typedef ArrayA<float, 4> ArrayA4f;

	/** Convenience typedef. */
	typedef ArrayA<int, 4> ArrayA4i;

	/** \brief Billboard element.
	 *
	 * Elementary billboard type.
	 *
	 * The billboard stores color in a single uint32, when assigning new
	 * billboards, precalculate color to speed up the process.
	 */
	typedef struct BillboardElementStruct
	{
		public:
			/** Texcoord morph. */
			uint32_t m_morph;

			/** Texcoord offset s. */
			float m_s;

			/** Texcoord offset t. */
			float m_t;

			/** Color. */
			uint32_t m_col;
			
			/** X coordinate. */
			float m_x;

			/** Y coordinate. */
			float m_y;

			/** Z coordinate. */
			float m_z;

			/** Billboard size. */
			float m_tex_size;

		public:
			/** Default value for morph. */
			static const uint32_t MORPH_DEFAULT = 0x007F007F;

		public:
			/** \brief Empty constructor. */
			BillboardElementStruct() { }

			/** \brief Constructor.
			 *
			 * @param col Color.
			 * @param pos Position.
			 * @param ts Texture size.
			 */
			BillboardElementStruct(uint32_t col, const math::vec3f &pos, float ts) :
				m_morph(MORPH_DEFAULT),
				m_s(0.0f),
				m_t(0.0f),
				m_col(col),
				m_x(pos.x()),
				m_y(pos.y()),
				m_z(pos.z()),
				m_tex_size(ts) { }

			/** \brief Constructor.
			 *
			 * @param tm Texture morph.
			 * @param tt Texture coords.
			 * @param col Color.
			 * @param pos Position.
			 * @param ts Texture size.
			 */
			BillboardElementStruct(uint32_t tm, const math::vec2f &tt,
					uint32_t col, const math::vec3f &pos, float ts) :
				m_morph(tm),
				m_s(tt.x()),
				m_t(tt.y()),
				m_col(col),
				m_x(pos.x()),
				m_y(pos.y()),
				m_z(pos.z()),
				m_tex_size(ts) { }

		public:
			/** Set all content.
			 *
			 * @param tm Texture morph.
			 * @param tt Texture coords.
			 * @param col Color.
			 * @param pos Position.
			 * @param ts Texture size.
			 */
			inline void set(uint32_t tm, const math::vec2f &tt, uint32_t col,
					const math::vec3f &pos, float ts)
			{
				m_morph = tm;
				m_s = tt.x();
				m_t = tt.y();
				m_col = col;
				m_x = pos.x();
				m_y = pos.y();
				m_z = pos.z();
				m_tex_size = ts;
			}

			/** Set all content.
			 *
			 * @param col Color.
			 * @param pos Position.
			 * @param ts Texture size.
			 */
			inline void set(uint32_t col, const math::vec3f &pos, float ts)
			{
				this->set(MORPH_DEFAULT, math::vec2f(0.0f, 0.0f), col, pos, ts);
			}
	} BillboardElement;

	/** \brief Generic complex array struct.
	 */
	template <typename T> class ArrayGeneric
	{
		protected:
			/** Internal array. */
			T *m_array;

			/** Current insert pointer. */
			T *m_insert;

			/** Current insert count. */
			unsigned m_insert_count;

			/** Array size (elements). */
			unsigned m_size;

		public:
			/** \brief Accessor.
			 *
			 * @return Current number of elements.
			 */
			inline unsigned getCount() const
			{
				return m_insert_count;
			}

		public:
			/** \brief Constructor. */
			ArrayGeneric(unsigned size) :
				m_array(NULL),
				m_size(size)
			{
				if(0 < m_size)
				{
					m_array = new T[m_size];
				}
				this->reset();
			}

			/** \brief Destructor. */
			~ArrayGeneric()
			{
				delete[] m_array;
			}

		public:
			/** \brief Reset the feeding.
			 */
			inline void reset()
			{
				m_insert = m_array;
				m_insert_count = 0;
			}
	};

	/** \brief Billboard array struct.
	 *
	 * A specific form used for drawing billboards.
	 *
	 * Doesn't really share code with the other arrays.
	 */
	class ArrayBillboard : public ArrayGeneric<BillboardElement>
	{
		public:
			/** \brief Constructor. */
			ArrayBillboard(unsigned size);

			/** \brief Destructor. */
			~ArrayBillboard() { }

		public:
			/** \brief Feed this array.
			 *
			 * @param am Texture morph attribute.
			 * @param at Texcoord attribute.
			 * @param ac Color attribute.
			 * @param av Vertex attribute.
			 */
			void feed(unsigned am, unsigned at, unsigned ac, unsigned av);

		public:
			/** \brief Set an element within the array.
			 *
			 * @param tm Texture morph.
			 * @param tt Texture coords.
			 * @param col Color.
			 * @param pos Position.
			 * @param ts Texture size.
			 */
			inline void add(uint32_t tm, const math::vec2f &tt, uint32_t col,
					const math::vec3f &pos, float ts)
			{
				if(m_insert_count >= m_size)
				{
					return;
				}
				m_insert->set(tm, tt, col, pos, ts);
				++m_insert;
				++m_insert_count;
			}

			/** \brief Set an element within the array.
			 *
			 * @param col Color.
			 * @param pos Position.
			 * @param ts Texture size.
			 */
			inline void add(uint32_t col, const math::vec3f &pos, float ts)
			{
				this->add(BillboardElement::MORPH_DEFAULT, math::vec2f(0.0f, 0.0f),
						col, pos, ts);
			}

			/** \brief Feed this array.
			 *
			 * @param am Texture morph attribute.
			 * @param at Texcoord attribute.
			 * @param ac Color attribute.
			 * @param av Vertex attribute.
			 */
			inline void feed(const Attribute &am, const Attribute &at,
					const Attribute &ac, const Attribute &av)
			{
				this->feed(am.id(), at.id(), ac.id(), av.id());
			}
	};
}

#endif
