#include "ft_glyph.hpp"
#include "ft_face.hpp"

//#include <limits.h>
#include <math.h>
#include <sstream>
//#include <stdlib.h>

//#include <algorithm>
//#include <iostream>

#include <boost/throw_exception.hpp>

FT_Library FtFace::ftlib = NULL;

/** \brief Distance between two coordinates.
 *
 * @param x1 First X coordinate.
 * @param y1 First Y coordinate.
 * @param x2 Second X coordinate.
 * @param y2 Second Y coordinate.
 * @return Distance as integer.
 */
static int idist(int x1, int y1, int x2, int y2)
{
	int dx = x2 - x1,
			dy = y2 - y1;
	return static_cast<int>(
			lrintf(sqrtf(static_cast<float>(dx * dx + dy * dy))));
}

/** \brief Return the value at a given location in a freetype glyph.
 *
 * Will return 0 if the point is 'outside' the bitmap.
 *
 * @param glyph Glyph to examine.
 * @param px X coordinate.
 * @param py Y coordinate.
 * @param size Big size.
 * @return Value found.
 */
static uint8_t get_ftbitmap_value(const FT_GlyphSlot glyph, int px, int py,
		int size)
{
	FT_Bitmap *bitmap = &(glyph->bitmap);

	int ww = bitmap->width,
			hh = bitmap->rows,
			minx = (size - ww) / 2,
			miny = (size - hh) / 2,
			maxx = minx + ww - 1,
			maxy = miny + hh - 1;

	//std::cout << "Glyph: " << minx << ", " << miny << ", " << bitmap->width <<
	//	", " << bitmap->rows << std::endl;

	if((px < minx) || (px > maxx) || (py < miny) || (py > maxy))
	{
		return 0;
	}
	return bitmap->buffer[(py - miny) * bitmap->width + (px - minx)];
}

/** \brief Return the distance field value of a coordinate.
 *
 * Ineffective. Don't care.
 *
 * @param glyph Glyph to examine.
 * @param px X coordinate.
 * @param py Y coordinate.
 * @param bsize Big size.
 * @param ssize Small size.
 * @return Value found.
 */
static uint8_t get_ftbitmap_dfield_value(const FT_GlyphSlot glyph, int px, int py,
		int bsize, int ssize)
{
	px = px * bsize / ssize + ssize / 2;
	py = py * bsize / ssize + ssize / 2;
	int closest = 128;

	if(get_ftbitmap_value(glyph, px, py, bsize) >= 128)
	{
		for(int ii = px - 128; (ii <= px + 128); ++ii)
		{
			for(int jj = py - 128; (jj <= py + 128); ++jj)
			{
				int dist = idist(ii, jj, px, py);
				if(dist < closest)
				{
					if(get_ftbitmap_value(glyph, ii, jj, bsize) < 128)
					{
						closest = dist;
					}
				}
			}
		}
		return static_cast<uint8_t>(std::min(128 + closest, 255));
	}
	else
	{
		for(int ii = px - 128; (ii <= px + 128); ++ii)
		{
			for(int jj = py - 128; (jj <= py + 128); ++jj)
			{
				int dist = idist(ii, jj, px, py);
				if(dist < closest)
				{
					if(get_ftbitmap_value(glyph, ii, jj, bsize) >= 128)
					{
						closest = dist;
					}
				}
			}
		}
		return static_cast<uint8_t>(std::max(128 - closest, 0));
	}
}

FtFace::FtFace(const std::string &filename) throw(std::runtime_error) :
	_face(NULL),
	_size(0)
{
	if(!FtFace::ftlib)
	{
		if(FT_Init_FreeType(&FtFace::ftlib))
		{
			throw std::string("could not init FreeType");
		}
	}
	if(FT_New_Face(FtFace::ftlib, filename.c_str(), 0, &(_face)))
	{
		std::stringstream err;
		err << "could not load font: " << filename;
		BOOST_THROW_EXCEPTION(std::runtime_error(err.str()));
	}
}

FtFace::~FtFace()
{
	if(_face)
	{
		FT_Done_Face(_face);
	}
}

uint8_t* FtFace::crunch(unsigned targetsize)
{
	uint8_t *ret = new uint8_t[targetsize * targetsize];

  int ii,
			jj,
			rsz = _size;
  const FT_GlyphSlot g = _face->glyph;
#pragma omp parallel for shared(ret, targetsize, rsz) private(jj)
	for(ii = 0; (ii < static_cast<int>(targetsize)); ++ii)
	{
		for(jj = 0; (jj < static_cast<int>(targetsize)); ++jj)
		{
			ret[ii * targetsize + jj] = get_ftbitmap_dfield_value(g, jj, ii, rsz,
					static_cast<int>(targetsize));
		}
	}

	return ret;
}

bool FtFace::hasGlyph(int unicode)
{
	return (FT_Get_Char_Index(_face, unicode) > 0);
}

FtGlyph* FtFace::renderGlyph(unsigned unicode, unsigned targetsize)
{
	int idx = FT_Get_Char_Index(_face, unicode);

	if(idx <= 0)
	{
		return NULL;
	}

	if(FT_Load_Glyph(_face, idx, FT_LOAD_DEFAULT))
	{
		//std::cerr << "Warning: could not load glyph: " << unicode << ", " <<
		//	idx << std::endl;
		return NULL;
	}

	FT_GlyphSlot glyph = _face->glyph;
	if(glyph->format != FT_GLYPH_FORMAT_BITMAP)
	{
		if(FT_Render_Glyph(glyph, FT_RENDER_MODE_NORMAL))
		{
			//std::cerr << "Warning: could not render glyph: " << unicode << ", " <<
			//	idx << std::endl;
			return NULL;
		}
	}

	// Crunch the bitmap.
	uint8_t *crunched = this->crunch(targetsize);
	FT_Bitmap *bitmap = &(glyph->bitmap);

	// The divisions by 64 are due to the advance values being expressed as
	// 1/64ths of a pixel.
	float sz = static_cast<float>(_size),
				ww = static_cast<float>(bitmap->width) / sz,
				hh = static_cast<float>(bitmap->rows) / sz,
				lt = static_cast<float>(glyph->bitmap_left) / sz,
				tp = static_cast<float>(glyph->bitmap_top) / sz,
				ax = static_cast<float>(glyph->advance.x) / sz / 64.0f,
				ay = static_cast<float>(glyph->advance.y) / sz / 64.0f;

	return new FtGlyph(unicode, ww, hh, lt, tp, ax, ay, targetsize, targetsize,
			crunched);
}

void FtFace::setSize(int sz)
{
	if(FT_Set_Pixel_Sizes(_face, 0, sz))
	{
		std::stringstream sstream;
		sstream << "could not set font size to " << sz;
		throw sstream.str();
	}
	_size = sz;
}

