#ifndef DATA_STORE_HPP
#define DATA_STORE_HPP

#include "data/generic.hpp"

#include <map>
#include <string>

namespace data
{
	/** \brief Actual function executed to canonize data store names.
	 *
	 * @param op Filename to canonize.
	 * @return Processed name.
	 */
	extern std::string store_name_canonize(const std::string &op);

	/** \brief Actual function executed to loaderize data store names.
	 *
	 * @param op Filename to loaderize.
	 * @return Processed name.
	 */
	extern std::string store_name_loaderize(const std::string &op);

	/** \brief Meta-creator base class for the store. */
	template <class T> class StoreCreator
	{
		public:
			/** \brief Constructor. */
			StoreCreator() { }

			/** \brief Destructor. */
			virtual ~StoreCreator() { }

		public:
			/** \brief Create a new object.
			 *
			 * @param op Data to pass to the constructor.
			 */
			virtual T* create(const std::string &op) const = 0;
	};

	/** \brief Meta-creator, no params. */
	template <class T, class R> class StoreCreatorVoid :
		public StoreCreator<T>
	{
		public:
			/** \brief Constructor. */
			StoreCreatorVoid() { }

			/** \brief Destructor. */
			virtual ~StoreCreatorVoid() { }

		public:
			/** \brief Create a new object.
			 *
			 * @param op Data to pass to the constructor.
			 */
			virtual T* create(const std::string &op) const
			{
				return new R(op);
			}
	};

	/** \brief Meta-creator one parameter. */
	template<class T, class R, typename P> class StoreCreatorSingle :
		public StoreCreator<T>
	{
		private:
			/** Default parameter. */
			P m_param;

		public:
			/** \brief Constructor.
			 *
			 * @param op Default parameter.
			 */
			StoreCreatorSingle(const P &op) :
				m_param(op) { }

			/** \brief Destructor. */
			virtual ~StoreCreatorSingle() { }

		public:
			/** \brief Create a new object.
			 *
			 * @param op Data to pass to the constructor.
			 */
			virtual T* create(const std::string &op) const
			{
				return new R(op, m_param);
			}
	};

	/** \brief Generic data storage.
	 *
	 * Saves loaded or just stored objects into singular entities or if they're
	 * not singular, into vectors of entities.
	 *
	 * Data inserted into this storage must have a constructor that implements
	 * construction with a const C string.
	 *
	 * Type T is used for storage. The class internally uses type R in loading
	 * phases.
	 */
	template <class T> class Store
	{
		public:
			/** Typedef for stored values. */
			typedef T value_type;

			/** Type for containers. */
			typedef std::vector<boost::shared_ptr<T> > container_type;

			/** Type for store. */
			typedef std::map<std::string, container_type> store_type;

		protected:
			/** Map of objects. */
			std::map<std::string, container_type> m_data;

			/** Meta-append. */
			std::string m_meta;

			/** Single-append. */
			std::string m_single;

		public:
			/** \brief Accessor.
			 *
			 * @return Single ending.
			 */
			const std::string& getEndingSingle() const
			{
				return m_single;
			}

		public:
			/** \brief Empty constructor. */
			Store(const char *ss, const char *mm = NULL) :
				m_single(ss)
			{
				if(mm)
				{
					m_meta.assign(mm);
				}
			}

			/** \brief Destructor. */
			virtual ~Store() { }

		public:
			/** \brief Clear everything. */
			void clear()
			{
				m_data.clear();
			}

			/** \brief Throw an error if a key is found.
			 *
			 * @param pfname Key.
			 */
			void denyExists(const std::string &pfname)
			{
				if(m_data.end() != m_data.find(pfname))
				{
					BOOST_THROW_EXCEPTION(std::runtime_error(
								std::string("storage already contains '") + pfname +
								std::string("'")));
				}
				return;
			}

			/** \brief Get all objects from the store.
			 *
			 * Throws an error if not found.
			 *
			 * @return pfname Filename originally loaded as.
			 */
			container_type& getAll(const std::string &pfname)
			{
				if(m_data.end() == m_data.find(pfname))
				{
					BOOST_THROW_EXCEPTION(std::runtime_error(
								std::string("no '") + pfname +
								std::string("' available in the store")));
				}
				return m_data[pfname];
			}

			/** \brief Get all objects from the store.
			 *
			 * Throws an error if not found.
			 *
			 * @return pfname Filename originally loaded as.
			 */
			inline container_type& getAll(const char *pfname)
			{
				return this->getAll(std::string(pfname));
			}

			/** \brief Get one object from the store.
			 *
			 * Throws an error if not found.
			 *
			 * @param pfname Filename originally leaded as.
			 */
			inline T* get(const std::string &pfname)
			{
				return this->getAll(pfname).front().get();
			}

			/** \brief Get one object from the store.
			 *
			 * Throws an error if not found.
			 *
			 * @param pfname Filename originally leaded as.
			 * @return Pointer to value.
			 */
			inline T* get(const char *pfname)
			{
				return this->get(std::string(pfname));
			}

			/** \brief Try to load given object.
			 *
			 * Appends the given append string with the tails specified.
			 *
			 * If multi-ending is found, calls multi-loader method, then appends all
			 * contents.
			 *
			 * If multi-loader method is not found but single is found, loads object
			 * with it and stores it.
			 *
			 * Throws an error if nothing found, otherwise returns a reference to a
			 * vector of all objects stored.
			 *
			 * @param pfname Filename.
			 * @return Object vector.
			 */
			template <class R> container_type& load(const std::string &pfname)
			{
				std::string lname = loaderize(pfname);
				typename store_type::iterator iter = m_data.find(canonize(lname));
				if(m_data.end() != iter)
				{
					return iter->second;
				}

				StoreCreatorVoid<T, R> creator;

				container_type* mload = this->loadMetaWrapper(lname, creator);
				if(mload)
				{
					return *mload;
				}
				return this->loadSingleWrapper(lname, creator);
			}

			/** \brief Try to load given object.
			 *
			 * Similar to the basic load, but passes one boolean parameter to the
			 * constructor in addition.
			 *
			 * @param pfname Filename.
			 * @param flag Extra flag to pass to constructor.
			 * @return Object vector.
			 */
			template <class R> container_type& load(const std::string &pfname, bool flag)
			{
				std::string lname = loaderize(pfname);
				typename store_type::iterator iter = m_data.find(canonize(lname));
				if(m_data.end() != iter)
				{
					return iter->second;
				}

				StoreCreatorSingle<T, R, bool> creator(flag);

				container_type* mload = this->loadMetaWrapper(lname, creator);
				if(mload)
				{
					return *mload;
				}
				return this->loadSingleWrapper(lname, creator);
			}

			/** \brief Try to load given object.
			 *
			 * Similar to the basic load, but passes one float parameter to the
			 * constructor in addition.
			 *
			 * @param pfname Filename.
			 * @param val Extra value to pass to constructor.
			 * @return Object vector.
			 */
			template <class R> container_type& load(const std::string &pfname, float val)
			{
				std::string lname = loaderize(pfname);
				typename store_type::iterator iter = m_data.find(canonize(lname));
				if(m_data.end() != iter)
				{
					return iter->second;
				}

				StoreCreatorSingle<T, R, float> creator(val);

				container_type* mload = this->loadMetaWrapper(lname, creator);
				if(mload)
				{
					return *mload;
				}
				return this->loadSingleWrapper(lname, creator);
			}

			/** \brief Wrapper for load.
			 *
			 * @param pfname Filename.
			 */
			template <class R> inline container_type& load(const char *pfname)
			{
				return this->load<R>(std::string(pfname));
			}

			/** \brief Wrapper for load.
			 *
			 * @param pfname Filename.
			 * @param flag Flag.
			 */
			template <class R> inline container_type& load(const char *pfname,
					bool flag)
			{
				return this->load<R>(std::string(pfname), flag);
			}

			/** \brief Wrapper for multi-loader.
			 *
			 * @param pfname Filename head.
			 * @param creator Creator used.
			 * @return True if something loaded and found, false otherwise.
			 */
			container_type* loadMetaWrapper(const std::string &pfname,
					const StoreCreator<T> &creator)
			{
				std::string filename;
				try
				{
					filename = open_search(pfname + m_meta);
				}
				catch(...)
				{
					return false;
				}
				container_type container;
				this->loadMeta(container, filename, creator);
				// If reaching here, meta-load has succeeded, note that not necessarily 
				// anything is written into the container. In these cases, we must
				// assume that a meta-datafile was in question and just return it as
				// normal. This kind of metafile will be an empty vector, but it will
				// in some cases tell requestees that the metafile was indeed loaded.
				container_type &ref = m_data[canonize(filename)] = container;
				stl_trim(ref);
				return &ref;
			}

			/** \brief Wrapper for single-loader.
			 *
			 * Since this is the last resort, will always return false.
			 *
			 * @param pfname Filename used.
			 * @param creator Creator used.
			 */
			inline container_type& loadSingleWrapper(const std::string &pfname,
					const StoreCreator<T> &creator)
			{
				std::string filename;
				try
				{
					filename = open_search(pfname + m_single);
				}
				catch(...)
				{
					BOOST_THROW_EXCEPTION(std::runtime_error(
								std::string("no files found for '") + pfname +
								std::string("'")));

				}
				T *obj = creator.create(filename);
				container_type &ref = m_data[canonize(filename)] = container_type();
				ref.push_back(boost::shared_ptr<T>(obj));
				stl_trim(ref);
				return ref;
			}

			/** \brief Put something into the store.
			 *
			 * This has been loaded elsewhere.
			 *
			 * @param pfname Filename mockup to use.
			 * @param op Object to put.
			 */
			T* put(const std::string &pfname, T *op)
			{
				if(m_data.end() != m_data.find(pfname))
				{
					BOOST_THROW_EXCEPTION(std::runtime_error(
								std::string("slot '") + pfname +
								std::string("' required for manual insert already taken")));
				}
				container_type &container = m_data[pfname] = container_type();
				container.push_back(boost::shared_ptr<T>(op));
				stl_trim(container);
				return op;
			}

			/** \brief Put something into the store.
			 *
			 * This has been loaded elsewhere.
			 *
			 * @param pfname Filename mockup to use.
			 * @param op Object to put.
			 */
			T* put(const char *pfname, T *op)
			{
				return this->put(std::string(pfname), op);
			}

			/** \brief Remove data from the store.
			 *
			 * Will throw an error if nothing found.
			 *
			 * @param pfname Data to delete.
			 */
			void remove(const std::string &pfname)
			{
				if(m_data.end() == m_data.find(pfname))
				{
					BOOST_THROW_EXCEPTION(std::runtime_error(
								std::string("no '") + pfname + "' to remove in the store"));
				}
				m_data.erase(m_data.find(pfname));
			}

			/** \brief Remove data from the store.
			 *
			 * Will throw an error if nothing found.
			 *
			 * @param pfname Data to delete.
			 */
			void remove(const char *pfname)
			{
				this->remove(std::string(pfname));
			}

		protected:
			/** \brief Meta-load.
			 *
			 * Default implementation very simply tries to load as normal.
			 *
			 * @param dst Target vector to store to.
			 * @param pfname Existing filename to load from.
			 * @param creator Object used for creating a new objects.
			 */
			virtual void loadMeta(container_type &dst, const std::string &pfname,
					const StoreCreator<T> &creator)
			{
				dst.push_back(boost::shared_ptr<T>(creator.create(pfname)));
			}

		public:
			/** \brief Create storage name from load name.
			 *
			 * Quite equal to basename of a file.
			 *
			 * Only used in load() and associates - put puts the files exactly under
			 * their names.
			 *
			 * @param pfname Filename used for loading before search or append.
			 * @return Canonical name.
			 */
			inline std::string canonize(const std::string &op)
			{
				return store_name_canonize(op);
			}

			/** \brief Create a 'load name'.
			 *
			 * i.e. remove everything after the last dot.
			 *
			 * @param pfname Filename passed to loader function.
			 * @return Search name.
			 */
			inline std::string loaderize(const std::string &op)
			{
				return store_name_loaderize(op);
			}
	};
}

#endif
