/***************************************************************************
*   Copyright (C) 2010 by
*    Kai Lindholm <megantti@gmail.com>
*    Santtu Keskinen <laquendi@gmail.com>
*   
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
* 
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*   
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
****************************************************************************/

#ifndef _WORLD_H
#define _WORLD_H

#include <vector>
#include <list>
#include <map>
#include <queue>
#include <stack>
#include "ant.h"
#include "commandcenter.h"
#include "creature.h"
#include "event.h"
#include "map.h"
#include "mushroom.h"
#include "object.h"
#include "object_map.h"
#include "wind.h"
#include "antheap.h"
#include "trap.h"

class ant_heap;

class world {
	public:
		world(animation_loader *l, event_handler &e, creature::creature_type);
		~world();
		std::vector<mushroom*> &get_mushrooms() { return mushrooms; }
		std::vector<trap*> &get_traps() { return traps; }
		std::vector<command_center*> &get_command_centers() { return command_centers; }
		creature &get_player() { return player; }
		mushroom* nearest_mushroom(sf::Vector2f pos, double limit, bool accept_growing = false, bool accept_invisible = true);
		command_center* nearest_cc(sf::Vector2f pos, double limit, bool accept_invisible = true);
		ant* nearest_ant(sf::Vector2f pos, double limit);
		ant_heap* nearest_ant_heap(sf::Vector2f pos, double limit);
		std::list<selectable*> get_m_and_cc_in_range(sf::Vector2f pos, double limit);

		map &get_map() { return tile_map; }
		void check_win();

		// call this when you are going to remove a mushroom
		void disconnect_mushroom(mushroom *m);

		selectable* get_selected();
		void select(sf::Vector2f);
		void select(selectable*);
		void select(ID id);
		void clear_selection() { selected = &player; selected_ch = true; }
		bool select_mushroom(sf::Vector2f);
		bool select_cc(sf::Vector2f);
		bool select_ant(sf::Vector2f);
		bool select_ant_heap(sf::Vector2f);

		bool collision(const object &obj);
	
		void move_player(sf::Vector2f pos);

		void activate_spell_cooldown(selectable *, spell &casted_spell);

		bool cast(selectable *caster, spell &casted_spell, sf::Vector2f mouse);
		bool selected_changed();

		void cc_tree_selection(sf::Vector2f mouse);
		void set_cc_tree_visited();
		void cc_add_energy(int energy);

		void update_mushroom_stats();

		bool is_win() { return won; }
		bool is_lose() { return lost; }

		void update(pclock &game_time);
		void pause();
#ifndef NO_GRAPHICS
		void draw(sf::RenderTarget &target, particle_engine &e);
#endif
		
		bool out_of_bounds(sf::Vector2f);
		
		sf::Vector2f get_ant_target_location();

		std::vector<ant*> &get_ants() { return ants; }

		void remove_dead();
		void delete_obj(object *obj);
		void add_obj(object *obj);

		void obj_map_add(object *arg) { obj_map.add(arg); }
		void obj_map_remove(object *arg) { obj_map.remove(arg); }

		void move_object(object *obj, sf::Vector2f);

		float ant_dir_goodness(sf::Vector2f dir, ant* ant);

		std::pair<sf::Vector2f, mushroom::mushroom_type> get_deceased_m();
		bool has_deceased_m() { return !deceased_m.empty(); }

		animation_loader *anim_loader;
		event_handler &handler;

		std::string killed_ants_string();
		std::string casted_spells_string();
		std::string died_mushrooms_string();
		std::string current_mushrooms_string();
		std::string wind_string();

	private:
		void win();
		void lose();

		template<class T> void update_objs(std::vector<T*> &l);
#ifndef NO_GRAPHICS
		template<class T> void draw_objs(std::vector<T*> &l, sf::RenderTarget &target);
#endif
		template<class T> void remove_dead_objs(std::vector<T*> &l);
		template<class T> void update_spells(std::vector<T*> &l);
		template<class T> void remove_target(std::vector<T*> &l, object *target);

		template<class T> T* nearest(sf::Vector2f, double range, std::vector<T*> &, bool (*check)(T*));

		std::vector<mushroom*> mushrooms;
		std::vector<command_center*> command_centers;
		std::vector<ant_heap*> ant_heaps;
		std::vector<ant*> ants;
		std::vector<trap*> traps;
		creature player;
		map tile_map;
		object_map obj_map;
		wind game_wind;
		
		// for giving exp every second
		int last_second;

		bool selected_ch;
		selectable *selected;

		bool won, lost;

		std::stack<std::pair<sf::Vector2f, mushroom::mushroom_type> > deceased_m;

		sf::Vector2f ant_target_location;
		
		int killed_ants, casted_spells, died_mushrooms;
};

template<> void world::remove_dead_objs(std::vector<ant*> &l);
template<> void world::remove_dead_objs(std::vector<mushroom*> &l);

template<class T> void world::update_objs(std::vector<T*> &l) 
{
	for(unsigned int a = 0; a < l.size(); a++) {
		obj_map.remove(l[a]);
		l[a]->update(*this);
		obj_map.add(l[a]);
	}
}

#ifndef NO_GRAPHICS
template<class T> void world::draw_objs(std::vector<T*> &l, sf::RenderTarget &target)
{
	for(unsigned int a = 0; a < l.size(); a++) {
		l[a]->draw(target);
	}
}
#endif

template<class T> void world::remove_dead_objs(std::vector<T*> &l)
{
	for(int a = 0; a < (int)l.size(); a++) {
		if(l[a]->to_removed()) {
			delete_obj(l[a]);
			delete l[a];
			l.erase(l.begin() + a);
			a--;
		}
	}
}

template<class T> void world::remove_target(std::vector<T*> &l, object *target)
{
	for(int a = 0; a < (int)l.size(); a++) {
		if(l[a]->get_target() == target)
			l[a]->set_target(NULL);
	}
}

template<class T> T* world::nearest(sf::Vector2f pos, double range, std::vector<T*> &l, bool (*check)(T*))
{
	double min_dist = -1;
	int best = -1;

	for(unsigned int a = 0; a < l.size(); a++) {
		double d = dist(pos, l[a]->get_center());
		if(d < range) {
			if(best == -1 || d < min_dist) {
				if(check(l[a])) {
					best = a;
					min_dist = d;
				}
			}
		}
	}
	if(best==-1)
		return NULL;
	return l[best];
}

template<class T> void world::update_spells(std::vector<T*> &l)
{
	for(unsigned int a = 0; a < l.size(); a++) {
		for(unsigned int b = 0; b < l[a]->get_spells().size(); b++) {
			l[a]->get_spells()[b].update();
		}
	}
}


#endif
