/***************************************************************************
*   Copyright (C) 2010 by
*    Kai Lindholm <megantti@gmail.com>
*    Santtu Keskinen <laquendi@gmail.com>
*   
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
* 
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*   
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
****************************************************************************/

#include "world.h"
#include "antheap.h"
#include <sstream>

world::world(animation_loader *l, event_handler &e, creature::creature_type ct) : 
	anim_loader(l), handler(e), player(l, ct), tile_map(l), 
	obj_map(tile_map.w * TILE_W / OBJMAP_TILE_W, tile_map.h * TILE_H / OBJMAP_TILE_H),
	game_wind(140.0f, 10.0f, 180.0f), last_second(0), selected_ch(false), 
	selected(&player), won(false), lost(false), killed_ants(0), casted_spells(0), 
	died_mushrooms(0)
{
	for(map::goal_iter i = tile_map.goals.begin(); i != tile_map.goals.end(); i++) {
		obj_map.add(&(*i));
	}
	command_centers.push_back(new command_center(l));

	ant_heaps.push_back(new ant_heap(l));
	ant_heaps.back()->set_pos(sf::Vector2f(1050.0f, 500.0f));

	command_centers.back()->set_pos(sf::Vector2f(400.0f, 540.0f));
	player.set_pos(sf::Vector2f(300.0f, 560.0f));

	add_obj(command_centers.back());
	add_obj(ant_heaps.back());
	add_obj(&player);
}

world::~world()
{
	for(unsigned int a = 0; a < mushrooms.size(); a++)
		delete mushrooms[a];
	for(unsigned int a = 0; a < command_centers.size(); a++)
		delete command_centers[a];
	for(unsigned int a = 0; a < ant_heaps.size(); a++)
		delete ant_heaps[a];
	for(unsigned int a = 0; a < ants.size(); a++)
		delete ants[a];
}

static bool m_check_growing(mushroom *m)
{
	return !m->is_growing() && !m->is_dead();
}

static bool m_check_invisible(mushroom *m)
{
	return !m->is_invisible() && !m->is_dead();
}

static bool m_check_growing_invisible(mushroom *m)
{
	return (!m->is_invisible()) && (!m->is_growing()) && !m->is_dead();
}

static bool cc_check_invisible(command_center *cc)
{
	return !cc->is_invisible();
}

template<class T> bool all(T*)
{
	return true;
}

mushroom* world::nearest_mushroom(sf::Vector2f pos, double limit, bool accept_growing, bool accept_invisible)
{
	if(accept_growing && accept_invisible)
		return nearest(pos, limit, mushrooms, all);
	if(accept_growing && !accept_invisible)
		return nearest(pos, limit, mushrooms, m_check_invisible);
	if(!accept_growing && accept_invisible)
		return nearest(pos, limit, mushrooms, m_check_growing);
	return nearest(pos, limit, mushrooms, m_check_growing_invisible);
	
}

command_center* world::nearest_cc(sf::Vector2f pos, double limit, bool accept_invisible)
{
	if(accept_invisible) {
		return nearest(pos, limit, command_centers, all);
	} else {
		return nearest(pos, limit, command_centers, cc_check_invisible);
	}

}

ant* world::nearest_ant(sf::Vector2f pos, double limit)
{
	return nearest(pos, limit, ants, all);
}

ant_heap* world::nearest_ant_heap(sf::Vector2f pos, double limit)
{
	return nearest(pos, limit, ant_heaps, all);
}

std::list<selectable*> world::get_m_and_cc_in_range(sf::Vector2f pos, double limit)
{
	std::list<selectable*> ret;
	for(size_t a = 0; a < mushrooms.size(); a++) {
		if(dist(pos, mushrooms[a]->get_center()) <= limit)
			ret.push_back(mushrooms[a]);
	}
	for(size_t a = 0; a < command_centers.size(); a++) {
		if(dist(pos, command_centers[a]->get_center()) <= limit)
			ret.push_back(command_centers[a]);
	}
	return ret;
}

void world::disconnect_mushroom(mushroom *m)
{
	for(unsigned int a = 0; a < mushrooms.size(); a++) {
		mushrooms[a]->disconnect(m);
	}
	for(unsigned int a = 0; a < command_centers.size(); a++) {
		command_centers[a]->disconnect_mushroom(m);
	}
}

void world::pause()
{
	for(unsigned int a = 0; a < mushrooms.size(); a++)
		mushrooms[a]->pause();
	for(unsigned int a = 0; a < command_centers.size(); a++)
		command_centers[a]->pause();
	for(unsigned int a = 0; a < ants.size(); a++)
		ants[a]->pause();
	player.pause();
}

void world::update_mushroom_stats()
{
	for (size_t a = 0; a < mushrooms.size(); a++) {
		const int *attribs = get_player().get_mushroom_attribs();
		mushrooms[a]->set_attributes(attribs[0], attribs[1], attribs[2]);
	}
}

void world::update(pclock &game_time)
{
	if(won || lost) {
		return;
	}

	ant_target_location = obj_map.ant_target_location();

	for(int a = 0; a < (int)ants.size(); a++) {
		if(ants[a]->get_pos().x < 0) {
			delete_obj(ants[a]);
			ants.erase(ants.begin()+a);
			a--;
		}
	}
	
	update_spells(mushrooms);
	update_spells(command_centers);
	for(unsigned int a = 0; a < player.get_spells().size(); a++) {
		player.get_spells()[a].update();
	}

	update_objs(mushrooms);
	update_objs(command_centers);
	update_objs(ants);
	update_objs(ant_heaps);
	update_objs(traps);

	update_mushroom_stats();

	if(command_centers.size() == 0) {
		lose();
	}

	mushroom *m = dynamic_cast<mushroom*>(get_selected());
	if(m && m->is_dead()) {
		clear_selection();
	}

	
	game_wind.update(*this);

	tile_map.update();

	obj_map.remove(&player);
	player.update(*this);
	obj_map.add(&player);
	
	// floats are pain in butt so this is kinda stupid
	if(floor(game_time.get_time()) != last_second) {
		last_second = floor(game_time.get_time());
		get_player().gain_exp(*this, 1);
	}

	// check if creature has reached spell casting range
	if(get_player().moving_to_target) {
		if(get_player().spell_to_cast.in_range(&get_player(), get_player().spell_target_pos, *this)) {
			cast(&get_player(), get_player().spell_to_cast, get_player().spell_target_pos);
			get_player().stop_moving();
		}
	}

	remove_dead();
	
	for(unsigned int a = 0; a < command_centers.size(); a++) {
		command_centers[a]->tree_fix(*this);
	}

}
		
void world::move_player(sf::Vector2f pos)
{
	get_player().set_move_pos(pos);
}

void world::remove_dead()
{
	remove_dead_objs(mushrooms);
	remove_dead_objs(command_centers);
	remove_dead_objs(ants);
	remove_dead_objs(traps);
}
		
void world::move_object(object *obj, sf::Vector2f mv)
{
	obj_map.remove(obj);
	if((obj->get_pos() + mv).x > SCREEN_W - SIDEBAR_W + 50) {
		mv.x = SCREEN_W - SIDEBAR_W + 50 - obj->get_pos().x;
	}

	obj->set_pos(obj->get_pos() + mv);
	if(collision(*obj)) {
		obj->set_pos(obj->get_pos() - mv);
	}
	obj_map.add(obj);
}

void world::check_win()
{
	for(size_t a = 0; a < command_centers.size(); a++) {
		if(command_centers[a]->win(*this))
			win();
	}
}

void world::add_obj(object *obj)
{
	mushroom *m = dynamic_cast<mushroom*>(obj);

	if(m) {
		if(tile_map.near_goal(obj, GOAL_WIN_RANGE)) {
			for(size_t a = 0; a < command_centers.size(); a++) {
				if(command_centers[a]->win_connected(m, *this))
					win();
			}
		}
	}

	obj_map.add(obj);
	handler.spawn(obj);
}

void world::delete_obj(object *obj)
{
	if(selected == obj) {
		selected_ch = true;
		selected = &player;
	}

	remove_target(command_centers, obj);
	remove_target(mushrooms, obj);
	remove_target(ants, obj);
	remove_target(traps, obj);

	handler.die(obj);
	obj_map.remove(obj);
	mushroom *m = dynamic_cast<mushroom*>(obj);
	if(m) {
		disconnect_mushroom(m);
	}
}

bool world::selected_changed()
{
	if(selected_ch) {
		selected_ch = false;
		return true;
	} else {	
		return false;
	}
}

void world::cc_tree_selection(sf::Vector2f mouse)
{
	mushroom *m = nearest_mushroom(mouse, SELECT_MUSHROOM_RANGE, true);
	if(m)
	{
		command_center *cur_cc = dynamic_cast<command_center*>(get_selected());
		if(cur_cc)
		{
			cur_cc->tree_add_rm(m, *this);
		}
	}
}

void world::set_cc_tree_visited()
{
	for(unsigned int a = 0; a < mushrooms.size(); a++) {
		mushrooms[a]->cc_tree_visited = false;
	}
}

void world::cc_add_energy(int energy)
{
	for(size_t a = 0; a < command_centers.size(); a++) {
		command_centers[a]->add_energy(energy);
	}
}

#ifndef NO_GRAPHICS


void world::draw(sf::RenderTarget &target, particle_engine &e)
{
	tile_map.draw(target);
	for(unsigned int a = 0; a < mushrooms.size(); a++) {
		mushroom *m1 = mushrooms[a];
		std::list<mushroom*> conns = m1->get_connections();
		for(std::list<mushroom*>::iterator b = conns.begin(); b != conns.end(); b++) {
			mushroom *m2 = *b;
			target.Draw(sf::Shape::Line(m1->get_center().x, m1->get_center().y, 
					m2->get_center().x, m2->get_center().y, 1.0f, sf::Color::Blue));
		}
	}
	for(unsigned int a = 0; a < command_centers.size(); a++) {
		command_center *cc = command_centers[a];
		const std::list<mushroom*> &conns = cc->get_connected_mushrooms();
		for(std::list<mushroom*>::const_iterator b = conns.begin(); b != conns.end(); b++) {
			mushroom *m = *b;
			target.Draw(sf::Shape::Line(cc->get_center().x, cc->get_center().y, 
					m->get_center().x, m->get_center().y, 1.0f, sf::Color::Blue));
		}
	}

	draw_objs(mushrooms, target);
	draw_objs(traps, target);
	draw_objs(command_centers, target);
	draw_objs(ants, target);
	draw_objs(ant_heaps, target);

	player.draw(target);
	get_selected()->draw_border(target, sf::Color::Blue);
		
	for(unsigned int a = 0; a < mushrooms.size(); a++) {
		mushrooms[a]->draw_attack_anim(target, e);
	};

	command_center *cc = dynamic_cast<command_center*>(get_selected());
	if(cc) {
		const std::list<mushroom*> &li = cc->get_tree();
		std::list<mushroom*>::const_iterator it;
		for(it = li.begin(); it != li.end(); it++) {
			(*it)->draw_border(target, sf::Color::Yellow);
		}
	}

	game_wind.draw(target);
}
#endif

void world::select(sf::Vector2f pos)
{
	selectable *old = selected;
	if(select_mushroom(pos)) {
		if(old != selected)
			selected_ch = true;
		return;
	}

	if(select_cc(pos)) {
		if(old != selected)
			selected_ch = true;
		return;
	}
	
	if(select_ant(pos)) {
		if(old != selected)
			selected_ch = true;
		return;
	}

	if(select_ant_heap(pos)) {
		if(old != selected)
			selected_ch = true;
		return;
	}

	selected = &player;
	if(old != selected)
		selected_ch = true;
}

void world::select(selectable* sel)
{
	bool exists = false;
	if (&player == sel) 
		exists = true;
	if(!exists)
		for (size_t a = 0; a < mushrooms.size(); a++) {
			if (mushrooms[a] == sel){
				exists = true;
				break;
			}
		}
	if(!exists)
		for (size_t a = 0; a < command_centers.size(); a++) {
			if (command_centers[a] == sel) {
				exists = true;
				break;
			}
		}
	if(!exists)
		for (size_t a = 0; a < ants.size(); a++) {
			if (ants[a] == sel) {
				exists = true;
				break;
			}
		}

	if (exists) {
		selected = sel;
		selected_ch = true;
	}
}

void world::select(ID id)
{
	bool exists = false;
	if (player.get_id() == id) {
		exists = true;
		selected = &player;
	}
	if(!exists)
		for (size_t a = 0; a < mushrooms.size(); a++) {
			if (mushrooms[a]->get_id() == id){
				selected = mushrooms[a];
				exists = true;
				break;
			}
		}
	if(!exists)
		for (size_t a = 0; a < command_centers.size(); a++) {
			if (command_centers[a]->get_id() == id) {
				selected = command_centers[a];
				exists = true;
				break;
			}
		}
	if(!exists)
		for (size_t a = 0; a < ants.size(); a++) {
			if (ants[a]->get_id() == id) {
				selected = ants[a];
				exists = true;
				break;
			}
		}

	if (exists) {
		selected_ch = true;
	}
}

bool world::select_mushroom(sf::Vector2f pos)
{
	mushroom *shroom = nearest_mushroom(pos, SELECT_MUSHROOM_RANGE, true);
	if(shroom) {
		selected = shroom;
		return true;
	}
	return false;
}

bool world::select_cc(sf::Vector2f pos)
{
	command_center *cc = nearest_cc(pos, SELECT_CC_RANGE);
	if(cc) {
		selected = cc;
		return true;
	}
	return false;
}

bool world::select_ant(sf::Vector2f pos)
{
	ant *a = nearest_ant(pos, SELECT_ANT_RANGE);
	if(a) {
		selected = a;
		return true;
	}
	return false;
}

bool world::select_ant_heap(sf::Vector2f pos)
{
	ant_heap *h = nearest_ant_heap(pos, SELECT_CC_RANGE);
	if(h) {
		selected = h;
		return true;
	}
	return false;
}

bool world::collision(const object &obj)
{
	creature *p = dynamic_cast<creature*>(&const_cast<object&>(obj));
	if(!p || p->get_type()!=creature::griffin)
		if(tile_map.collision(obj))
			return true;

	if(obj_map.collision(&obj))
		return true;
	return false;
}

void world::activate_spell_cooldown(selectable *caster, spell &casted_spell)
{
	std::vector<spell> &spells = caster->get_spells();
	std::vector<spell>::iterator it;
	for(it = spells.begin(); it != spells.end(); it++)
	{
		float cd=0.0f;
		cd = casted_spell.get_g_cd();
		it->update_cooldown(cd);
		
		if(casted_spell.get_spell_type() == it->get_spell_type()) {
			cd = casted_spell.get_s_cd();
			it->update_cooldown(cd);

		}
	}
}

bool world::cast(selectable *caster, spell &casted_spell, sf::Vector2f mouse)
{
	bool cast_successful = false;
	if(casted_spell.in_range(caster, mouse, *this))
	{
		cast_successful = casted_spell.cast(caster, mouse, *this);
		if(cast_successful) {
			casted_spells++;
			activate_spell_cooldown(caster, casted_spell);
			caster->use_energy(casted_spell.get_energy_cost(caster));
			if(casted_spell.get_spell_type() == spell::cc_grow_mushroom) {
				command_center *caster_cc = dynamic_cast<command_center*>(caster);
				caster_cc->mushrooms_growd++;
			}	
		}
	} else {
		//check if caster should go running towards target
		creature *creatr = dynamic_cast<creature*>(caster);
		if(mouse.y >= casted_spell.get_y_limit() && creatr)
		{
			creatr->move_towards_cast_target(casted_spell, mouse);
		}	
	}
	return cast_successful;
}

selectable* world::get_selected()
{
	if(selected)
		if(selected->ok())
			return selected;
	return dynamic_cast<selectable*>(&player);
}

void world::win()
{
	won = true;
	handler.win();
}

void world::lose()
{
	lost = true;
	handler.lose();
}

std::pair<sf::Vector2f, mushroom::mushroom_type> world::get_deceased_m()
{
	std::pair<sf::Vector2f, mushroom::mushroom_type> ret = deceased_m.top();
	deceased_m.pop();
	return ret;
}

template<> void world::remove_dead_objs(std::vector<ant*> &l)
{
	for(int a = 0; a < (int)l.size(); a++) {
		if(l[a]->is_dead()) {

			get_player().gain_exp(*this, 25);
			killed_ants++;

			delete_obj(l[a]);
			delete l[a];
			l.erase(l.begin() + a);
			a--;
		}
	}	
}

template<> void world::remove_dead_objs(std::vector<mushroom*> &l)
{
	for(int a = 0; a < (int)l.size(); a++) {
		if(l[a]->is_dead() && !l[a]->is_saved()) {
			deceased_m.push(std::make_pair(l[a]->get_pos(), l[a]->get_type()));
			died_mushrooms++;
		}

		if(l[a]->to_removed()) {
			delete_obj(l[a]);
			delete l[a];
			l.erase(l.begin() + a);
			a--;
		}
	}	
}

bool world::out_of_bounds(sf::Vector2f p) 
{
	return (p.x < 0 || p.x >= tile_map.w * TILE_W || p.y < 0 || p.y >= tile_map.h * TILE_H);
}

sf::Vector2f world::get_ant_target_location()
{
	return ant_target_location;
}

float world::ant_dir_goodness(sf::Vector2f dir, ant *mover)
{
	sf::Vector2f normal = sf::Vector2f(-dir.y, dir.x);
	float ret = 1;
	sf::Vector2f pos = mover->get_center();
	int blockers = obj_map.objs_on_vector(pos, sf::Vector2f(pos.x+dir.x*150, pos.y+dir.y*150), mover);
	blockers += obj_map.objs_on_vector(sf::Vector2f(pos.x + normal.x*mover->get_h()/2, pos.y + normal.y*mover->get_h()/2),
					sf::Vector2f(pos.x+dir.x*150 + normal.x*mover->get_h()/2, pos.y+dir.y*150 + normal.y*mover->get_h()/2), mover);
	blockers += obj_map.objs_on_vector(sf::Vector2f(pos.x - normal.x*mover->get_h()/2, pos.y - normal.y*mover->get_h()/2),
					sf::Vector2f(pos.x+dir.x*150 - normal.x*mover->get_h()/2, pos.y+dir.y*150 - normal.y*mover->get_h()/2), mover);
	ret /= blockers;
	return ret;
}

std::string world::killed_ants_string()
{
	std::stringstream ss(std::stringstream::out);
	if(killed_ants == 1)
		ss << killed_ants << " ant killed";
	else
		ss << killed_ants << " ants killed";
	return ss.str();
}
std::string world::casted_spells_string()
{
	std::stringstream ss(std::stringstream::out);
	if(casted_spells == 1)
		ss << casted_spells << " spell casted";
	else
		ss << casted_spells << " spells casted";
	return ss.str();
}

std::string world::died_mushrooms_string()
{
	std::stringstream ss(std::stringstream::out);
	if(died_mushrooms == 1)
		ss << died_mushrooms << " mushroom died";
	else
		ss << died_mushrooms << " mushrooms died";
	return ss.str();

}
std::string world::current_mushrooms_string()
{
	std::stringstream ss(std::stringstream::out);
	if(mushrooms.size() == 1)
		ss << mushrooms.size() << " mushroom";
	else
		ss << mushrooms.size() << " mushrooms";
	return ss.str();
	
}

std::string world::wind_string()
{
	std::stringstream ss(std::stringstream::out);
	ss.setf(std::ios::fixed, std::ios::floatfield);
	ss.precision(1);
	ss << "Next wind: " << game_wind.get_time_left();
	return ss.str();
}
